
export class NodePoolTool<T extends cc.Component> {
    private nodePool: cc.NodePool;
    private prefab: cc.Prefab;
    private targetNode: cc.Node;
    private component: new (...args: any[]) => T;

    /**
     * @param {cc.Prefab} prefab 需要加载的预制节点
     * @param {cc.Node} targetNode 需要添加到的目标节点
     * @param {cc.Component} component prefab上绑定的组件
     * @param {any} extraParams 方法需要的额外参数
     */
    constructor (prefab: cc.Prefab, targetNode: cc.Node, component: new (...args: any[]) => T) {
        this.nodePool = new cc.NodePool();
        this.prefab = prefab;
        this.targetNode = targetNode;
        this.component = component;
    }
    getComps() {
        return this.targetNode.children.map(node => node.getComponent(this.component))
    }
    getChildren() {
        return this.targetNode.children;
    }
    /**
     * 根据data加载节点 会调用实例化时设定的method  传参顺序为  dataItem，index， extraParams
     * @param data 需要加载的数据集
     * @param loadOnceCb 加载出一个节点后的回调函数
     * @param isClear  是否清除之前的节点
     */
    loadNodes<A> (data: A[], loadOnceCb?: (comp?: T, dataItem?: A, index?: number) => void, isClear = true) {
        const isVaild = this.checkLoadParams(data, this.nodePool, this.prefab, this.component, this.targetNode);
        if (!isVaild) {
            return [];
        }
        isClear && this.clear();
        const nodes: cc.Node[] = [];
        data.forEach((d, index) => {
            const comp = this.add();
            nodes.push(comp.node);
            if (!loadOnceCb) {
                return;
            }
            loadOnceCb && loadOnceCb(comp, d, index);
        });
        return nodes;
    }

    add() {
        const node = this.nodePool.get() || cc.instantiate(this.prefab);
        this.targetNode.addChild(node);
        return node.getComponent(this.component);
    }

    clear() {
        if (!this.targetNode) {
            return;
        }
        this.targetNode.children.slice().forEach(node => node.getComponent(this.component) && this.nodePool.put(node));
    }

    put(node: cc.Node) {
        this.nodePool.put(node);
    }

    checkLoadParams (data: any[], nodePool: cc.NodePool, prefab: cc.Prefab, component, targetNode: cc.Node) {
        if (!data || !Array.isArray(data)) {
            cc.error('KQBatchLoadNodes------loadNodes------data error ', data);
            return false;
        }

        if (!(nodePool instanceof cc.NodePool)) {
            cc.error('KQBatchLoadNodes------loadNodes------nodePool is not cc.NodePool ', nodePool);
            return false;
        }

        if (!(prefab instanceof cc.Prefab)) {
            cc.error('KQBatchLoadNodes------loadNodes------prefab is not cc.Prefab ', prefab);
            return false;
        }

        if (!component || !(component instanceof Object)) {
            cc.error('KQBatchLoadNodes------loadNodes------component error ', component);
            return false;
        }

        if (!targetNode || !(targetNode instanceof Node)) {
            cc.error('KQBatchLoadNodes------loadNodes------targetNode is not cc.Node ', targetNode);
            return false;
        }

        return true;
    }
}