

const { ccclass, property, requireComponent } = cc._decorator;


/**绘制线段时候，如果与上一个点距离多近，就忽略该点的绘制 */
const MIN_IGNORE_LENGTH = 3;

/**绘制线段时候，如果与上一个点距离多远，就忽略该点的绘制 */
const MAX_IGNORE_LENGTH = 30;

/**
 * 逻辑拖尾组件，使用 cc.Graphics 绘制的拖尾组件
 */
@ccclass
@requireComponent(cc.Graphics)
export default class LineMotionStreak extends cc.Component {


    @property
    public lineColor: cc.Color = cc.Color.WHITE;

    @property({ tooltip: '线的宽度' })
    public lineWidth: number = 10;

    @property({ tooltip: '决定拖尾有多长' })
    historySize: number = 20;

    @property({ tooltip: "决定拖尾有多平滑,会影响性能" })
    ropeSize: number = 5;

    private historyX: number[] = [];
    private historyY: number[] = [];
    private points: cc.Vec2[] = [];
    private ctx: cc.Graphics;

    @property(cc.Node)
    public targetNode:cc.Node = null;

    onLoad() {
        this.ctx = this.getComponent(cc.Graphics);
    }

    onEnable() {
        this.initData();
    }

    initData() {
        this.historyX.length = 0;
        this.historyY.length = 0;
        this.points.length = 0;

        let worldPos = this.node.convertToWorldSpaceAR(cc.Vec2.ZERO);
        for (let i = 0; i < this.historySize; i++) {
            this.historyX.push(worldPos.x);
            this.historyY.push(worldPos.y);
        }

        //初始化
        for (let j = 0; j < this.ropeSize; j++) {
            this.points.push(cc.v2());
        }
    }


    update(dt) {

        if(this.targetNode ==null){
            return;
        }

        //记录当前世界坐标
        let worldPos = this.targetNode.convertToWorldSpaceAR(cc.Vec2.ZERO);

        this.historyX.pop();
        this.historyX.unshift(worldPos.x);
        this.historyY.pop();
        this.historyY.unshift(worldPos.y);

        // 更新历史点记录
        for (let k = 0; k < this.ropeSize; k++) {
            let p = this.points[k];
            // 通过 cubicInterpolation 函数让路径曲线更平滑
            let ix = cubicInterpolation(this.historyX, k / this.ropeSize * this.historySize);
            let iy = cubicInterpolation(this.historyY, k / this.ropeSize * this.historySize);
            p.x = ix;
            p.y = iy;
        }

        this.rendererLine();
    }

    rendererLine() {
        this.ctx.clear();
        
        this.ctx.lineWidth = this.lineWidth;
        this.ctx.lineCap = cc.Graphics.LineCap.ROUND;
        this.ctx.strokeColor = this.lineColor;

        let firstPos = this.targetNode.getParent().convertToNodeSpaceAR(this.points[0]);
        this.ctx.moveTo(firstPos.x, firstPos.y);

        let lastPos = this.points[0];
        for (let i = 1; i < this.points.length; i++) {
            const pos = this.points[i];

            //坐标过于贴近时跳过，防止绘制时出现毛刺
            let mag =  pos.sub(lastPos).magSqr() ;
            if (mag<= MIN_IGNORE_LENGTH * MIN_IGNORE_LENGTH )continue;
            let spacePos = this.targetNode.getParent().convertToNodeSpaceAR(pos);
            this.ctx.lineTo(spacePos.x, spacePos.y);
            lastPos = pos;
        }

        this.ctx.stroke();

    }
}

/**
 * Cubic interpolation based on https://github.com/osuushi/Smooth.js
 */
function clipInput(k, arr) {
    if (k < 0) k = 0;
    if (k > arr.length - 1) k = arr.length - 1;
    return arr[k];
}

function getTangent(k, factor, array) {
    return factor * (clipInput(k + 1, array) - clipInput(k - 1, array)) / 2;
}

function cubicInterpolation(array, t, tangentFactor = 1) {
    let k = Math.floor(t);
    let m = [getTangent(k, tangentFactor, array), getTangent(k + 1, tangentFactor, array)];
    let p = [clipInput(k, array), clipInput(k + 1, array)];
    t -= k;
    let t2 = t * t;
    let t3 = t * t2;
    return (2 * t3 - 3 * t2 + 1) * p[0] + (t3 - 2 * t2 + t) * m[0] + (-2 * t3 + 3 * t2) * p[1] + (t3 - t2) * m[1];
}
