/**
 * 进度回调
 */
export type ProcessCallBack = (completedCount: number, totalCount: number, item: any) => void;
/**
 * 完成回调
 */
export type CompletedCallback = (error: Error, resource: any) => void;
/**
 * 组加载完成回调
 */
export type CompletedArrayCallback = (error: Error, resouce: any[]) => void | void;
/**
 * 路径加载完成回调
 */
export type CompletedDirCallBack = (error: Error, resources: any[]) => void | null;


/**
 * 缓存的资源信息
 */
export class CacheRefInfo {
    name: string;
    url: string;
    type: typeof cc.Asset;
    obj: any;
    useKey?: string;
}

/**
 * 加载参数
 */
export class LoadResArgs {
    url?: string;
    urls: string[];
    type?: typeof cc.Asset;
    onCompleted?: (CompletedCallback | CompletedArrayCallback | CompletedDirCallBack);
    onProcess?: ProcessCallBack;
    use?: string
}

/**
 * 释放参数
 */
export class RelaseResArgs {
    url?: string;
    urls?: string[];
    type?: typeof cc.Asset;
    use?: string;
}

let isChildClassOf = cc.js["isChildClassOf"]
if (!isChildClassOf) isChildClassOf = cc["isChildClassOf"];


export class AssetManager {
    private static instance: AssetManager = null;
    public static getInstance(): AssetManager {
        if (!this.instance) {
            this.instance = new AssetManager();
        }
        return this.instance;
    }

    /**
     * 处理一些需要缓存起来的资源信息
     */
    private resInfo: { [key: string]: CacheRefInfo } = {}


    /**
     * 获取当前内存占用资源
     */
    getResCount(): any {
        return this.resInfo;
    }


    /**
     * 获取一个资源的缓存信息
     * @param name 
     */
    private getOrSetResCacheInfo(name: string) {
        return this.resInfo[name];
    }


    /**
     * 对加载到的一个资源进行缓存
     * @param name 资源名称
     * @param type 资源类型
     * @param obj 资源实例
     * @param use 资源被引用的标记key
     */
    private cacheCurResRefInfo(name: string, type: typeof cc.Asset, obj: any, use: string) {
        let resouce = this.getOrSetResCacheInfo(name);
        if (resouce) {
            if (resouce.useKey != use) {
                //重复加载的资源，需要增加引用计数，如果一个资源只是被重复引用，则不需要增加计数
                resouce.obj.addRef();
            }
        } else {
            //第一次构建的资源，存储起来
            let cacheRes = new CacheRefInfo();
            cacheRes.name = name;
            cacheRes.type = type;
            cacheRes.obj = obj; //资源obj
            cacheRes.url = name;
            cacheRes.obj.addRef();//第一次加载需要添加引用计数
            cacheRes.useKey = use;
            this.resInfo[name] = cacheRes;
        }
    }


    /**
     * 参数配置
     */
    private getOrSetLoadResArgs(): LoadResArgs {
        if (arguments.length < 1) {
            console.error(`doLoadResArgs Error ${arguments}`)
            return null;
        }
        if (arguments.length == 1 && (arguments[0] instanceof LoadResArgs)) {
            return arguments[0];
        }

        let result: LoadResArgs = new LoadResArgs();
        if (typeof arguments[0] == 'string') result.url = arguments[0];
        if (arguments[0] instanceof Array) result.urls = arguments[0];

        for (let i: number = 0; i < arguments.length; i++) {
            if (i == 1 && isChildClassOf(arguments[i], cc.Asset)) {
                result.type = arguments[i];
            } else if (i == arguments.length - 1 && typeof arguments[i] == 'string') {
                result.use = arguments[i];
            }
            else if (typeof arguments[i] == 'function') {
                if (arguments.length > i + 1 && typeof arguments[i + 1] == 'function') {
                    result.onProcess = arguments[i];
                } else {
                    result.onCompleted = arguments[i];
                }
            }
        }
        return result;
    }

    /**
     * 加载资源
     * @param url 加载的资源url
     * @param urls 加载的资源url数组
     * @param type 加载的资源类型
     * @param onProcess 加载的进度
     * @param onCompleted 加载完成时的回调
     * @param use 资源被引用的标记
     */
    loadRes(url: string, use?: string)
    loadRes(url: string, onCompleted: CompletedCallback, use?: string)
    loadRes(url: string, type: typeof cc.Asset)
    loadRes(url: string, type: typeof cc.Asset, onCompleted: CompletedCallback, use?: string)
    loadRes(url: string, onProcess: ProcessCallBack, onCompleted: CompletedCallback, use?: string)
    loadRes(url: string, type: typeof cc.Asset, onProcess: ProcessCallBack, onCompleted: CompletedCallback, use?: string)
    loadRes() {
        let loadResArgs: LoadResArgs = this.getOrSetLoadResArgs.apply(this, arguments);
        let finishCallBack = (error: Error, resource: any) => {
            if (!error) {
                this.cacheCurResRefInfo(loadResArgs.url, loadResArgs.type, resource, loadResArgs.use);
            }
            if (loadResArgs.onCompleted) {
                loadResArgs.onCompleted(error, resource);
            }
        };
        let res = this.getOrSetResCacheInfo(loadResArgs.url);
        if (res && res.obj) {
            finishCallBack(null, res.obj);
        } else {
            cc.resources.load(loadResArgs.url, loadResArgs.type, loadResArgs.onProcess, finishCallBack);
        }
    }


    /**
     * 通过资源的url去释放资源，注：这里会对资源进行decRef处理
     * @param url 资源的url
     * @param decRef 当资源的refCount引用计数为0之后，引擎会自动释放这个资源
     */
    releaseResByDecRef(url: string, type?: typeof cc.Asset) {
        let info = this.getOrSetResCacheInfo(url);
        if (info && info.obj) {
            info.obj.decRef();
            console.log(`[调试]:释放了${url}的资源，该资源的引用计数为：`, info.obj.refCount);
            if (info.obj.refCount == 0) {
                delete this.resInfo[url];
            }
        }
    }


    /**
     * 通过资源实例的方式释放资源，资源可以是`Prefab`、`SpriteAtlas`等
     * @param asset 资源实例
     * @param tag  release 系列接口（例如 release、releaseAsset、releaseAll）会直接释放资源，而不会进行释放检查，只有其依赖资源会进行释放检查。所以当显式调用 release 系列接口时，可以确保资源本身一定会被释放。
     */
    releaseResByAsset(asset: any) {
        cc.assetManager.releaseAsset(asset);
    }
}

export let assetManager: AssetManager = AssetManager.getInstance();
