import { assetManager } from "./AssetManager";
import { UIView } from "./UIView";


/**
 * ui的加载信息
 */
export interface uiLoadArgs {
    viewName: string,
    type: typeof cc.Asset,
    url: string,
    zOrder: number,
    parent: string,
    preventTouch?: boolean,
    isQuickClose?: boolean,
    isCacheRes?: boolean
}


/**
 * ui的配置
 */
export enum uiConfig {
    uiTest1,
    uiTest2,
}


export let UICF: { [key: number]: uiLoadArgs } = {
    [uiConfig.uiTest1]: { url: '/prefab/uiTest1', viewName: 'uiTest1', type: cc.Prefab, zOrder: 1, parent: 'uiRoot', isCacheRes: false },
    [uiConfig.uiTest2]: { url: '/prefab/uiTest1', viewName: 'uiTest1', type: cc.Prefab, zOrder: 1, parent: 'uiRoot', isCacheRes: false },
}



export class UIManager {

    private static instance: UIManager = null;
    public static getInstance(): UIManager {
        if (!this.instance) {
            this.instance = new UIManager();
        }
        return this.instance;
    }
    /**
     * 用于缓存ui预制件的实例，注意各ui在关闭的时候，请还原脚本内的申请的变量内存空间
     */
    private UIStack: any = {};

    private _useKey: string = "UseKey_By_UIManager";

    /**
     * 打开ui
     */
    public open(uiInfo): void {
        let uiArgsConf = UICF[uiInfo.index];
        var viewInstance: cc.Node = this.getView(uiArgsConf.viewName);
        if (!viewInstance) {
            assetManager.loadRes(uiArgsConf.url, uiArgsConf.type, (error, prefab) => {
                if (!error) {
                    console.log(`[调试]:加载了${uiArgsConf.viewName}的预制件资源`, prefab);
                    let viewNode: cc.Node = cc.instantiate(prefab);
                    let child = cc.director.getScene().getChildByName('Canvas');
                    let viewRoot = child.getChildByName(uiArgsConf.parent);
                    viewNode.zIndex = uiArgsConf.zOrder;
                    viewRoot.addChild(viewNode);
                    this.UIStack[uiArgsConf.viewName] = viewNode;
                    let viewClass: UIView = viewNode.getComponent(uiArgsConf.viewName);
                    if (viewClass && viewClass.onOpen) {
                        viewClass.onOpen(uiInfo.data);
                        viewClass.addEventListener();
                        uiInfo.delayCallBack && uiInfo.delayCallBack();
                    }
                }
            }, this._useKey);
        } else {
            let child = cc.director.getScene().getChildByName('Canvas');
            let viewRoot = child.getChildByName(uiArgsConf.parent);
            viewRoot.addChild(viewInstance);
            viewInstance.zIndex = uiArgsConf.zOrder;
            viewInstance.active = true;
            var viewClass: UIView = viewInstance.getComponent(uiArgsConf.viewName);
            if (viewClass && viewClass.onRelive) {
                viewClass.onOpen(uiInfo.data);
                viewClass.addEventListener();
                uiInfo.delayCallBack && uiInfo.delayCallBack();
            }
        };
        if (uiInfo.successCallBack) {
            uiInfo.successCallBack();
        }
        if (uiArgsConf.isQuickClose) {
            let backGround = viewInstance.getChildByName('background');
            if (!backGround) {
                backGround = new cc.Node()
                backGround.name = 'background';
                backGround.setContentSize(cc.winSize);
                viewInstance.addChild(backGround, -1);
            }
            backGround.targetOff(cc.Node.EventType.TOUCH_START);
            backGround.on(cc.Node.EventType.TOUCH_START, (event: cc.Event.EventCustom) => {
                event.stopPropagation();
                this.close(uiInfo.index);
            }, backGround);
        }
    }


    /**
     * 预加载某些ui，适应一些情况的处理
     * @param viewName ui对应的预制件名称
     * @param this.UIStack 将ui存储在栈中，关闭时进行释放
     */
    public preloadUI(viewIndex: number, finishCallBack?: Function) {
        let uiArgsConf = UICF[viewIndex];
        assetManager.loadRes(uiArgsConf.url, cc.Prefab, (error, prefab) => {
            if (!error) {
                let viewNode: cc.Node = cc.instantiate(prefab);
                //只有加到canvas任意节点下，脚本的生命周期函数才会被调用
                this.UIStack[uiArgsConf.viewName] = viewNode;
                finishCallBack && finishCallBack(true);
            } else {
                finishCallBack && finishCallBack(false)
            }
        })
    }

    /**
     * 获取指定ui
     */
    public getView(viewName: string) {
        var viewNode: cc.Node = this.UIStack[viewName];
        if (viewNode) {
            return viewNode;
        } else {
            //debug.error("[调试]:在UIStack中找不到指定UI", viewName)
            return null;
        }
    }


    /**
     * 隐藏指定ui
     */
    public hide(viewIndex: number, flag: boolean) {
        let uiArgsConf = UICF[viewIndex];
        var viewInstance: cc.Node = this.UIStack[uiArgsConf.viewName];
        if (viewInstance != null) {
            viewInstance.active = flag;
        }
    }


    /**
     * 根据指定的viewIndex关闭界面
     * @param viewIndex 界面的Index
     * @param tag 当调用close关闭界面的时候，需要判断isCacheRes标记，如果存在标记，则当前界面只会从主场景节点下移除，而不会`destroy`，当前界面所有引用的资源也不会被释放
     */
    public close(viewIndex: number) {
        let uiArgsConf = UICF[viewIndex];
        var viewInstance: cc.Node = this.UIStack[uiArgsConf.viewName];
        if (!viewInstance) {
            console.error("[调试]:UI已经被释放了", uiArgsConf.viewName)
            return;
        }
        var viewClass: UIView = viewInstance.getComponent(uiArgsConf.viewName);
        if (viewClass) {
            viewClass.onClose();
            viewClass.removeEventListener();
        } else {
            console.error("[调试]:预制件上找不到该脚本", uiArgsConf.viewName)
        }
        if (viewInstance) {
            let child = cc.director.getScene().getChildByName('Canvas');
            let viewRoot = child.getChildByName(uiArgsConf.parent)
            viewRoot.removeChild(viewInstance);
            if (!uiArgsConf.isCacheRes) {
                viewInstance.destroy();
                assetManager.releaseResByDecRef(uiArgsConf.url);
                delete this.UIStack[uiArgsConf.viewName];
            }
        }
    }

    /**
     * 释放
     */
    realse() {
        this.UIStack = null;
    }
}

export let uiManager: UIManager = UIManager.getInstance();