import { _decorator, Component, Node, Vec3, Vec2, animation } from 'cc';
import { AStarNode, E_Node_Type } from './AStarNode';
const { ccclass, property } = _decorator;

/// <summary>
/// AStar寻路算法的基本原理就是不停的找自己周围的点，选出一个新的点作为起点再循环的找
/// 1.寻路消耗公式：
///           f(寻路消耗)=g(离起点的距离)+h(离终点的距离)
/// 2.开启列表：
/// 每次从新的点找周围的点时，如果周围的点已经在开启列表或者关闭列表中了，我们就不去管它了
/// 3.关闭列表：
/// 每次往关闭列表中放点时，我们都应该判断这个点是不是和终点一样，如果是一样证明路径找完了，如果不一样，继续找。
/// 4.格子对象的父对象   
/// </summary>
@ccclass('AStartMgr')
export class AStarMgr extends Component {

    private static instance: AStarMgr = null;

    public static Instance(): AStarMgr {
        return this.instance;
    }

    /**
     * 地图的宽
     */
    private mapW: number;

    /**
     * 地图的高
     */
    private mapH: number;

    /**
     * 地图相关所有的格子对象容器
     */
    public nodes: Array<any> = new Array<any>();

    /**
     * 开启列表
     */
    private openLst: Array<any> = new Array<any>();

    /**
     *  关闭列表
     */
    private closeLst: Array<any> = new Array<any>();

    /**
     * 路径
     */
    private path: Array<any> = new Array<any>();

    constructor() {
        super();
        AStarMgr.instance = this;
    }


    public Init() {
        //AStarMgr.instance=this;
        console.log("初始化AStarMgr...");
    }

    /**
     * 初始化地图
     * @param w 地图的宽
     * @param h 地图的高
     */
    public InitMapInfo(w, h) {
        //根据宽高 创建格子 阻挡的问题 我们可以随机阻挡
        //因为我们现在没有地图相关的数据

        let self = this;

        //记录宽高
        self.mapW = w;
        self.mapH = h;

        //声明容器可以装多少个格子
        //self.nodes= new AStarNode[w][h];

        //生成格子
        for (let i: number = 0; i < w; ++i) {
            //console.log("self.nodes.length:" + self.nodes.length);
            self.nodes[i] = [];
            for (let j: number = 0; j < h; ++j) {
                //用三目运算符随机生成阻挡格子
                //应该从地图配置表读取生成的
                let nodeObj: AStarNode = new AStarNode(i, j, (self.RandomNum(0, 100) < 20 ? E_Node_Type.Stop : E_Node_Type.Walk));
                self.nodes[i][j] = nodeObj;  
            }
        }
    }

    /**
     * 寻路方法
     * @param startPos 开始点
     * @param endPos 结束点
     */
    public FindPath(startPos: Vec2, endPos: Vec2) {
        let self = this;
        //实际项目中 传入的点往往是 坐标系中的点
        //我们这里省略换算的步骤 直接认为它是传进来的格子坐标

        //首先判断 传入的两个点 是否合法
        //1.首先 要在地图范围内
        //如果不合法 应该直接 返回null 意味着不能寻路
        if (startPos.x < 0 || startPos.x >= this.mapW ||
            startPos.y < 0 || startPos.y >= this.mapH ||
            endPos.x < 0 || endPos.x >= this.mapW ||
            endPos.y < 0 || endPos.y >= this.mapH) {
            console.log("开始或者结束点在地图格子范围外");
            return null;
        }
        //2.要不是阻挡
        //得到起点和终点  对应的格子
        let start: AStarNode = self.nodes[startPos.x][startPos.y];
        let end: AStarNode = self.nodes[endPos.x][endPos.y];

        if (start.type == E_Node_Type.Stop || end.type == E_Node_Type.Stop) {
            console.log("开始或者结束点是阻挡");
            return null;
        }

        //清空上一次相关的数据 避免他们影响  这一次的寻路计算

        //清空关闭和开启列表
        self.closeLst = [];
        self.openLst = [];

        //把开始点放入关闭列表中
        start.father = null;
        start.f = 0;
        start.g = 0;
        start.h = 0;
        self.closeLst.push(start);

        while (true) {
            //从起点开始 找周围的点 并放入开启列表中
            //左上  x-1 y-1
            self.FindNearlyNodeToOpenLst(start.x - 1, start.y - 1, 1.4, start, end);

            //上 x  y-1
            self.FindNearlyNodeToOpenLst(start.x, start.y - 1, 1, start, end);

            //右上  x+1 y-1
            self.FindNearlyNodeToOpenLst(start.x + 1, start.y - 1, 1.4, start, end);

            //左 x-1 y
            self.FindNearlyNodeToOpenLst(start.x - 1, start.y, 1, start, end);

            //右 x+1 y
            self.FindNearlyNodeToOpenLst(start.x + 1, start.y, 1, start, end);

            //左下 x-1 y+1
            self.FindNearlyNodeToOpenLst(start.x - 1, start.y + 1, 1.4, start, end);

            //下 x y+1
            self.FindNearlyNodeToOpenLst(start.x, start.y + 1, 1, start, end);

            //右下 x+1 y+1
            self.FindNearlyNodeToOpenLst(start.x + 1, start.y + 1, 1.4, start, end);

            //死路判断  开启列表为空 都还没有找到终点 就认为是死路
            if (self.openLst.length == 0) {
                console.log("死路...");
                return null;
            }

            //选出开启列表中 寻路消耗最小的点
            self.openLst.sort(self.SortOpenLst);
            console.log("****************");
            for (let i = 0; i < self.openLst.length; ++i) {
                let targetNode = self.openLst[i];
                console.log("点:" + targetNode.x + " ," + targetNode.y + "  :g=" + targetNode.g + "  h=" + targetNode.h + "  f=" + targetNode.f);
            }

            //放入关闭列表中 然后再从开启列表中移除
            self.closeLst.push(self.openLst[0]);
            //找得这个点 又变成新的起点 进入下一次寻路计算了
            start = self.openLst[0];
            self.openLst.shift();
            //如果这个点已经是终点 那么得到最终结果返回出去
            //如果这个点 不是终点 那么继续寻路
            if (start == end) {
                //找完了 找到路径
                //返回路径
                self.path = [];
                self.path.push(end);
                //father是空时是找到start点，start点的father就是空
                while (end.father != null) {
                    self.path.push(end.father);
                    end = end.father;
                }

                //列表翻正得到正确的路径
                self.path.reverse();
                return self.path;
            }
        }
    }

    /**
     * 排序函数
     * @param a 
     * @param b 
     */
    private SortOpenLst(a, b): any {
        if (a.f > b.f)
            return 1;
        else if (a.f == b.f)
            return 1;
        else
            return -1;
    }


    /**
     *  把临近的点放入开启列表中的函数
     * @param x 
     * @param y 
     * @param g 
     * @param father 
     * @param end 
     */
    private FindNearlyNodeToOpenLst(x, y, g, father, end) {

        let self = this;

        //边界判断
        if (x < 0 || x >= self.mapW ||
            y < 0 || y >= self.mapH)
            return;

        //在范围内 再去取点
        let tempNode = self.nodes[x][y];
        //判断这些点 是否是边界 是否是阻挡 是否在开启或者关闭列表 如果都不是 才放入开启列表
        if (tempNode == null ||
            tempNode.type == E_Node_Type.Stop ||
            self.IsInArray(self.closeLst, tempNode) ||
            self.IsInArray(self.openLst, tempNode))
            return;

        //计算f值
        //f=g+h
        //记录父对象
        tempNode.father = father;
        //计算g  我离起点的距离  就是我父亲离起点的距离 +我离我父亲的距离
        tempNode.g = father.g + g;
        //曼哈顿街区算法
        tempNode.h = Math.abs(end.x - tempNode.x) + Math.abs(end.y - tempNode.y);
        tempNode.f = tempNode.g + tempNode.h;

        //如果通过了上面的合法验证 存放到开启列表中
        self.openLst.push(tempNode);
    }

    /**
     * 是否在数组里
     * @param arr 
     * @param target
     */
    private IsInArray(arr: Array<any>, target): boolean {

        let isExit: boolean = false;

        arr.find(function (val) {
            if (val == target)
                return isExit = true;
        })

        return isExit;
    }


    /**
     * 生成随机数
     * @param min 最小值
     * @param max 最大值
     */
    private RandomNum(min: number, max): number {
        switch (arguments.length) {
            case 1:
                return parseInt((Math.random() * min + 1).toString(), 10)
                break;
            case 2:
                return parseInt((Math.random() * (max - min + 1) + min).toString(), 10);
                break;
            default:
                return 0;
                break;
        }
    }


}
