// 加载管理模块，资源加载、场景加载等。为了实现加载进度统计
export default class LoadManager {
    /**
     * 场景要加载的资源列表
     */
    private rateOfLoadRes: number[];
    private totalCount: number;
    private completedCount: number;
    /**
     * 场景加载率
     */
    private rateOfLoadScene: number;
    private completeCallback: Function;

    /**
     * 加载场景，同时会去加载loadResList.json里该场景名对应的资源
     * @param name 场景名
     * @param completeCallback 所有资源加载完成后的回调函数
     */
    loadScene(name: string, completeCallback: Function = null) {
        this.completeCallback = completeCallback;

        //加载场景
        let self = this;
        cc.director.preloadScene(name, (c, t, i) => {
            self.rateOfLoadScene = c / t;
        }, (e, a) => {
            cc.director.loadScene(name, () => {
                self.rateOfLoadScene = 1;
                self.checkComplete();
            })
        });
    }

    setCompleteCallback(callFunc: Function) {
        this.completeCallback = callFunc;
    }

    /**
     * 获得资源加载率
     * 1表示加载完或没有加载任务
     */
    getLoadedRate(): number {
        if (this.totalCount === this.completedCount && (this.rateOfLoadScene === 1 || this.rateOfLoadScene === 0))
            return 1;

        let sum: number = 0;
        let r: number;
        for (let i = 0; i < this.rateOfLoadRes.length; i++)
            sum += this.rateOfLoadRes[i];
        return (sum + this.rateOfLoadScene) / (this.rateOfLoadRes.length + 1);
    }

    /**
     * 加载单个资源
     * @param url 路径
     */
    loadRes(url: string, type: any) {
        let self = this;

        //加载任务下标
        let no: number = this.rateOfLoadRes.length;
        this.rateOfLoadRes.push(0);
        this.totalCount++;

        cc.loader.loadRes(url, type, (e, r: any) => {
            if (e) {
                console.error(e);
                return;
            }
            self.rateOfLoadRes[no] = 1;
            self.completedCount++;

            self.checkComplete()
        });
    }

    loadResArray(urls: string[], type: any) {
        let url: string;
        for (url of urls)
            this.loadRes(url, type);
    }

    /**
     * 加载文件夹下的所有资源
     * @param url 文件夹路径
     */
    loadResDir(url: string, type: any) {
        let self = this;
        let no: number = this.rateOfLoadRes.length;
        this.rateOfLoadRes.push(0);
        this.totalCount++;

        cc.loader.loadResDir(url, type, (totalCount: number, completedCount: number, item) => {
            self.rateOfLoadRes[no] = completedCount / totalCount;
        }, (e, r: any[]) => {
            if (e) {
                console.error(e);
                return;
            }
            self.rateOfLoadRes[no] = 1;
            self.completedCount++;

            self.checkComplete()
        })
    }

    /**
     * 加载完后执行回调函数和清理加载记录
     */
    private checkComplete() {
        if (this.getLoadedRate() === 1) {
            if (this.completeCallback)
                this.completeCallback();
            this.initRecord();
        }
    }

    /**
     * 初始化加载记录
     */
    private initRecord() {
        this.rateOfLoadRes = [];
        this.totalCount = 0;
        this.completedCount = 0;
        this.rateOfLoadScene = 0;
        this.completeCallback = null;
    }
}

let loadMgr: LoadManager = new LoadManager();
export { loadMgr };
