import PlayerAnim from "./playerAnim";
import { eventMgr, EventType } from "../common/eventManager";
import PlayerState from "./playerState";
import MathUtil from "../common/mathUtil";

const { ccclass, property } = cc._decorator;

@ccclass
export default class Player extends cc.Component {

    private anim: PlayerAnim = null;
    private rigid: cc.RigidBody = null;
    private collider: cc.BoxCollider = null;

    /* 属性 */
    private state: PlayerState;
    /**
     * 移动速度
     */
    readonly moveSpeed: number = 100;
    /**
     * 空中移动速度
     */
    readonly speedInAir: number = 50;
    /**
     * 跳跃力
     */
    readonly jumpForce: number = 100;
    readonly climbWallSpeed: number = 50;

    /* 记录 */

    /**
     * 点击跳跃键的次数
     */
    private clickJumpButCount: number;
    /**
     * 是否点击了移动按钮，1为右
     */
    private isClickMoveBut: number;
    /**
     * 此时碰撞的碰撞器
     */
    private colliderArray: cc.BoxCollider[];

    /* 控制 */
    private isUpdate: boolean = false;
    /**
     * 在地面
     */
    private inLand: boolean = false;

    onLoad() {
        this.anim = this.node.getComponent("playerAnim");
        this.rigid = this.node.getComponent(cc.RigidBody);
        this.collider = this.node.getComponent(cc.BoxCollider);

        eventMgr.onEvent(EventType.ResLoadComplete, () => {
            this.init();
            this.anim.init();
            this.isUpdate = true;
        }, this)
    }

    private init() {
        this.clickJumpButCount = 0;
        this.isClickMoveBut = 0;
        this.colliderArray = [];
        this.inLand = false;
        this.setState(PlayerState.idleState);
    }

    setState(s: PlayerState) {
        this.state = s;
        this.state.setPlayer(this);
    }

    getState(): PlayerState {
        return this.state;
    }

    getAnim(): PlayerAnim {
        return this.anim;
    }

    getClickJumpButCount(): number {
        return this.clickJumpButCount;
    }

    getClickMoveBut(): number {
        return this.isClickMoveBut;
    }


    /*          用户或环境输入相关的函数            */
    moveBut(dir: number) {
        this.isClickMoveBut = dir;
        this.state.moveBut(dir);
    }

    releaseMoveBut() {
        this.isClickMoveBut = 0;
        this.state.releaseMoveBut();
    }

    jumpBut() {
        if (this.clickJumpButCount === 2)
            return;
        this.clickJumpButCount++;
        this.state.jumpBut();
    }

    /**
     * 落地
     */
    land() {
        this.clickJumpButCount = 0;
        this.state.land();
    }

    downSpeed() {
        this.state.downSpeed();
    }

    upSpeed() {
        this.state.upSpeed();
    }

    dead() {
        this.state.dead();
        this.anim.dead();
        this.isUpdate = false;
    }

    /*                 刚体相关的函数                      */
    onBeginContact(contact, selfCollider: cc.BoxCollider, otherCollider: cc.BoxCollider) {
        this.colliderArray.push(otherCollider);

        if (!this.inLand) {
            this.inLand = this.checkLand();
            if (this.inLand)
                this.land();
        }
    }

    onEndContact(contact, selfCollider: cc.BoxCollider, otherCollider: cc.BoxCollider) {
        MathUtil.removeItemFromArray(otherCollider, this.colliderArray);

        if (this.inLand)
            this.inLand = this.checkLand();
    }

    getRigid(): cc.RigidBody {
        return this.rigid;
    }

    /**
     * 移动
     * @param dir 
     * @param speed 
     */
    move(dir: number, speed: number) {
        this.node.scaleX = dir;
        this.rigid.linearVelocity.x = dir * speed;
    }

    addForce(f: cc.Vec2) {
        this.rigid.applyLinearImpulse(f, this.rigid.getLocalCenter(), true);
    }

    /*                状态检查                               */

    /**
     * 检查y轴线速度
     */
    private checkYSpeed() {
        if (this.rigid.linearVelocity.y > 1)
            this.upSpeed();
        else if (this.rigid.linearVelocity.y < -1)
            this.downSpeed();
    }

    /**
    * 检查是否在地面
    */
    private checkLand(): boolean {
        let ret = false;

        //角色
        let size: cc.Size = this.collider.size;
        let wp: cc.Vec2 = MathUtil.getWorldCoordinate(this.node);
        let minY: number = wp.y - size.height / 2;

        //角色接触的碰撞器
        let other: cc.Vec2;
        let c: cc.BoxCollider;
        let maxY: number;

        for (c of this.colliderArray) {
            //碰撞器世界坐标
            other = MathUtil.getWorldCoordinate(c.node).add(c.offset);
            maxY = other.y + c.size.height / 2;

            //碰撞器是否在角色下面，即碰撞器最高点小于角色最低点
            if (minY >= maxY) {
                ret = true;
                break;
            }
        }

        return ret;
    }

    update(dt) {
        if (!this.isUpdate)
            return;

        this.checkYSpeed();
    }
}
