/****************************************************************************
 Copyright (c) 2014-2016 Chukong Technologies Inc.
 Copyright (c) 2017-2018 Xiamen Yaji Software Co., Ltd.

 http://www.cocos2d-x.org

 Permission is hereby granted, free of charge, to any person obtaining a copy
 of this software and associated documentation files (the "Software"), to deal
 in the Software without restriction, including without limitation the rights
 to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the Software is
 furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 THE SOFTWARE.
 ****************************************************************************/

#include "platform/CCPlatformConfig.h"

// Webview not available on tvOS
#if (CC_TARGET_PLATFORM == CC_PLATFORM_IOS) && !defined(CC_TARGET_OS_TVOS)

#include "WebView-inl.h"
#include "platform/CCApplication.h"
#include "platform/ios/CCEAGLView-ios.h"
#include "platform/CCFileUtils.h"
#import <WebKit/WebKit.h>

static std::string getFixedBaseUrl(const std::string& baseUrl)
{
    std::string fixedBaseUrl;
    if (baseUrl.empty() || baseUrl.at(0) != '/') {
        fixedBaseUrl = [[[NSBundle mainBundle] resourcePath] UTF8String];
        fixedBaseUrl += "/";
        fixedBaseUrl += baseUrl;
    }
    else {
        fixedBaseUrl = baseUrl;
    }

    size_t pos = 0;
    while ((pos = fixedBaseUrl.find(" ")) != std::string::npos) {
        fixedBaseUrl.replace(pos, 1, "%20");
    }

    if (fixedBaseUrl.at(fixedBaseUrl.length() - 1) != '/') {
        fixedBaseUrl += "/";
    }

    return fixedBaseUrl;
}

@interface UIWebViewWrapper : NSObject
@property (nonatomic) std::function<bool(std::string url)> shouldStartLoading;
@property (nonatomic) std::function<void(std::string url)> didFinishLoading;
@property (nonatomic) std::function<void(std::string url)> didFailLoading;
@property (nonatomic) std::function<void(std::string url)> onJsCallback;

@property(nonatomic, readonly, getter=canGoBack) BOOL canGoBack;
@property(nonatomic, readonly, getter=canGoForward) BOOL canGoForward;

+ (instancetype)webViewWrapper;

- (void)setVisible:(bool)visible;

- (void)setBounces:(bool)bounces;

- (void)setFrameWithX:(float)x y:(float)y width:(float)width height:(float)height;

- (void)setJavascriptInterfaceScheme:(const std::string &)scheme;

- (void)loadData:(const std::string &)data MIMEType:(const std::string &)MIMEType textEncodingName:(const std::string &)encodingName baseURL:(const std::string &)baseURL;

- (void)loadHTMLString:(const std::string &)string baseURL:(const std::string &)baseURL;

- (void)loadUrl:(const std::string &)urlString;

- (void)loadFile:(const std::string &)filePath;

- (void)stopLoading;

- (void)reload;

- (void)evaluateJS:(const std::string &)js;

- (void)goBack;

- (void)goForward;

- (void)setScalesPageToFit:(const bool)scalesPageToFit;
@end


@interface UIWebViewWrapper () <WKUIDelegate,WKNavigationDelegate>
@property(nonatomic, retain) WKWebView *wkWebView;
@property(nonatomic, copy) NSString *jsScheme;
@end

@implementation UIWebViewWrapper {

}

+ (instancetype)webViewWrapper {
    return [[[self alloc] init] autorelease];
}

- (instancetype)init {
    self = [super init];
    if (self) {
        self.wkWebView = nil;
        self.shouldStartLoading = nullptr;
        self.didFinishLoading = nullptr;
        self.didFailLoading = nullptr;
    }
    return self;
}

- (void)dealloc {
    self.wkWebView.UIDelegate = nil;
    self.wkWebView.navigationDelegate = nil;
    [self.wkWebView removeFromSuperview];
    self.wkWebView = nil;
    self.jsScheme = nil;
    [super dealloc];
}

- (void)setupWebView {
    if (!self.wkWebView) {
        self.wkWebView = [[[WKWebView alloc] init] autorelease];
        self.wkWebView.UIDelegate = self;
        self.wkWebView.navigationDelegate = self;
        [self.wkWebView setOpaque:NO];
        [self.wkWebView setBackgroundColor:[UIColor clearColor]];
    }
//    self.uiWebView.allowsInlineMediaPlayback = YES;
    if (!self.wkWebView.superview) {
        auto eaglview = (CCEAGLView*)cocos2d::Application::getInstance()->getView();
        [eaglview addSubview:self.wkWebView];
    }
}

- (void)setVisible:(bool)visible {
    self.wkWebView.hidden = !visible;
}

- (void)setBounces:(bool)bounces {
    self.wkWebView.scrollView.bounces = bounces;
}

- (void)setFrameWithX:(float)x y:(float)y width:(float)width height:(float)height {
    if (!self.wkWebView) {[self setupWebView];}
    CGRect newFrame = CGRectMake(x, y, width, height);
    if (!CGRectEqualToRect(self.wkWebView.frame, newFrame)) {
        self.wkWebView.frame = CGRectMake(x, y, width, height);
    }
}

- (void)setJavascriptInterfaceScheme:(const std::string &)scheme {
    self.jsScheme = @(scheme.c_str());
}

- (void)loadData:(const std::string &)data MIMEType:(const std::string &)MIMEType textEncodingName:(const std::string &)encodingName baseURL:(const std::string &)baseURL {
    auto path = [[NSBundle mainBundle] resourcePath];
    path = [path stringByAppendingPathComponent:@(baseURL.c_str())];
    auto url = [NSURL fileURLWithPath:path];
    [self.wkWebView loadData:[NSData dataWithBytes:data.c_str() length:data.length()] MIMEType:@(MIMEType.c_str()) characterEncodingName:@(encodingName.c_str()) baseURL:url];
//    [self.wkWebView loadData:[NSData dataWithBytes:data.c_str() length:data.length()]
//                    MIMEType:@(MIMEType.c_str())
//            textEncodingName:@(encodingName.c_str())
//                     baseURL:[NSURL URLWithString:@(getFixedBaseUrl(baseURL).c_str())]];
}

- (void)loadHTMLString:(const std::string &)string baseURL:(const std::string &)baseURL {
    if (!self.wkWebView) {[self setupWebView];}
    auto path = [[NSBundle mainBundle] resourcePath];
    path = [path stringByAppendingPathComponent:@(baseURL.c_str())];
    auto url = [NSURL fileURLWithPath:path];
    [self.wkWebView loadHTMLString:@(string.c_str()) baseURL:url];
//    [self.uiWebView loadHTMLString:@(string.c_str()) baseURL:[NSURL URLWithString:@(getFixedBaseUrl(baseURL).c_str())]];
}

- (void)loadUrl:(const std::string &)urlString {
    if (!self.wkWebView) {[self setupWebView];}
    NSURL *url = [NSURL URLWithString:@(urlString.c_str())];
    NSURLRequest *request = [NSURLRequest requestWithURL:url];
    [self.wkWebView loadRequest:request];
}

- (void)loadFile:(const std::string &)filePath {
    if (!self.wkWebView) {[self setupWebView];}
    NSURL *url = [NSURL fileURLWithPath:@(filePath.c_str())];
    NSURLRequest *request = [NSURLRequest requestWithURL:url];
    [self.wkWebView loadRequest:request];
}

- (void)stopLoading {
    [self.wkWebView stopLoading];
}

- (void)reload {
    [self.wkWebView reload];
}

- (BOOL)canGoForward {
    return self.wkWebView.canGoForward;
}

- (BOOL)canGoBack {
    return self.wkWebView.canGoBack;
}

- (void)goBack {
    [self.wkWebView goBack];
}

- (void)goForward {
    [self.wkWebView goForward];
}

- (void)evaluateJS:(const std::string &)js {
    if (!self.wkWebView) {[self setupWebView];}
    [self.wkWebView evaluateJavaScript:@(js.c_str()) completionHandler:nil];
}

- (void)setScalesPageToFit:(const bool)scalesPageToFit {
//    if (!self.wkWebView) {[self setupWebView];}
//    self.uiWebView.scalesPageToFit = scalesPageToFit;
}
#pragma mark - WKNavigationDelegate
- (void)webView:(WKWebView *)webView decidePolicyForNavigationAction:(WKNavigationAction *)navigationAction decisionHandler:(void (^)(WKNavigationActionPolicy))decisionHandler {
    NSString *url = [webView.URL absoluteString];
    if ([[webView.URL scheme] isEqualToString:self.jsScheme]) {
        self.onJsCallback([url UTF8String]);
        decisionHandler(WKNavigationActionPolicyCancel);
        return;
    }
    if (self.shouldStartLoading && url) {
        if (self.shouldStartLoading([url UTF8String]) )
            decisionHandler(WKNavigationActionPolicyAllow);
        else
            decisionHandler(WKNavigationActionPolicyCancel);

        return;
    }

    decisionHandler(WKNavigationActionPolicyAllow);
}

- (void)webView:(WKWebView *)webView didFinishNavigation:(WKNavigation *)navigation {
    if (self.didFinishLoading) {
        NSString *url = [webView.URL absoluteString];
        self.didFinishLoading([url UTF8String]);
    }
}

- (void)webView:(WKWebView *)webView didFailProvisionalNavigation:(WKNavigation *)navigation withError:(NSError *)error {
    if (self.didFailLoading) {
        NSString *errorInfo = error.userInfo[NSURLErrorFailingURLStringErrorKey];
        if (errorInfo) {
            self.didFailLoading([errorInfo UTF8String]);
        }
    }
}

#pragma WKUIDelegate

// Implement js alert function.
- (void)webView:(WKWebView *)webView runJavaScriptAlertPanelWithMessage:(NSString *)message initiatedByFrame:(WKFrameInfo *)frame completionHandler:(void (^)())completionHandler
{
    UIAlertController *alertController = [UIAlertController alertControllerWithTitle:message
                                                                             message:nil
                                                                      preferredStyle:UIAlertControllerStyleAlert];
    [alertController addAction:[UIAlertAction actionWithTitle:@"Ok"
                                                        style:UIAlertActionStyleCancel
                                                      handler:^(UIAlertAction *action) {
                                                          completionHandler();
                                                      }]];

    auto rootViewController = [UIApplication sharedApplication].keyWindow.rootViewController;
    [rootViewController presentViewController:alertController animated:YES completion:^{}];
}
#pragma mark - UIWebViewDelegate
//- (BOOL)webView:(UIWebView *)webView shouldStartLoadWithRequest:(NSURLRequest *)request navigationType:(UIWebViewNavigationType)navigationType {
//    NSString *url = [[request URL] absoluteString];
//    if ([[[request URL] scheme] isEqualToString:self.jsScheme]) {
//        self.onJsCallback([url UTF8String]);
//        return NO;
//    }
//    if (self.shouldStartLoading && url) {
//        return self.shouldStartLoading([url UTF8String]);
//    }
//    return YES;
//}
//
//- (void)webViewDidFinishLoad:(UIWebView *)webView {
//    if (self.didFinishLoading) {
//        NSString *url = [[webView.request URL] absoluteString];
//        self.didFinishLoading([url UTF8String]);
//    }
//}
//
//- (void)webView:(UIWebView *)webView didFailLoadWithError:(NSError *)error {
//    if (self.didFailLoading) {
//        NSString *url = error.userInfo[NSURLErrorFailingURLStringErrorKey];
//        if (url) {
//            self.didFailLoading([url UTF8String]);
//        }
//    }
//}

@end



namespace cocos2d {

    WebViewImpl::WebViewImpl(WebView *webView)
    : _uiWebViewWrapper([UIWebViewWrapper webViewWrapper]),
    _webView(webView) {
        [_uiWebViewWrapper retain];

        _uiWebViewWrapper.shouldStartLoading = [this](std::string url) {
            if (this->_webView->_onShouldStartLoading) {
                return this->_webView->_onShouldStartLoading(this->_webView, url);
            }
            return true;
        };
        _uiWebViewWrapper.didFinishLoading = [this](std::string url) {
            if (this->_webView->_onDidFinishLoading) {
                this->_webView->_onDidFinishLoading(this->_webView, url);
            }
        };
        _uiWebViewWrapper.didFailLoading = [this](std::string url) {
            if (this->_webView->_onDidFailLoading) {
                this->_webView->_onDidFailLoading(this->_webView, url);
            }
        };
        _uiWebViewWrapper.onJsCallback = [this](std::string url) {
            if (this->_webView->_onJSCallback) {
                this->_webView->_onJSCallback(this->_webView, url);
            }
        };
    }

    WebViewImpl::~WebViewImpl(){
        [_uiWebViewWrapper release];
        _uiWebViewWrapper = nullptr;
    }

    void WebViewImpl::setJavascriptInterfaceScheme(const std::string &scheme) {
        [_uiWebViewWrapper setJavascriptInterfaceScheme:scheme];
    }

    void WebViewImpl::loadData(const Data &data,
                               const std::string &MIMEType,
                               const std::string &encoding,
                               const std::string &baseURL) {

        std::string dataString(reinterpret_cast<char *>(data.getBytes()), static_cast<unsigned int>(data.getSize()));
        [_uiWebViewWrapper loadData:dataString MIMEType:MIMEType textEncodingName:encoding baseURL:baseURL];
    }

    void WebViewImpl::loadHTMLString(const std::string &string, const std::string &baseURL) {
        [_uiWebViewWrapper loadHTMLString:string baseURL:baseURL];
    }

    void WebViewImpl::loadURL(const std::string &url) {
        [_uiWebViewWrapper loadUrl:url];
    }

    void WebViewImpl::loadFile(const std::string &fileName) {
        auto fullPath = cocos2d::FileUtils::getInstance()->fullPathForFilename(fileName);
        [_uiWebViewWrapper loadFile:fullPath];
    }

    void WebViewImpl::stopLoading() {
        [_uiWebViewWrapper stopLoading];
    }

    void WebViewImpl::reload() {
        [_uiWebViewWrapper reload];
    }

    bool WebViewImpl::canGoBack() {
        return _uiWebViewWrapper.canGoBack;
    }

    bool WebViewImpl::canGoForward() {
        return _uiWebViewWrapper.canGoForward;
    }

    void WebViewImpl::goBack() {
        [_uiWebViewWrapper goBack];
    }

    void WebViewImpl::goForward() {
        [_uiWebViewWrapper goForward];
    }

    void WebViewImpl::evaluateJS(const std::string &js) {
        [_uiWebViewWrapper evaluateJS:js];
    }

    void WebViewImpl::setBounces(bool bounces) {
        [_uiWebViewWrapper setBounces:bounces];
    }

    void WebViewImpl::setScalesPageToFit(const bool scalesPageToFit) {
        [_uiWebViewWrapper setScalesPageToFit:scalesPageToFit];
    }

    void WebViewImpl::setVisible(bool visible){
        [_uiWebViewWrapper setVisible:visible];
    }

    void WebViewImpl::setFrame(float x, float y, float width, float height){
        auto eaglview = (CCEAGLView*)cocos2d::Application::getInstance()->getView();
        auto scaleFactor = [eaglview contentScaleFactor];
        [_uiWebViewWrapper setFrameWithX:x/scaleFactor
                                       y:y/scaleFactor
                                   width:width/scaleFactor
                                  height:height/scaleFactor];
    }
} //namespace cocos2d

#endif // CC_TARGET_PLATFORM == CC_PLATFORM_IOS
