const { ccclass, property } = cc._decorator;

@ccclass
export default class MainScene extends cc.Component {

    @property({
        type: cc.Camera,
        tooltip: "主摄像头",
    })
    camera: cc.Camera = null;

    @property({
        type: cc.ScrollView,
        tooltip: "装地图的滚动面板",
    })
    scrollView: cc.ScrollView = null;

    @property({
        type: cc.Node,
        tooltip: "地图节点",
    })
    bg: cc.Node = null;

    @property({
        type: cc.Node,
        tooltip: "玩家角色节点",
    })
    player: cc.Node = null;

    @property({
        type: cc.Prefab,
        tooltip: "事件点预制体",
    })
    eventPoint: cc.Prefab = null;

    private static _inst: MainScene = null;
    public static get inst(): MainScene {
        return this._inst;
    }

    start() {
        cc.view.setOrientation(cc.macro.ORIENTATION_LANDSCAPE);

        MainScene._inst = this;
        window['MainScene'] = MainScene;

        this.onScroll(this.scrollView);

        this.initEventPoint();
        this.movePlayer(1);

        this.scrollView.content.on(cc.Node.EventType.TOUCH_END, this.onBgClick, this);
    }

    /**
     * 面板内容点击事件
     * @param ev 
     */
    onBgClick(ev: cc.Event.EventMouse) {
        // 根据点击的点获取一条由屏幕射向屏幕内的射线
        let ray = this.camera.getRay(ev.getLocation());
        // 根据传入的根节点向下检测，并返回检测结果
        // 返回的结果包含了节点和距离
        let results = (<any>cc.geomUtils.intersect).raycast(this.player.parent, ray);
        cc.log(results)
        for (let i = 0; i < results.length; i++) {
            if (results[i].node.name == "EventPoint") {
                this.movePlayer(results[i].node["event_id"]);
            }
        }
    }

    /**
     * 面板滚动事件，当滚动的时候改变z轴
     * @param sv 
     */
    onScroll(sv: cc.ScrollView) {
        let angleX = this.bg.eulerAngles.x;
        this.bg.z = sv.content.y * Math.tan(Math.PI * angleX / 180) + 200;
    }

    /** 当前所处事件id */
    curEventId = null;

    /**
     * 将玩家移动到目标事件id处
     * @param id 
     */
    async movePlayer(id: number) {
        id = Math.floor(id);
        if (this.curEventId === id) return;
        this.curEventId = id;
        let data = this.getEventData(id);
        this.player.position = cc.v3(data.posX, data.posY, 0);
        this.scrollView.scrollTo(cc.v2((data.posX + this.bg.width / 2) / this.bg.width, (data.posY + this.bg.height / 2) / this.bg.height), 0.2);
    }

    /**
     * 获取事件配置
     * @param id 
     */
    getEventData(id: number): mapEventData {
        return this.eventData[id] || <any>{};
    }

    eventData: mapEventData[] = <any>{
        1: { posX: 16, posY: -24 },
        2: { posX: 214, posY: 98 },
        3: { posX: 293, posY: 130 },
        4: { posX: 366, posY: 130 },
        5: { posX: 96, posY: 185 },
        6: { posX: 256, posY: 207 },
        7: { posX: 406, posY: 176 },
        8: { posX: 22, posY: 178 },
        9: { posX: 111, posY: 239 },
        10: { posX: -9, posY: 277 },
        11: { posX: -85, posY: 292 },
        12: { posX: -153, posY: 233 },
        13: { posX: -40, posY: 219 },
        14: { posX: -65, posY: 100 },
        15: { posX: -179, posY: 7 },
    };

    /** 存储事件点实例 */
    eventCellList = []

    /**
     * 添加事件点到地图上
     */
    initEventPoint() {
        for (const i in this.eventData) {
            const data = this.eventData[i];
            let point = cc.instantiate(this.eventPoint);
            point.parent = this.player.parent;
            point.position = cc.v3(data.posX, data.posY, 0);
            point.zIndex = -1;
            point["event_id"] = i;
            this.eventCellList.push(point);
        }
    }

}

/** 地图事件配置 */
interface mapEventData {
    /** 坐标x */
    posX: number
    /** 坐标y */
    posY: number
}
