/**
 * 事件名
 */
export enum EVENT_NAME {
    EXAMPLE = 'example',
};

type Listener = (event: EVENT_NAME, arg: any) => void;
interface EventList { target: Object; listener: Listener; }

export default class Events {
    private static _eventsMap: Map<EVENT_NAME, EventList[]> = new Map();
    
    /**
     * @method 注册事件
     * @param {string} event 事件名
     * @param {Listener} listener 处理事件的监听函数
     * @param {object} target 监听函数绑定的this对象
     */
    public static on(event: EVENT_NAME, listener: Listener, target: Object) {
        if (!listener) {
            console.warn(`[on] event: ${event} 事件监听函数为空`);
        }
    
        let list = this._eventsMap.get(event);
        if (!list) {
            list = [];
            this._eventsMap.set(event, list);
        }
    
        let idx = list.findIndex((e) => e.target === target && e.listener === listener);
        if (idx !== -1) {
            console.warn(`[on] event: ${event} 重复注册`);
        }
    
        list.push({
            target: target,
            listener: listener
        });
    }

    /**
     * @method 移除事件
     * @param {string} event 事件名
     * @param {Listener} listener 处理事件的监听函数
     * @param {object} target 监听函数绑定的this对象
     */
    public static off(event: EVENT_NAME, listener: Listener, target: Object) {
        let list = this._eventsMap.get(event);
        if (!list) {
            console.warn(`[off] event: ${event} 事件不存在，无法移除`);
            return;
        }

        let idx = list.findIndex((e) => e.target === target && e.listener === listener);
        if (idx !== -1) {
            list.splice(idx, 1);
        }

        if (list.length === 0) {
            this._eventsMap.delete(event);
        }
    }

    /**
     * @method 派发事件
     * @param {string} event 事件名
     * @param {Array} args 事件参数，若需要传多个参数，请修改args为数组或者使用object
     */
    public static emit(event: EVENT_NAME, args: any = null) {
        console.info(`[events.emit] event:${event}`);
        let list = this._eventsMap.get(event);
        if (!list) {
            console.warn(`[emit] event: ${event} 事件不存在`);
        } else {
            list.forEach((eventObject) => eventObject.listener.call(eventObject.target, event, args));
        }
    }
}
