
                (function() {
                    var nodeEnv = typeof require !== 'undefined' && typeof process !== 'undefined';
                    var __module = nodeEnv ? module : {exports:{}};
                    var __filename = 'preview-scripts/assets/Script/Common/Bezier/bezierUtils.js';
                    var __require = nodeEnv ? function (request) {
                        return cc.require(request);
                    } : function (request) {
                        return __quick_compile_project__.require(request, __filename);
                    };
                    function __define (exports, require, module) {
                        if (!nodeEnv) {__quick_compile_project__.registerModule(__filename, module);}"use strict";
cc._RF.push(module, '8725d0YLEVJk7Zouq78bYl3', 'bezierUtils');
// Script/Common/Bezier/bezierUtils.js

"use strict";

exports.__esModule = true;
exports.utils = void 0;

var _bezier = require("./bezier.js");

// math-inlining.
var abs = Math.abs,
    cos = Math.cos,
    sin = Math.sin,
    acos = Math.acos,
    atan2 = Math.atan2,
    sqrt = Math.sqrt,
    pow = Math.pow; // cube root function yielding real roots

function crt(v) {
  return v < 0 ? -pow(-v, 1 / 3) : pow(v, 1 / 3);
} // trig constants


var pi = Math.PI,
    tau = 2 * pi,
    quart = pi / 2,
    // float precision significant decimal
epsilon = 0.000001,
    // extremas used in bbox calculation and similar algorithms
nMax = Number.MAX_SAFE_INTEGER || 9007199254740991,
    nMin = Number.MIN_SAFE_INTEGER || -9007199254740991,
    // a zero coordinate, which is surprisingly useful
ZERO = {
  x: 0,
  y: 0,
  z: 0
}; // Bezier utility functions

var utils = {
  // Legendre-Gauss abscissae with n=24 (x_i values, defined at i=n as the roots of the nth order Legendre polynomial Pn(x))
  Tvalues: [-0.0640568928626056260850430826247450385909, 0.0640568928626056260850430826247450385909, -0.1911188674736163091586398207570696318404, 0.1911188674736163091586398207570696318404, -0.3150426796961633743867932913198102407864, 0.3150426796961633743867932913198102407864, -0.4337935076260451384870842319133497124524, 0.4337935076260451384870842319133497124524, -0.5454214713888395356583756172183723700107, 0.5454214713888395356583756172183723700107, -0.6480936519369755692524957869107476266696, 0.6480936519369755692524957869107476266696, -0.7401241915785543642438281030999784255232, 0.7401241915785543642438281030999784255232, -0.8200019859739029219539498726697452080761, 0.8200019859739029219539498726697452080761, -0.8864155270044010342131543419821967550873, 0.8864155270044010342131543419821967550873, -0.9382745520027327585236490017087214496548, 0.9382745520027327585236490017087214496548, -0.9747285559713094981983919930081690617411, 0.9747285559713094981983919930081690617411, -0.9951872199970213601799974097007368118745, 0.9951872199970213601799974097007368118745],
  // Legendre-Gauss weights with n=24 (w_i values, defined by a function linked to in the Bezier primer article)
  Cvalues: [0.1279381953467521569740561652246953718517, 0.1279381953467521569740561652246953718517, 0.1258374563468282961213753825111836887264, 0.1258374563468282961213753825111836887264, 0.121670472927803391204463153476262425607, 0.121670472927803391204463153476262425607, 0.1155056680537256013533444839067835598622, 0.1155056680537256013533444839067835598622, 0.1074442701159656347825773424466062227946, 0.1074442701159656347825773424466062227946, 0.0976186521041138882698806644642471544279, 0.0976186521041138882698806644642471544279, 0.086190161531953275917185202983742667185, 0.086190161531953275917185202983742667185, 0.0733464814110803057340336152531165181193, 0.0733464814110803057340336152531165181193, 0.0592985849154367807463677585001085845412, 0.0592985849154367807463677585001085845412, 0.0442774388174198061686027482113382288593, 0.0442774388174198061686027482113382288593, 0.0285313886289336631813078159518782864491, 0.0285313886289336631813078159518782864491, 0.0123412297999871995468056670700372915759, 0.0123412297999871995468056670700372915759],
  arcfn: function arcfn(t, derivativeFn) {
    var d = derivativeFn(t);
    var l = d.x * d.x + d.y * d.y;

    if (typeof d.z !== "undefined") {
      l += d.z * d.z;
    }

    return sqrt(l);
  },
  compute: function compute(t, points, _3d) {
    // shortcuts
    if (t === 0) {
      points[0].t = 0;
      return points[0];
    }

    var order = points.length - 1;

    if (t === 1) {
      points[order].t = 1;
      return points[order];
    }

    var mt = 1 - t;
    var p = points; // constant?

    if (order === 0) {
      points[0].t = t;
      return points[0];
    } // linear?


    if (order === 1) {
      var ret = {
        x: mt * p[0].x + t * p[1].x,
        y: mt * p[0].y + t * p[1].y,
        t: t
      };

      if (_3d) {
        ret.z = mt * p[0].z + t * p[1].z;
      }

      return ret;
    } // quadratic/cubic curve?


    if (order < 4) {
      var mt2 = mt * mt,
          t2 = t * t,
          a,
          b,
          c,
          d = 0;

      if (order === 2) {
        p = [p[0], p[1], p[2], ZERO];
        a = mt2;
        b = mt * t * 2;
        c = t2;
      } else if (order === 3) {
        a = mt2 * mt;
        b = mt2 * t * 3;
        c = mt * t2 * 3;
        d = t * t2;
      }

      var _ret = {
        x: a * p[0].x + b * p[1].x + c * p[2].x + d * p[3].x,
        y: a * p[0].y + b * p[1].y + c * p[2].y + d * p[3].y,
        t: t
      };

      if (_3d) {
        _ret.z = a * p[0].z + b * p[1].z + c * p[2].z + d * p[3].z;
      }

      return _ret;
    } // higher order curves: use de Casteljau's computation


    var dCpts = JSON.parse(JSON.stringify(points));

    while (dCpts.length > 1) {
      for (var i = 0; i < dCpts.length - 1; i++) {
        dCpts[i] = {
          x: dCpts[i].x + (dCpts[i + 1].x - dCpts[i].x) * t,
          y: dCpts[i].y + (dCpts[i + 1].y - dCpts[i].y) * t
        };

        if (typeof dCpts[i].z !== "undefined") {
          dCpts[i].z = dCpts[i].z + (dCpts[i + 1].z - dCpts[i].z) * t;
        }
      }

      dCpts.splice(dCpts.length - 1, 1);
    }

    dCpts[0].t = t;
    return dCpts[0];
  },
  computeWithRatios: function computeWithRatios(t, points, ratios, _3d) {
    var mt = 1 - t,
        r = ratios,
        p = points;
    var f1 = r[0],
        f2 = r[1],
        f3 = r[2],
        f4 = r[3],
        d; // spec for linear

    f1 *= mt;
    f2 *= t;

    if (p.length === 2) {
      d = f1 + f2;
      return {
        x: (f1 * p[0].x + f2 * p[1].x) / d,
        y: (f1 * p[0].y + f2 * p[1].y) / d,
        z: !_3d ? false : (f1 * p[0].z + f2 * p[1].z) / d,
        t: t
      };
    } // upgrade to quadratic


    f1 *= mt;
    f2 *= 2 * mt;
    f3 *= t * t;

    if (p.length === 3) {
      d = f1 + f2 + f3;
      return {
        x: (f1 * p[0].x + f2 * p[1].x + f3 * p[2].x) / d,
        y: (f1 * p[0].y + f2 * p[1].y + f3 * p[2].y) / d,
        z: !_3d ? false : (f1 * p[0].z + f2 * p[1].z + f3 * p[2].z) / d,
        t: t
      };
    } // upgrade to cubic


    f1 *= mt;
    f2 *= 1.5 * mt;
    f3 *= 3 * mt;
    f4 *= t * t * t;

    if (p.length === 4) {
      d = f1 + f2 + f3 + f4;
      return {
        x: (f1 * p[0].x + f2 * p[1].x + f3 * p[2].x + f4 * p[3].x) / d,
        y: (f1 * p[0].y + f2 * p[1].y + f3 * p[2].y + f4 * p[3].y) / d,
        z: !_3d ? false : (f1 * p[0].z + f2 * p[1].z + f3 * p[2].z + f4 * p[3].z) / d,
        t: t
      };
    }
  },
  derive: function derive(points, _3d) {
    var dpoints = [];

    for (var p = points, d = p.length, c = d - 1; d > 1; d--, c--) {
      var list = [];

      for (var j = 0, dpt; j < c; j++) {
        dpt = {
          x: c * (p[j + 1].x - p[j].x),
          y: c * (p[j + 1].y - p[j].y)
        };

        if (_3d) {
          dpt.z = c * (p[j + 1].z - p[j].z);
        }

        list.push(dpt);
      }

      dpoints.push(list);
      p = list;
    }

    return dpoints;
  },
  between: function between(v, m, M) {
    return m <= v && v <= M || utils.approximately(v, m) || utils.approximately(v, M);
  },
  approximately: function approximately(a, b, precision) {
    return abs(a - b) <= (precision || epsilon);
  },
  length: function length(derivativeFn) {
    var z = 0.5,
        len = utils.Tvalues.length;
    var sum = 0;

    for (var i = 0, t; i < len; i++) {
      t = z * utils.Tvalues[i] + z;
      sum += utils.Cvalues[i] * utils.arcfn(t, derivativeFn);
    }

    return z * sum;
  },
  map: function map(v, ds, de, ts, te) {
    var d1 = de - ds,
        d2 = te - ts,
        v2 = v - ds,
        r = v2 / d1;
    return ts + d2 * r;
  },
  lerp: function lerp(r, v1, v2) {
    var ret = {
      x: v1.x + r * (v2.x - v1.x),
      y: v1.y + r * (v2.y - v1.y)
    };

    if (v1.z !== undefined && v2.z !== undefined) {
      ret.z = v1.z + r * (v2.z - v1.z);
    }

    return ret;
  },
  pointToString: function pointToString(p) {
    var s = p.x + "/" + p.y;

    if (typeof p.z !== "undefined") {
      s += "/" + p.z;
    }

    return s;
  },
  pointsToString: function pointsToString(points) {
    return "[" + points.map(utils.pointToString).join(", ") + "]";
  },
  copy: function copy(obj) {
    return JSON.parse(JSON.stringify(obj));
  },
  angle: function angle(o, v1, v2) {
    var dx1 = v1.x - o.x,
        dy1 = v1.y - o.y,
        dx2 = v2.x - o.x,
        dy2 = v2.y - o.y,
        cross = dx1 * dy2 - dy1 * dx2,
        dot = dx1 * dx2 + dy1 * dy2;
    return atan2(cross, dot);
  },
  // round as string, to avoid rounding errors
  round: function round(v, d) {
    var s = "" + v;
    var pos = s.indexOf(".");
    return parseFloat(s.substring(0, pos + 1 + d));
  },
  dist: function dist(p1, p2) {
    var dx = p1.x - p2.x,
        dy = p1.y - p2.y;
    return sqrt(dx * dx + dy * dy);
  },
  closest: function closest(LUT, point) {
    var mdist = pow(2, 63),
        mpos,
        d;
    LUT.forEach(function (p, idx) {
      d = utils.dist(point, p);

      if (d < mdist) {
        mdist = d;
        mpos = idx;
      }
    });
    return {
      mdist: mdist,
      mpos: mpos
    };
  },
  abcratio: function abcratio(t, n) {
    // see ratio(t) note on http://pomax.github.io/bezierinfo/#abc
    if (n !== 2 && n !== 3) {
      return false;
    }

    if (typeof t === "undefined") {
      t = 0.5;
    } else if (t === 0 || t === 1) {
      return t;
    }

    var bottom = pow(t, n) + pow(1 - t, n),
        top = bottom - 1;
    return abs(top / bottom);
  },
  projectionratio: function projectionratio(t, n) {
    // see u(t) note on http://pomax.github.io/bezierinfo/#abc
    if (n !== 2 && n !== 3) {
      return false;
    }

    if (typeof t === "undefined") {
      t = 0.5;
    } else if (t === 0 || t === 1) {
      return t;
    }

    var top = pow(1 - t, n),
        bottom = pow(t, n) + top;
    return top / bottom;
  },
  lli8: function lli8(x1, y1, x2, y2, x3, y3, x4, y4) {
    var nx = (x1 * y2 - y1 * x2) * (x3 - x4) - (x1 - x2) * (x3 * y4 - y3 * x4),
        ny = (x1 * y2 - y1 * x2) * (y3 - y4) - (y1 - y2) * (x3 * y4 - y3 * x4),
        d = (x1 - x2) * (y3 - y4) - (y1 - y2) * (x3 - x4);

    if (d == 0) {
      return false;
    }

    return {
      x: nx / d,
      y: ny / d
    };
  },
  lli4: function lli4(p1, p2, p3, p4) {
    var x1 = p1.x,
        y1 = p1.y,
        x2 = p2.x,
        y2 = p2.y,
        x3 = p3.x,
        y3 = p3.y,
        x4 = p4.x,
        y4 = p4.y;
    return utils.lli8(x1, y1, x2, y2, x3, y3, x4, y4);
  },
  lli: function lli(v1, v2) {
    return utils.lli4(v1, v1.c, v2, v2.c);
  },
  makeline: function makeline(p1, p2) {
    return new _bezier.Bezier(p1.x, p1.y, (p1.x + p2.x) / 2, (p1.y + p2.y) / 2, p2.x, p2.y);
  },
  findbbox: function findbbox(sections) {
    var mx = nMax,
        my = nMax,
        MX = nMin,
        MY = nMin;
    sections.forEach(function (s) {
      var bbox = s.bbox();
      if (mx > bbox.x.min) mx = bbox.x.min;
      if (my > bbox.y.min) my = bbox.y.min;
      if (MX < bbox.x.max) MX = bbox.x.max;
      if (MY < bbox.y.max) MY = bbox.y.max;
    });
    return {
      x: {
        min: mx,
        mid: (mx + MX) / 2,
        max: MX,
        size: MX - mx
      },
      y: {
        min: my,
        mid: (my + MY) / 2,
        max: MY,
        size: MY - my
      }
    };
  },
  shapeintersections: function shapeintersections(s1, bbox1, s2, bbox2, curveIntersectionThreshold) {
    if (!utils.bboxoverlap(bbox1, bbox2)) return [];
    var intersections = [];
    var a1 = [s1.startcap, s1.forward, s1.back, s1.endcap];
    var a2 = [s2.startcap, s2.forward, s2.back, s2.endcap];
    a1.forEach(function (l1) {
      if (l1.virtual) return;
      a2.forEach(function (l2) {
        if (l2.virtual) return;
        var iss = l1.intersects(l2, curveIntersectionThreshold);

        if (iss.length > 0) {
          iss.c1 = l1;
          iss.c2 = l2;
          iss.s1 = s1;
          iss.s2 = s2;
          intersections.push(iss);
        }
      });
    });
    return intersections;
  },
  makeshape: function makeshape(forward, back, curveIntersectionThreshold) {
    var bpl = back.points.length;
    var fpl = forward.points.length;
    var start = utils.makeline(back.points[bpl - 1], forward.points[0]);
    var end = utils.makeline(forward.points[fpl - 1], back.points[0]);
    var shape = {
      startcap: start,
      forward: forward,
      back: back,
      endcap: end,
      bbox: utils.findbbox([start, forward, back, end])
    };

    shape.intersections = function (s2) {
      return utils.shapeintersections(shape, shape.bbox, s2, s2.bbox, curveIntersectionThreshold);
    };

    return shape;
  },
  getminmax: function getminmax(curve, d, list) {
    if (!list) return {
      min: 0,
      max: 0
    };
    var min = nMax,
        max = nMin,
        t,
        c;

    if (list.indexOf(0) === -1) {
      list = [0].concat(list);
    }

    if (list.indexOf(1) === -1) {
      list.push(1);
    }

    for (var i = 0, len = list.length; i < len; i++) {
      t = list[i];
      c = curve.get(t);

      if (c[d] < min) {
        min = c[d];
      }

      if (c[d] > max) {
        max = c[d];
      }
    }

    return {
      min: min,
      mid: (min + max) / 2,
      max: max,
      size: max - min
    };
  },
  align: function align(points, line) {
    var tx = line.p1.x,
        ty = line.p1.y,
        a = -atan2(line.p2.y - ty, line.p2.x - tx),
        d = function d(v) {
      return {
        x: (v.x - tx) * cos(a) - (v.y - ty) * sin(a),
        y: (v.x - tx) * sin(a) + (v.y - ty) * cos(a)
      };
    };

    return points.map(d);
  },
  roots: function roots(points, line) {
    line = line || {
      p1: {
        x: 0,
        y: 0
      },
      p2: {
        x: 1,
        y: 0
      }
    };
    var order = points.length - 1;
    var aligned = utils.align(points, line);

    var reduce = function reduce(t) {
      return 0 <= t && t <= 1;
    };

    if (order === 2) {
      var _a = aligned[0].y,
          _b = aligned[1].y,
          _c = aligned[2].y,
          _d = _a - 2 * _b + _c;

      if (_d !== 0) {
        var m1 = -sqrt(_b * _b - _a * _c),
            m2 = -_a + _b,
            _v = -(m1 + m2) / _d,
            v2 = -(-m1 + m2) / _d;

        return [_v, v2].filter(reduce);
      } else if (_b !== _c && _d === 0) {
        return [(2 * _b - _c) / (2 * _b - 2 * _c)].filter(reduce);
      }

      return [];
    } // see http://www.trans4mind.com/personal_development/mathematics/polynomials/cubicAlgebra.htm


    var pa = aligned[0].y,
        pb = aligned[1].y,
        pc = aligned[2].y,
        pd = aligned[3].y;
    var d = -pa + 3 * pb - 3 * pc + pd,
        a = 3 * pa - 6 * pb + 3 * pc,
        b = -3 * pa + 3 * pb,
        c = pa;

    if (utils.approximately(d, 0)) {
      // this is not a cubic curve.
      if (utils.approximately(a, 0)) {
        // in fact, this is not a quadratic curve either.
        if (utils.approximately(b, 0)) {
          // in fact in fact, there are no solutions.
          return [];
        } // linear solution:


        return [-c / b].filter(reduce);
      } // quadratic solution:


      var _q = sqrt(b * b - 4 * a * c),
          a2 = 2 * a;

      return [(_q - b) / a2, (-b - _q) / a2].filter(reduce);
    } // at this point, we know we need a cubic solution:


    a /= d;
    b /= d;
    c /= d;
    var p = (3 * b - a * a) / 3,
        p3 = p / 3,
        q = (2 * a * a * a - 9 * a * b + 27 * c) / 27,
        q2 = q / 2,
        discriminant = q2 * q2 + p3 * p3 * p3;
    var u1, v1, x1, x2, x3;

    if (discriminant < 0) {
      var mp3 = -p / 3,
          mp33 = mp3 * mp3 * mp3,
          r = sqrt(mp33),
          t = -q / (2 * r),
          cosphi = t < -1 ? -1 : t > 1 ? 1 : t,
          phi = acos(cosphi),
          crtr = crt(r),
          t1 = 2 * crtr;
      x1 = t1 * cos(phi / 3) - a / 3;
      x2 = t1 * cos((phi + tau) / 3) - a / 3;
      x3 = t1 * cos((phi + 2 * tau) / 3) - a / 3;
      return [x1, x2, x3].filter(reduce);
    } else if (discriminant === 0) {
      u1 = q2 < 0 ? crt(-q2) : -crt(q2);
      x1 = 2 * u1 - a / 3;
      x2 = -u1 - a / 3;
      return [x1, x2].filter(reduce);
    } else {
      var sd = sqrt(discriminant);
      u1 = crt(-q2 + sd);
      v1 = crt(q2 + sd);
      return [u1 - v1 - a / 3].filter(reduce);
    }
  },
  droots: function droots(p) {
    // quadratic roots are easy
    if (p.length === 3) {
      var a = p[0],
          b = p[1],
          c = p[2],
          d = a - 2 * b + c;

      if (d !== 0) {
        var m1 = -sqrt(b * b - a * c),
            m2 = -a + b,
            v1 = -(m1 + m2) / d,
            v2 = -(-m1 + m2) / d;
        return [v1, v2];
      } else if (b !== c && d === 0) {
        return [(2 * b - c) / (2 * (b - c))];
      }

      return [];
    } // linear roots are even easier


    if (p.length === 2) {
      var _a2 = p[0],
          _b2 = p[1];

      if (_a2 !== _b2) {
        return [_a2 / (_a2 - _b2)];
      }

      return [];
    }

    return [];
  },
  curvature: function curvature(t, d1, d2, _3d, kOnly) {
    var num,
        dnm,
        adk,
        dk,
        k = 0,
        r = 0; //
    // We're using the following formula for curvature:
    //
    //              x'y" - y'x"
    //   k(t) = ------------------
    //           (x'² + y'²)^(3/2)
    //
    // from https://en.wikipedia.org/wiki/Radius_of_curvature#Definition
    //
    // With it corresponding 3D counterpart:
    //
    //          sqrt( (y'z" - y"z')² + (z'x" - z"x')² + (x'y" - x"y')²)
    //   k(t) = -------------------------------------------------------
    //                     (x'² + y'² + z'²)^(3/2)
    //

    var d = utils.compute(t, d1);
    var dd = utils.compute(t, d2);
    var qdsum = d.x * d.x + d.y * d.y;

    if (_3d) {
      num = sqrt(pow(d.y * dd.z - dd.y * d.z, 2) + pow(d.z * dd.x - dd.z * d.x, 2) + pow(d.x * dd.y - dd.x * d.y, 2));
      dnm = pow(qdsum + d.z * d.z, 3 / 2);
    } else {
      num = d.x * dd.y - d.y * dd.x;
      dnm = pow(qdsum, 3 / 2);
    }

    if (num === 0 || dnm === 0) {
      return {
        k: 0,
        r: 0
      };
    }

    k = num / dnm;
    r = dnm / num; // We're also computing the derivative of kappa, because
    // there is value in knowing the rate of change for the
    // curvature along the curve. And we're just going to
    // ballpark it based on an epsilon.

    if (!kOnly) {
      // compute k'(t) based on the interval before, and after it,
      // to at least try to not introduce forward/backward pass bias.
      var pk = utils.curvature(t - 0.001, d1, d2, _3d, true).k;
      var nk = utils.curvature(t + 0.001, d1, d2, _3d, true).k;
      dk = (nk - k + (k - pk)) / 2;
      adk = (abs(nk - k) + abs(k - pk)) / 2;
    }

    return {
      k: k,
      r: r,
      dk: dk,
      adk: adk
    };
  },
  inflections: function inflections(points) {
    if (points.length < 4) return []; // FIXME: TODO: add in inflection abstraction for quartic+ curves?

    var p = utils.align(points, {
      p1: points[0],
      p2: points.slice(-1)[0]
    }),
        a = p[2].x * p[1].y,
        b = p[3].x * p[1].y,
        c = p[1].x * p[2].y,
        d = p[3].x * p[2].y,
        v1 = 18 * (-3 * a + 2 * b + 3 * c - d),
        v2 = 18 * (3 * a - b - 3 * c),
        v3 = 18 * (c - a);

    if (utils.approximately(v1, 0)) {
      if (!utils.approximately(v2, 0)) {
        var t = -v3 / v2;
        if (0 <= t && t <= 1) return [t];
      }

      return [];
    }

    var d2 = 2 * v1;
    if (utils.approximately(d2, 0)) return [];
    var trm = v2 * v2 - 4 * v1 * v3;
    if (trm < 0) return [];
    var sq = Math.sqrt(trm);
    return [(sq - v2) / d2, -(v2 + sq) / d2].filter(function (r) {
      return 0 <= r && r <= 1;
    });
  },
  bboxoverlap: function bboxoverlap(b1, b2) {
    var dims = ["x", "y"],
        len = dims.length;

    for (var i = 0, dim, l, t, d; i < len; i++) {
      dim = dims[i];
      l = b1[dim].mid;
      t = b2[dim].mid;
      d = (b1[dim].size + b2[dim].size) / 2;
      if (abs(l - t) >= d) return false;
    }

    return true;
  },
  expandbox: function expandbox(bbox, _bbox) {
    if (_bbox.x.min < bbox.x.min) {
      bbox.x.min = _bbox.x.min;
    }

    if (_bbox.y.min < bbox.y.min) {
      bbox.y.min = _bbox.y.min;
    }

    if (_bbox.z && _bbox.z.min < bbox.z.min) {
      bbox.z.min = _bbox.z.min;
    }

    if (_bbox.x.max > bbox.x.max) {
      bbox.x.max = _bbox.x.max;
    }

    if (_bbox.y.max > bbox.y.max) {
      bbox.y.max = _bbox.y.max;
    }

    if (_bbox.z && _bbox.z.max > bbox.z.max) {
      bbox.z.max = _bbox.z.max;
    }

    bbox.x.mid = (bbox.x.min + bbox.x.max) / 2;
    bbox.y.mid = (bbox.y.min + bbox.y.max) / 2;

    if (bbox.z) {
      bbox.z.mid = (bbox.z.min + bbox.z.max) / 2;
    }

    bbox.x.size = bbox.x.max - bbox.x.min;
    bbox.y.size = bbox.y.max - bbox.y.min;

    if (bbox.z) {
      bbox.z.size = bbox.z.max - bbox.z.min;
    }
  },
  pairiteration: function pairiteration(c1, c2, curveIntersectionThreshold) {
    var c1b = c1.bbox(),
        c2b = c2.bbox(),
        r = 100000,
        threshold = curveIntersectionThreshold || 0.5;

    if (c1b.x.size + c1b.y.size < threshold && c2b.x.size + c2b.y.size < threshold) {
      return [(r * (c1._t1 + c1._t2) / 2 | 0) / r + "/" + (r * (c2._t1 + c2._t2) / 2 | 0) / r];
    }

    var cc1 = c1.split(0.5),
        cc2 = c2.split(0.5),
        pairs = [{
      left: cc1.left,
      right: cc2.left
    }, {
      left: cc1.left,
      right: cc2.right
    }, {
      left: cc1.right,
      right: cc2.right
    }, {
      left: cc1.right,
      right: cc2.left
    }];
    pairs = pairs.filter(function (pair) {
      return utils.bboxoverlap(pair.left.bbox(), pair.right.bbox());
    });
    var results = [];
    if (pairs.length === 0) return results;
    pairs.forEach(function (pair) {
      results = results.concat(utils.pairiteration(pair.left, pair.right, threshold));
    });
    results = results.filter(function (v, i) {
      return results.indexOf(v) === i;
    });
    return results;
  },
  getccenter: function getccenter(p1, p2, p3) {
    var dx1 = p2.x - p1.x,
        dy1 = p2.y - p1.y,
        dx2 = p3.x - p2.x,
        dy2 = p3.y - p2.y,
        dx1p = dx1 * cos(quart) - dy1 * sin(quart),
        dy1p = dx1 * sin(quart) + dy1 * cos(quart),
        dx2p = dx2 * cos(quart) - dy2 * sin(quart),
        dy2p = dx2 * sin(quart) + dy2 * cos(quart),
        // chord midpoints
    mx1 = (p1.x + p2.x) / 2,
        my1 = (p1.y + p2.y) / 2,
        mx2 = (p2.x + p3.x) / 2,
        my2 = (p2.y + p3.y) / 2,
        // midpoint offsets
    mx1n = mx1 + dx1p,
        my1n = my1 + dy1p,
        mx2n = mx2 + dx2p,
        my2n = my2 + dy2p,
        // intersection of these lines:
    arc = utils.lli8(mx1, my1, mx1n, my1n, mx2, my2, mx2n, my2n),
        r = utils.dist(arc, p1); // arc start/end values, over mid point:

    var s = atan2(p1.y - arc.y, p1.x - arc.x),
        m = atan2(p2.y - arc.y, p2.x - arc.x),
        e = atan2(p3.y - arc.y, p3.x - arc.x),
        _; // determine arc direction (cw/ccw correction)


    if (s < e) {
      // if s<m<e, arc(s, e)
      // if m<s<e, arc(e, s + tau)
      // if s<e<m, arc(e, s + tau)
      if (s > m || m > e) {
        s += tau;
      }

      if (s > e) {
        _ = e;
        e = s;
        s = _;
      }
    } else {
      // if e<m<s, arc(e, s)
      // if m<e<s, arc(s, e + tau)
      // if e<s<m, arc(s, e + tau)
      if (e < m && m < s) {
        _ = e;
        e = s;
        s = _;
      } else {
        e += tau;
      }
    } // assign and done.


    arc.s = s;
    arc.e = e;
    arc.r = r;
    return arc;
  },
  numberSort: function numberSort(a, b) {
    return a - b;
  }
};
exports.utils = utils;

cc._RF.pop();
                    }
                    if (nodeEnv) {
                        __define(__module.exports, __require, __module);
                    }
                    else {
                        __quick_compile_project__.registerModuleFunc(__filename, function () {
                            __define(__module.exports, __require, __module);
                        });
                    }
                })();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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