import { _decorator, clamp01, Component, lerp, Node, v3, Vec3 } from 'cc';
const { ccclass, property } = _decorator;

@ccclass('MoveTest')
export class MoveTest extends Component {

    //目标路线，移动
    @property({ type: [Node] })
    pathNodes: Node[] = [];

    /**每段路径的距离(节省计算量，在路径第一次使用时计算) */
    private _pathDistances: number[] = [];
    private _pathProgress: number[] = [];

    //当前目标点序号
    curIndex: number = -1;
    //当前目标节点
    curTarget: Node = null;
    //一个节点
    nextTarget: Node = null;

    /** 移动速度 */
    @property
    moveSpeed: number = 10;


    @property
    loop: boolean = true;

    /**跑圈完成，如果是loop模式，则会循环，不会自动完成 */
    isComplete = false;
    isPlaying = false;

    /**跑圈进度 */
    progress: number = 0;


    /**移动的距离 */
    distance: number = 0;
    totalDistance: number = 0;

    start() {
        //从哪个节点开始跑
        this.scheduleOnce(() => {
            this.runStart(0);
        }, 1)
    }

    /**
     * 从哪个节点位置开始跑
     * @param index 
     */
    runStart(index: number) {

        //如果首位不相连，自动添加首节点到数组尾部，这样就是衔接
        if (this.loop) {
            let startNode = this.pathNodes[0];
            let endNode = this.pathNodes[this.pathNodes.length - 1];
            if (startNode != endNode) {
                this.pathNodes.push(startNode);
            }
        }

        this.isPlaying = true;
        //计算路径数据
        this.generatePathData();
        this.refreshIndexPath(index);

    }

    generatePathData() {
        let fullDistance = 0;

        //将路径转换为，距离进度数组，方便取值
        this._pathDistances.length = 0;

        for (let i = 0; i < this.pathNodes.length; i++) {
            const curNode = this.pathNodes[i];
            const nextNode = this.pathNodes[(i + 1)];
            if (curNode == null || nextNode == null) break;
            let distance = Vec3.distance(curNode.worldPosition, nextNode.worldPosition);
            fullDistance += distance;
            this._pathDistances.push(fullDistance);
        }
        this.totalDistance = fullDistance;

        //将百分比路径距离转换为，百分比进度数组，方便取值
        this._pathProgress.length = 0;

        for (let i = 0; i < this._pathDistances.length; i++) {
            const distance = this._pathDistances[i];
            let progress = distance / fullDistance;
            this._pathProgress.push(progress);
        }
    }

    /**
     * 将进度转换为 当前所在节点index 和 剩余路径的百分比数值
     */
    convertProgressToIndex(progress: number) {
        let curIndex = 0; //序号
        let leftProgress = 0;
        for (let i = 0; i < this._pathProgress.length; i++) {
            let pathProgress = this._pathProgress[i];
            let offsetProgress = 0;//单个路径的进度
            if (i == 0) {
                offsetProgress = pathProgress;
            } else {
                offsetProgress = this._pathProgress[i] - this._pathProgress[i - 1];
            }

            if (progress >= pathProgress) {
                curIndex = i + 1;
            } else {
                leftProgress = clamp01(1 - (pathProgress - progress) / offsetProgress);
                console.log("leftProgress计算:" + curIndex, "leftProgress:" + leftProgress);
                break;
            }
        }


        return { index: curIndex, pathProgress: leftProgress };

    }


    /**
     * 检查下一个目标
     */
    refreshIndexPath(curIndex: number) {
        if (this.curIndex == curIndex) return;

        //当前序号不同时，进行刷新，更新路径目标节点
        this.curIndex = curIndex;
        if (this.curIndex >= this.pathNodes.length) {
            if (this.loop) {
                //循环，到达终点继续循环跑
                this.curIndex = this.curIndex % this.pathNodes.length;
                let nextIndex = (this.curIndex + 1) % this.pathNodes.length;
                this.curTarget = this.pathNodes[this.curIndex];
                this.nextTarget = this.pathNodes[nextIndex];
            } else {
                //循环，到达终点中断移动目标
                this.curTarget = this.pathNodes[this.curIndex];
                this.nextTarget = null;
            }
        } else {
            this.curTarget = this.pathNodes[this.curIndex];
            this.nextTarget = this.pathNodes[this.curIndex + 1];
        }


    }

    complete() {
        //完成移动
        this.isComplete = true;
        alert("已经跑完一圈");
    }


    update(dt: number) {
        if (!this.isPlaying) return;
        if (this.isComplete) return;

        //移动距离 (以距离主导)
        this.distance += this.moveSpeed * dt;

        //移动进度
        if (this.loop) {
            this.progress = (this.distance / this.totalDistance) % 1;
        } else {
            this.progress = clamp01(this.distance / this.totalDistance);
        }


        //转换数据
        const result = this.convertProgressToIndex(this.progress);
        this.refreshIndexPath(result.index);


        //有移动目标的情况下，设置马的移动位置
        if (this.curTarget && this.nextTarget) {
            let pathProgress = result.pathProgress;
            let out = v3();
            let forward = v3();
            Vec3.lerp(out, this.curTarget.worldPosition, this.nextTarget.worldPosition, pathProgress);
            Vec3.lerp(forward, this.curTarget.forward, this.nextTarget.forward, pathProgress);
            this.node.forward = forward;
            this.node.worldPosition = out;
        } else {
            if (this.loop == false && this.progress >= 1) {
                this.complete();
            }
        }

    }
}

