// Learn TypeScript:
//  - https://docs.cocos.com/creator/2.4/manual/en/scripting/typescript.html
// Learn Attribute:
//  - https://docs.cocos.com/creator/2.4/manual/en/scripting/reference/attributes.html
// Learn life-cycle callbacks:
//  - https://docs.cocos.com/creator/2.4/manual/en/scripting/life-cycle-callbacks.html

const { ccclass, property } = cc._decorator;

// 物体震动时长
export const GRID_SHAKE_TIME = 0.32;
// 物体震动周期次数
export const GRID_SHAKE_COUNT = 1.5;
// 波浪强度
export const WAVE_POWER = 16;
// 波浪移动速度
export const WAVE_SPEED = 330;
// 波浪影响范围
export const WAVE_EFFECT_RANGE = 240;

export interface WaveInfo {
    pos: cc.Vec2;
    startTime: number;
}


@ccclass
export default class NewClass extends cc.Component {

    @property(cc.Sprite)
    sp: cc.Sprite = null;

    mat: cc.Material;
    onLoad(): void {
        this.mat = this.sp.getMaterial(0);
        this.mat.setProperty('GRID_SHAKE_TIME', GRID_SHAKE_TIME);
        this.mat.setProperty('GRID_SHAKE_COUNT', GRID_SHAKE_COUNT);
        this.mat.setProperty('WAVE_POWER', WAVE_POWER);
        this.mat.setProperty('WAVE_SPEED', WAVE_SPEED);
        this.mat.setProperty('WAVE_EFFECT_RANGE', WAVE_EFFECT_RANGE);
        this.mat.setProperty('textureSizeX', this.sp.node.width);
        this.mat.setProperty('textureSizeY', this.sp.node.height);

        this.node.on(cc.Node.EventType.TOUCH_START, this.onTouchStart, this);
        this.node.on(cc.Node.EventType.TOUCH_MOVE, this.onTouchStart, this);
    }

    waves: WaveInfo[] = [];
    lastInputTime: number = 0;
    lastWavePos: cc.Vec2 = cc.Vec2.ZERO;
    onTouchStart(event: cc.Event.EventTouch) {
        let now = Date.now();
        let location = this.node.parent.convertToNodeSpaceAR(event.getLocation());
        if (now - this.lastInputTime + cc.Vec2.distance(this.lastWavePos, location) > 100) {
            let info = {
                pos: location,
                startTime: now
            }
            // console.log('push,', location.x, location.y);
            this.waves.push(info);

            this.lastInputTime = now;
            this.lastWavePos = location;
        }
    }

    update(dt: number): void {
        let now = Date.now();
        let maxTime = WAVE_EFFECT_RANGE / WAVE_SPEED + GRID_SHAKE_TIME;
        while (this.waves.length > 0) {
            let wave = this.waves[0];
            if ((now - wave.startTime) / 1000 > maxTime) {
                this.waves.shift();
            } else {
                break;
            }
        }

        this.mat.setProperty('waveCnt', this.waves.length);

        if (this.waves.length > 0) {
            let spNode = this.sp.node;
            let testUV00 = cc.v2(spNode.x - spNode.width * spNode.anchorX, spNode.y + spNode.height * (1 - spNode.anchorY));

            let texture = new cc.Texture2D();
            let data = new Uint8Array(45 * 45 * 4).fill(0);
            // 波浪信息
            for (let i = 0; i < this.waves.length; i++) {
                const wave = this.waves[i];
                let waveX = wave.pos.x - testUV00.x;
                let waveY = testUV00.y - wave.pos.y;
                let time = (now - wave.startTime) / 1000;
                // console.log("WavePos:", wave.pos.x, wave.pos.y);
                // console.log("WaveUV", waveX / test.width, waveY / test.height);
                let r = waveX / spNode.width * 255;
                let g = waveY / spNode.height * 255;
                let b = time / maxTime * 255;;
                // if (time > maxTime) {
                //     b = 0;
                // }
                data[i * 4] = r;
                data[i * 4 + 1] = g;
                data[i * 4 + 2] = b;
            }
            texture.initWithData(data, cc.Texture2D.PixelFormat.RGBA8888, 45, 45);
            this.mat.setProperty('wavetexture', texture);
        }
    }

}
