/**
 * 翻译的 cocos2dx cc.TableView
 * 注意：
 *      1. cell 的锚点是 0.5, 0.5
 *      2. 只支持垂直滚动或水平滚动，不支持双向滚动
 * 函数：
 *      设置单元格间距的 setInterval(0, 0, 0)
 *      重新加载数据 reloadData(false)            添加参数(isUseLastOffset:是否使用上一次的容器偏移量)
 *      查找索引处的单元格 cellAtIndex(0)
 *      更新索引处的单元格 updateCellAtIndex(0)
 *      滚动到索引处的单元格 scrollToIndex(0)
 * 未实现：
 *      指定索引处插入新单元格 insertCellAtIndex
 *      删除指定索引处的单元格 removeCellAtIndex
 * 用法：
 *      import TableView, { TableViewCellNode } from "db://assets/script/extends/TableView";
 *      @ccclass('Helloworld')
 *      export class Helloworld extends Component {
 *          private tableData: number[] = [101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112];
 *          @property({type: TableView})
 *          private tableView: TableView = null!;
 *          protected start() {
 *              this.tableView.registerTableCellCount(this. onTableCellCount.bind(this));
 *              this.tableView.registerTableCellSize(this.onTableCellSize.bind(this));
 *              this.tableView.registerTableCellUpdate(this.onTableCellUpdate.bind(this));
 *              this.tableView.reloadData();
 *          }
 *          private onTableCellCount(): number {
 *              return this.tableData.length;
 *          }
 *          private onTableCellSize(idx: number): cc.Size {
 *              return cc.size(300, 100);// 通过idx可为每个cell设置单独size
 *          }
 *          private onTableCellUpdate(idx: number, cell: TableViewCellNode) {
 *              cc.log("idx = " + idx, "data = " + this.tableData[idx]);
 *          }
 *      }
 */
import { _decorator, CCFloat, Component, Enum, math, Node, ScrollView, Size, UITransform, Vec2 } from 'cc';

const { ccclass, property, requireComponent } = _decorator;

// 垂直填充顺序
export enum CellFillOrder {
    TopDown_LeftRight = 0,  // 垂直从上到下填充 水平从左到右填充
    BottomUp_RightLeft = 1, // 垂直从下往上填充 水平从右到左填充
}

export class TableViewCellNode extends Node {
    private _idx: number = -1;
    public static create(): TableViewCellNode {
        let cell = new TableViewCellNode("node_cell");
        cell.addComponent(UITransform);
        return cell;
    }
    public getIdx = (): number => this._idx;
    public setIdx = (idx: number) => this._idx = idx;
    public reset = () => this._idx = -1;
}

@ccclass('TableView')
@requireComponent(ScrollView)
export default class TableView extends Component {
    @property({displayName: "单元格尺寸相同"})
    private bCellSizeSame: boolean = true;

    @property({type: Enum(CellFillOrder), tooltip: "单元格垂直填充顺序"})
    private vordering: CellFillOrder = CellFillOrder.TopDown_LeftRight;

    @property({type: CCFloat, tooltip: "第一个单元格和边框的间隔"})
    private cellStartInterval: number = 0;
    @property({type: CCFloat, tooltip: "单元格之间的间隔"})
    private cellMidInterval: number = 0;
    @property({type: CCFloat, tooltip: "最后一个单元格和边框的间隔"})
    private cellEndInterval: number = 0;

    private _handlerCellCount: () => number = null!;                                    //单元格数量获取函数
    private _handlerCellSize: (idx: number) => Readonly<Size> = null!;                  //索引的单元格尺寸获取函数
    private _handlerCellUpdate: (idx: number, cell: TableViewCellNode) => void = null!; //索引的单元格更新函数
    private _handlerCellRecycle: (cell: TableViewCellNode) => void = null!;             //单元格回收函数

    private _indices: boolean[] = [];               //索引集用于查询所使用单元格的索引
    private _cellsPositions: number[] = [];         //所有单元格位置
    private _cellsUsed: TableViewCellNode[] = [];   //使用中的单元格
    private _cellsFreed: TableViewCellNode[] = [];  //未使用的单元格
    private _isUsedCellsDirty: boolean = false;     //脏标记对使用中的单元格进行排序

    private _scrollView!: ScrollView;
    public get scrollView(): ScrollView {
        if (!this._scrollView)
            this._scrollView = this.node.getComponent(ScrollView)!;
        return this._scrollView;
    }
    public get view(): UITransform {
        return this.scrollView.view!;
    }
    public get contentTransform(): UITransform {
        return this.scrollView.content!._uiProps.uiTransformComp!;
    }
    public get content(): Node {
        return this.scrollView.content!;
    }

    protected onEnable() {
        this.node.on('scrolling', this._scrollViewDidScroll, this);
        this.view.node.on(Node.EventType.SIZE_CHANGED, this._onMaskSizeChanged, this);
    }

    protected onDisable() {
        this.node.off('scrolling', this._scrollViewDidScroll, this);
        this.view.node.off(Node.EventType.SIZE_CHANGED, this._onMaskSizeChanged, this);
    }

    // 裁剪尺寸改变
    private _onMaskSizeChanged() {
        this._updateContentSize(true);
        this._scrollViewDidScroll();
    }

    // 注册单元格数量获取函数
    public registerTableCellCount(callback: () => number) {
        this._handlerCellCount = callback;
    }

    // 注册索引的单元格尺寸获取函数
    public registerTableCellSize(callback: (idx: number) => Readonly<Size>) {
        this._handlerCellSize = callback;
    }

    // 注册索引的单元格更新函数
    public registerTableCellUpdate(callback: (idx: number, cell: TableViewCellNode) => void) {
        this._handlerCellUpdate = callback;
    }

    // 注册单元格回收函数
    public registerCellRecycle(callback: (cell: TableViewCellNode) => void) {
        this._handlerCellRecycle = callback;
    }

    //设置垂直填充顺序
    public setCellFillOrder(fillOrder: CellFillOrder) {
        this.vordering = fillOrder;
    }

    //设置间隔
    public setInterval(cellMidInterval: number, cellStartInterval: number = 0, cellEndInterval: number = 0) {
        this.cellMidInterval = cellMidInterval;
        this.cellStartInterval = cellStartInterval;
        this.cellEndInterval = cellEndInterval;
    }

    //加载数据
    public reloadData(bUseLastOffset: boolean = false) {
        if (this.scrollView.horizontal === this.scrollView.vertical && this.scrollView.horizontal) {
            console.error("TableView only vertical or horizontal scrolling is supported");
            return;
        }
        for (const cell of this._cellsUsed) {
            if (this._handlerCellRecycle) {
                this._handlerCellRecycle(cell);
            }
            this._cellsFreed.push(cell);
            cell.reset();
            cell.active = false;
        }
        this._indices = [];
        this._cellsUsed = [];
        this._updateCellPositions();
        this._updateContentSize(bUseLastOffset);
        if (this._handlerCellCount() > 0)
            this._scrollViewDidScroll();
    }

    //查找索引处的单元格
    public cellAtIndex(idx: number) {
        if (this._indices[idx]) {
            for (let k in this._cellsUsed) {
                if (this._cellsUsed[k].getIdx() === idx)
                    return this._cellsUsed[k];
            }
        }
        return null;
    }

    //更新索引处的单元格
    public updateCellAtIndex(idx: number) {
        if (idx < 0)
            return;
        let countOfItems = this._handlerCellCount();
        if (0 === countOfItems || idx > countOfItems - 1)
            return;
        let cell = this.cellAtIndex(idx);
        if (cell)
            this._moveCellOutOfSight(cell);
        cell = this._createCellAtIndex(idx);
        this._setIndexForCell(idx, cell);
        this._addCellIfNecessary(cell);
    }

    //滚动到索引处的单元格
    public scrollToIndex(index: number, timeInSecond: number = 0.01) {
        let cellPos: number | undefined;
        if (this.bCellSizeSame) {
            let cellSize = this._handlerCellSize(0);
            cellPos = this.scrollView.horizontal ? cellSize.width : cellSize.height;
            cellPos = this.cellStartInterval + (cellPos + this.cellMidInterval) * index;
        } else {
            cellPos = this._cellsPositions[index];
        }
        if (cellPos == undefined)
            return;
        if (this.scrollView.horizontal) {
            if (this.vordering === CellFillOrder.TopDown_LeftRight)
                this.scrollView.scrollToOffset(new Vec2(cellPos, 0), Math.max(timeInSecond, 0.01));
            else
                this.scrollView.scrollToOffset(new Vec2(this.scrollView.getMaxScrollOffset().x - cellPos, 0), Math.max(timeInSecond, 0.01));
        } else {
            if (this.vordering === CellFillOrder.TopDown_LeftRight)
                this.scrollView.scrollToOffset(new Vec2(0, cellPos), Math.max(timeInSecond, 0.01));
            else
                this.scrollView.scrollToOffset(new Vec2(0, this.scrollView.getMaxScrollOffset().y - cellPos), Math.max(timeInSecond, 0.01));
        }
    }

    /** 下面是私有函数 */

    //更新单元格位置
    private _updateCellPositions() {
        if (this.bCellSizeSame)
            return;
        let cellsCount = this._handlerCellCount();
        this._cellsPositions = [];
        if (cellsCount > 0) {
            let currentPos = this.cellStartInterval;
            for (let i = 0; i < cellsCount; ++i) {
                this._cellsPositions[i] = currentPos;
                let size = this._handlerCellSize(i);
                if (this.scrollView.horizontal)
                    currentPos += size.width;
                else
                    currentPos += size.height;
                if (i < cellsCount - 1 && this.cellMidInterval != 0)
                    currentPos += this.cellMidInterval;
            }
            currentPos += this.cellEndInterval;
            this._cellsPositions[cellsCount] = currentPos;
        }
    }

    //更新内部容器尺寸
    private _updateContentSize(bUseLastOffset: boolean) {
        let size = new Size(this.view.width, this.view.height)
        let cellsCount = this._handlerCellCount();
        if (cellsCount > 0) {
            let maxPosition: number = 0;
            if (this.bCellSizeSame) {
                let cellSize = this._handlerCellSize(0);
                maxPosition = this.scrollView.horizontal ? cellSize.width : cellSize.height;
                maxPosition = this.cellStartInterval + (maxPosition + this.cellMidInterval) * cellsCount - this.cellMidInterval + this.cellEndInterval;
            } else {
                maxPosition = this._cellsPositions[cellsCount];
            }
            if (this.scrollView.horizontal && maxPosition > this.view.width)
                size.width = maxPosition;
            else if (this.scrollView.vertical && maxPosition > this.view.height)
                size.height = maxPosition;
        }
        if (size.width != this.contentTransform.width || size.height != this.contentTransform.height) {
            this.contentTransform.setContentSize(size);
        }
        if (!bUseLastOffset) {
            if (this.scrollView.horizontal)
                if (this.vordering === CellFillOrder.TopDown_LeftRight)
                    this.scrollView.scrollToLeft(0.01);
                else
                    this.scrollView.scrollToRight(0.01);
            else {
                if (this.vordering === CellFillOrder.TopDown_LeftRight)
                    this.scrollView.scrollToTop(0.01);
                else
                    this.scrollView.scrollToBottom(0.01);
            }
        } else {
            let offset = this.scrollView.getScrollOffset();
            offset.x *= -1;
            this.scrollView.scrollToOffset(offset, 0.01);
        }
    }

    //索引转坐标
    private _positionFromIndex(index: number): Vec2 {
        let size = this._handlerCellSize(index);
        let pos = new Vec2(0, 0);
        if (this.bCellSizeSame) {
            if (this.scrollView.horizontal)
                pos.x = this.cellStartInterval + (size.width + this.cellMidInterval) * index;
            else
                pos.y = this.cellStartInterval + (size.height + this.cellMidInterval) * index;
        } else {
            if (this.scrollView.horizontal)
                pos.x = this._cellsPositions[index];
            else
                pos.y = this._cellsPositions[index];
        }
        //转换坐标（cocos2dx的原点始终在左下角，而creator的原点是基于锚点来的）
        if (this.scrollView.vertical) {
            if (this.vordering === CellFillOrder.BottomUp_RightLeft)
                pos.y = pos.y + size.height / 2; //下往上
            else
                pos.y = this.contentTransform.height - pos.y - size.height / 2;
        } else {
            if (this.vordering === CellFillOrder.TopDown_LeftRight)
                pos.x = pos.x + size.width / 2; //左往右
            else
                pos.x = this.contentTransform.width - pos.x - size.width / 2;
        }
        if (this.scrollView.vertical) {
            pos.x = this.contentTransform.width * (0.5 - this.contentTransform.anchorX);
            pos.y = pos.y - this.contentTransform.height * this.contentTransform.anchorY;
        } else {
            pos.x = pos.x - this.contentTransform.width * this.contentTransform.anchorX;
            pos.y = this.contentTransform.height * (0.5 - this.contentTransform.anchorY);
        }
        return pos;
    }

    //偏移量转索引
    private _indexFromOffset(offset: Vec2): number {
        let low = 0;
        let high = this._handlerCellCount() - 1;
        let search = this.scrollView.horizontal ? offset.x : offset.y;
        if (this.bCellSizeSame) {
            let cellSize = this._handlerCellSize(0);
            low = this.scrollView.horizontal ? cellSize.width : cellSize.height;
            low = Math.floor((search - this.cellStartInterval) / (low + this.cellMidInterval));
            return math.clamp(low, 0, high);
        }
        while (low <= high) {
            let index = Math.floor(low + (high - low) / 2);
            let cellStart = this._cellsPositions[index];
            let cellEnd = this._cellsPositions[index + 1];
            if (search >= cellStart && search <= cellEnd)
                return index;
            else if (search < cellStart)
                high = index - 1;
            else
                low = index + 1;
        }
        return low <= 0 ? 0 : high;
    }

    //把单元格移除视线
    private _moveCellOutOfSight(cell: TableViewCellNode) {
        if (this._handlerCellRecycle) {
            this._handlerCellRecycle(cell);
        }
        this._cellsFreed.push(cell);
        let idx = this._cellsUsed.indexOf(cell);
        if (idx != -1)
            this._cellsUsed.splice(idx, 1);
        this._isUsedCellsDirty = true;
        delete this._indices[cell.getIdx()];
        cell.reset();
        cell.active = false;
    }

    //设置单元格索引
    private _setIndexForCell(index: number, cell: TableViewCellNode) {
        const pos = this._positionFromIndex(index);
        cell.setPosition(pos.x, pos.y);
        cell.setIdx(index);
    }

    //必要时添加单元格
    private _addCellIfNecessary(cell: TableViewCellNode) {
        if (!cell.parent)
            cell.parent = this.scrollView.content;
        this._cellsUsed.push(cell);
        this._isUsedCellsDirty = true;
        this._indices[cell.getIdx()] = true;
        cell.active = true;
    }

    //滚动时调用计算显示位置
    private _scrollViewDidScroll() {
        let countOfItems = this._handlerCellCount();
        if (0 === countOfItems)
            return;
        if (this._isUsedCellsDirty) {
            this._isUsedCellsDirty = false;
            this._cellsUsed.sort((a: TableViewCellNode, b: TableViewCellNode) => {
                return a.getIdx() - b.getIdx();
            })
        }

        //计算在显示范围的单元格的起始 startIdx 和结束 endIdx
        let startIdx = 0;
        let endIdx = 0;
        let maxIdx = Math.max(countOfItems - 1, 0);
        let offset = this.scrollView.getScrollOffset();
        offset.x *= -1;

        if (this.scrollView.vertical) {
            if (this.vordering === CellFillOrder.BottomUp_RightLeft)
                offset.y = this.contentTransform.height - offset.y - this.view.height; //下往上
        } else {
            if (this.vordering === CellFillOrder.BottomUp_RightLeft)
                offset.x = this.contentTransform.width - offset.x - this.view.width; //右往左
        }
        startIdx = this._indexFromOffset(offset.clone());

        offset.x += this.view.width;
        offset.y += this.view.height;

        endIdx = this._indexFromOffset(offset.clone());

        //移除不在 startIdx 和 endIdx 范围的 cell
        if (this._cellsUsed.length > 0) {
            let cell = this._cellsUsed[0];
            let idx = cell.getIdx();
            while (idx < startIdx) {
                this._moveCellOutOfSight(cell);
                if (this._cellsUsed.length > 0) {
                    cell = this._cellsUsed[0];
                    idx = cell.getIdx();
                } else
                    break;
            }
        }
        if (this._cellsUsed.length > 0) {
            let cell = this._cellsUsed[this._cellsUsed.length - 1];
            let idx = cell.getIdx();
            while (idx <= maxIdx && idx > endIdx) {
                this._moveCellOutOfSight(cell);
                if (this._cellsUsed.length > 0) {
                    cell = this._cellsUsed[this._cellsUsed.length - 1];
                    idx = cell.getIdx();
                } else
                    break
            }
        }

        //更新未在区域显示的cell
        for (let i = startIdx; i <= endIdx; ++i) {
            if (!this._indices[i])
                this.updateCellAtIndex(i);
        }
    }

    //创建索引处的单元格
    private _createCellAtIndex(idx: number): TableViewCellNode {
        let cell = this._cellsFreed.shift();
        if (!cell) {
            cell = TableViewCellNode.create();
        }
        cell.getComponent(UITransform)!.setContentSize(this._handlerCellSize(idx));
        this._handlerCellUpdate(idx, cell);
        return cell;
    }
}
