/*
 * @Author: your name
 * @Date: 2020-12-03 17:39:27
 * @LastEditTime: 2020-12-07 10:02:29
 * @LastEditors: Please set LastEditors
 * @Description: In User Settings Edit
 * @FilePath: /myScrollComp_3-0/assets/script/ScrollOption.ts
 */

import { CCFloat, Component, Enum, Layout, Node, Prefab, Rect, rect, ScrollView, Size, UITransform, v3, Vec3, _decorator } from 'cc';
const { ccclass, property, requireComponent } = _decorator;

/**
 * 移动到selectAera内所需时间
 */
const MOVE_TIME = 0.15;

/**
 * 偏移类型
 * NONE: 无偏移
 * EQUAL: 统一偏移
 * GRADUALLY: 逐级偏移
 */
enum OFFSET_TYPE {
    NONE = 1,
    EQUAL,
    GRADUALLY,
}
Enum(OFFSET_TYPE);

/**
 * 缩放类型
 * NONE: 无缩放
 * EQUAL: 统一缩放
 * GRADUALLY: 逐级缩放
 */
enum SCALE_TYPE {
    NONE = 1,
    EQUAL,
    GRADUALLY,
}
Enum(SCALE_TYPE);

@ccclass('ScrollOption')
@requireComponent(ScrollView)
export class ScrollOption extends Component {
    @property({
        type: Node,
        tooltip: '可滚动展示内容的节点，在该节点上需添加Layout组件，且瞄点都需要设为0.5',
    })
    public content: Node = null;

    @property({
        type: Prefab,
        tooltip: '滚动列表',
    })
    public scrollItem: Prefab = null;

    @property({
        type: OFFSET_TYPE,
        tooltip: '滚动列表偏移类型 NONE表示无偏移，EQUAL表示统一偏移，GRADUALLY表示逐级偏移',
    })
    public offsetType: OFFSET_TYPE = OFFSET_TYPE.NONE;

    @property({
        type: CCFloat,
        tooltip: '滚动列表偏移量，正数表示往上或者往右偏移，负数相反',
        visible: function() {
            return this.offsetType !== OFFSET_TYPE.NONE
        },
    })
    public offsetValue: number = 0.0;

    @property({
        type: SCALE_TYPE,
        tooltip: '滚动列表缩放类型 NONE表示无缩放，EQUAL表示统一缩放，GRADUALLY表示逐级缩放',
    })
    public scaleType: SCALE_TYPE = SCALE_TYPE.NONE;

    @property({
        type: CCFloat,
        range: [0, 1],
        tooltip: '滚动列表缩放量，范围为[0, 1]',
        visible: function() {
            return this.scaleType !== SCALE_TYPE.NONE
        },
    })
    public scaleValue: number = 0.0;


    private _inited: boolean = false;
    private _scrollView: ScrollView = null;                 // 该节点上的scrollView组件
    private _itemSize: Size = new Size(0, 0);               // 该节点上的scrollView组件
    private _midBoundingBox: Rect = rect(0, 0, 0, 0);       // 位于该节点中心位置的包围盒，大小应与scrollItem节点大小一致，貌似目前没啥用，先留着吧
    private _itemOffsetArr: number[] = [];                  // 记录每个item离scrollView左上角原点的偏移位置，如果是垂直排列则取y坐标，如果是横列则去x坐标
    private _isScrolling: boolean = false;
    private _delayCount: number = 0;

    /**
     * @description: 初始化该组件，请选择合适的时间点去触发，如果不手动调用，则默认在该组件onLoad阶段去初始化
     */    
    public init() {
        this._inited = true;

        // 获取该节点上的scrollView组件
        this._scrollView = this.getComponent(ScrollView);

        // 获取item节点的size
        this._itemSize = new Size(this.scrollItem.data.getComponent(UITransform).width, this.scrollItem.data.getComponent(UITransform).height);

        // 设置content节点上的layout属性，并且计算包围盒的大小及位置
        // paddingTop等参数为留白部分，大小为该节点的一半高减去item的一半高，如此可以使得滑到第一个item时正好在正中间显示
        const layout = this.content.getComponent(Layout);
        const uiTransform = this.getComponent(UITransform);
        if (this._scrollView.vertical) {
            layout.type = Layout.Type.VERTICAL;
            layout.paddingTop = (uiTransform.contentSize.height - this._itemSize.height) / 2;
            layout.paddingBottom = layout.paddingTop;
            this._midBoundingBox = rect(-uiTransform.contentSize.width / 2, -this._itemSize.height / 2, uiTransform.contentSize.width, this._itemSize.height);
        } else {
            layout.type = Layout.Type.HORIZONTAL;
            layout.paddingLeft = (uiTransform.contentSize.width - this._itemSize.width) / 2;
            layout.paddingRight = layout.paddingLeft;
            this._midBoundingBox = rect(-this._itemSize.width / 2, -uiTransform.height / 2, this._itemSize.width, uiTransform.height);
        }

        // 更新每个item离scrollView左上角原点的偏移位置，如果是垂直排列则取y坐标，如果是横列则去x坐标
        this.updateItemOffset();

        // 稍稍延迟一会，等待layout生效后更新视图
        this._delayCount++;
        this.scheduleOnce(() => {
            this._delayCount--;
            this.selectNearlyItem(true);
            this.updateScrollingView();
        }, 0.05);

        // 注册事件
        this.regEvents();
    }

    protected onLoad() {
        if (!this._inited) {
            this.init();
        }
    }

    protected onDestroy() {
        // 注销事件
        this.unregEvents();
    }

    private updateItemOffset() {
        this._itemOffsetArr = [];
        const layout = this.content.getComponent(Layout);
        const spacing = layout.type === Layout.Type.VERTICAL ? layout.spacingY : layout.spacingX;
        const size = this._scrollView.vertical ? this._itemSize.height : this._itemSize.width;
        let offset = 0;
        for (let i = 0, len = this.content.children.length; i < len; i++) {
            offset += (i === 0 ? 0 : (spacing + size));
            this._itemOffsetArr.push(offset);
        }

        // console.log('itemOffsetArr: ', this._itemOffsetArr);
    }

    private updateScrollingView() {
        for (let i = 0, len = this.content.children.length; i < len; i++) {
            const itemNode = this.content.children[i];
            const itemPos = itemNode.getPosition();
            let offset = this.getOffset(itemNode);
            let scale = this.getScale(itemNode);
            if (this._scrollView.vertical) {
                itemPos.x = offset;
                itemNode.setPosition(itemPos);
            } else {
                itemPos.y = offset;
                itemNode.setPosition(itemPos);
            }
            itemNode.setScale(v3(scale, scale, 1));
        }
    }

    private getOffset(itemNode: Node): number {
        const layout = this.content.getComponent(Layout);
        const viewPos = this.getPositionInView(itemNode);
        let mag = 0;        // item距离中心位置处的距离倍率，中间的为0，以此类推
        if (this._scrollView.vertical) {
            const spacing = layout.spacingY;
            const size = this._itemSize.height;
            mag = Math.abs(viewPos.y / (spacing + size));
        } else {
            const spacing = layout.spacingX;
            const size = this._itemSize.width;
            mag = Math.abs(viewPos.x / (spacing + size));
        }

        if (this.offsetType === OFFSET_TYPE.NONE) {
            return 0;
        } else if (this.offsetType === OFFSET_TYPE.EQUAL) {
            // 这里用绝对值是因为存在负数的情况，不用绝对值会引起大小判断错误
            let offset = Math.abs(mag * this.offsetValue) >= Math.abs(this.offsetValue) ? this.offsetValue : mag * this.offsetValue;
            return offset;
        } else if (this.offsetType === OFFSET_TYPE.GRADUALLY) {
            let offset = mag * this.offsetValue;
            return offset;
        }
    }

    private getPositionInView(item: Node): Vec3 {
        const itemPos = item.getPosition();
        const worldPos = item.parent.getComponent(UITransform).convertToWorldSpaceAR(itemPos);
        return this.node.getComponent(UITransform).convertToNodeSpaceAR(worldPos);
    }

    private getScale(itemNode: Node): number {
        const layout = this.content.getComponent(Layout);
        const viewPos = this.getPositionInView(itemNode);
        let mag = 0;        // item距离中心位置处的距离倍率，中间的为0，以此类推
        if (this._scrollView.vertical) {
            const spacing = layout.spacingY;
            const size = this._itemSize.height;
            mag = Math.abs(viewPos.y / (spacing + size));
        } else {
            const spacing = layout.spacingX;
            const size = this._itemSize.width;
            mag = Math.abs(viewPos.x / (spacing + size));
        }

        if (this.scaleType === SCALE_TYPE.NONE) {
            return 1;
        } else if (this.scaleType === SCALE_TYPE.EQUAL) {
            let scale = (1 - mag * this.scaleValue) <= (1 - this.scaleValue) ? (1 - this.scaleValue) : (1 - mag * this.scaleValue);
            return scale;
        } else if (this.scaleType === SCALE_TYPE.GRADUALLY) {
            let scale = 1 - mag * this.scaleValue <= 0.1 ? 0.1 : 1 - mag * this.scaleValue;
            return scale;
        }
    }

    protected regEvents() {
        this.node.on("scroll-began", this.scrollBegan, this);
        this.node.on("scrolling", this.updateScrollingView, this);
        this.node.on("scroll-ended", this.scrollEnded, this);
        this.content.on("child-added", this.childAdded, this);
        this.content.on("child-removed", this.childRemoved, this);
    }

    protected unregEvents() {
        this.node.off("scroll-began", this.scrollBegan, this);
        this.node.off("scrolling", this.updateScrollingView, this);
        this.node.off("scroll-ended", this.scrollEnded, this);
        this.content.off("child-added", this.childAdded, this);
        this.content.off("child-removed", this.childRemoved, this);
    }

    private scrollBegan() {
        if (this._isScrolling) {
            return;
        } 
        this._isScrolling = true;
    }

    private scrollEnded() {
        // 如果不是玩家操作的，那么就不能进入该回调，以免引起死循环
        if (!this._isScrolling) {
            return;
        }
        this._isScrolling = false;
        this.selectNearlyItem();
    }

    private selectNearlyItem(immediately: boolean = false) {
        const moveTime = immediately ? 0 : MOVE_TIME;
        if (this._scrollView.vertical) {
            const layout = this.content.getComponent(Layout);
            const spacing = layout.spacingY;
            const size = this._itemSize.height;
            const scrollOffsetY = Math.abs(this._scrollView.getScrollOffset().y);
            let offset = (spacing + size) / 2;
            for (let i = 0, len = this.content.children.length; i < len; i++) {
                if (scrollOffsetY < offset) {
                    let p = v3(0, this._itemOffsetArr[i], 0);
                    this._scrollView.scrollToOffset(p, moveTime);
                    return;
                }

                offset += (spacing + size);
            }
        } else {
            const layout = this.content.getComponent(Layout);
            const spacing = layout.spacingX;
            const size = this._itemSize.width;
            const scrollOffsetX = Math.abs(this._scrollView.getScrollOffset().x);
            let offset = (spacing + size) / 2;
            for (let i = 0, len = this.content.children.length; i < len; i++) {
                if (scrollOffsetX < offset) {
                    let p = v3(this._itemOffsetArr[i], 0, 0);
                    this._scrollView.scrollToOffset(p, moveTime);
                    return;
                }

                offset += (spacing + size);
            }
        }
    }

    private childAdded() {
        this._delayCount++;
        this.updateItemOffset();
        this.scheduleOnce(() => {
            this._delayCount--;
            this.selectNearlyItem(true);
            this.updateScrollingView();
        }, 0.05);
        
    }

    private childRemoved() {
        this._delayCount++;
        this.updateItemOffset();
        this.scheduleOnce(() => {
            this._delayCount--;
            this.selectNearlyItem(true);
            this.updateScrollingView();
        }, 0.05);
    }

    /**
     * @description: 移动到指定的item
     * @param {number} index
     */    
    public scrollTo(index: number) {
        if (index < 0 || index >= this.content.children.length) {
            return;
        }

        // 如果正好在动态增删节点，那么需要等待节点增删完毕，等待0.5s使得layout生效后再去移动
        if (this._delayCount > 0) {
            this.scheduleOnce(() => {
                if (this._scrollView.vertical) {
                    let p = v3(0, this._itemOffsetArr[index], 0);
                    this._scrollView.scrollToOffset(p, MOVE_TIME);
                } else {
                    let p = v3(this._itemOffsetArr[index], 0, 0);
                    this._scrollView.scrollToOffset(p, MOVE_TIME);
                }
            }, 0.05);
        } else {
            if (this._scrollView.vertical) {
                let p = v3(0, this._itemOffsetArr[index], 0);
                this._scrollView.scrollToOffset(p, MOVE_TIME);
            } else {
                let p = v3(this._itemOffsetArr[index], 0, 0);
                this._scrollView.scrollToOffset(p, MOVE_TIME);
            }
        }
    }
}
