/****************************************************************************
 * Copyright (c) 2018-2022 Xiamen Yaji Software Co., Ltd.
 *
 * http://www.cocos.com
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 ****************************************************************************/

package com.cocos.lib;

import android.app.Activity;
import android.content.Context;
import android.content.pm.ActivityInfo;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.res.AssetManager;
import android.content.res.Configuration;
import android.os.Build;
import android.os.Bundle;
import android.util.Log;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.Surface;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.FrameLayout;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;

import com.google.androidgamesdk.GameActivity;

import java.io.File;
import java.lang.ref.WeakReference;
import java.lang.reflect.Field;
import java.util.Arrays;

import dalvik.system.BaseDexClassLoader;

public class CocosActivityDelegate implements SurfaceHolder.Callback2 {
    private static final String _TAG = "CocosActivity";
    private WeakReference<Activity> mActivityRef;
    private FrameLayout mFrameLayout;
    private SurfaceView mSurfaceView;

    private CocosWebViewHelper mWebViewHelper = null;
    private CocosVideoHelper mVideoHelper = null;
    private CocosOrientationHelper mOrientationHelper = null;

    private CocosSensorHandler mSensorHandler;
    private long mNativeHandle;

    private SurfaceHolder mCurSurfaceHolder;

    private boolean mDestroyed;

    private static final String KEY_NATIVE_SAVED_STATE = "android:native_state";

    public static final String META_DATA_FUNC_NAME = "android.app.func_name";

    public static final String META_DATA_LIB_NAME = "android.app.lib_name";

    private native void onCreateNative(Context activity, String filePath);

    private native long loadNativeCode(String path, String funcname, String internalDataPath, String obbPath, String externalDataPath, AssetManager assetMgr, byte[] savedState);

    private native String getDlError();

    private native void unloadNativeCode(long handle);

    private native void onStartNative(long handle);

    private native void onResumeNative(long handle);

    private native byte[] onSaveInstanceStateNative(long handle);

    private native void onPauseNative(long handle);

    private native void onStopNative(long handle);

    private native void onConfigurationChangedNative(long handle);

    private native void onTrimMemoryNative(long handle, int level);

    private native void onWindowFocusChangedNative(long handle, boolean focused);

    private native void onSurfaceCreatedNative(long handle, Surface surface);

    private native void onSurfaceChangedNative(long handle, Surface surface, int format, int width, int height);

    private native void onSurfaceRedrawNeededNative(long handle, Surface surface);

    private native void onSurfaceDestroyedNative(long handle);

    private native boolean onTouchEventNative(long handle, MotionEvent motionEvent);

    private native boolean onKeyDownNative(long handle, KeyEvent keyEvent);

    private native boolean onKeyUpNative(long handle, KeyEvent keyEvent);

    private native void onWindowInsetsChangedNative(long handle);

    private String filePath;

    public CocosActivityDelegate(Activity activity,String filePath) {
        mActivityRef = new WeakReference<>(activity);
        this.filePath=filePath;
    }

    public Activity getActivity() {
        return mActivityRef.get();
    }

    public View getRootView() {
        return mFrameLayout;
    }

    protected void onSaveInstanceState(Bundle outState) {
        byte[] state = onSaveInstanceStateNative(mNativeHandle);
        if (state != null) {
            outState.putByteArray(KEY_NATIVE_SAVED_STATE, state);
        }
    }

    public boolean onTouchEvent(MotionEvent event) {
        return onTouchEventNative(mNativeHandle, event);
    }

    public boolean onKeyUp(final int keyCode, KeyEvent event) {
        return onKeyUpNative(mNativeHandle, event);
    }

    public boolean onKeyDown(final int keyCode, KeyEvent event) {
        return onKeyDownNative(mNativeHandle, event);
    }

    public void onConfigurationChanged(Configuration newConfig) {
        if (!mDestroyed) {
            onConfigurationChangedNative(mNativeHandle);
        }
    }

    public void onTrimMemory(int level) {
        if (!mDestroyed) {
            onTrimMemoryNative(mNativeHandle, level);
        }
    }

    public void onCreate(Bundle savedInstanceState) {
        onLoadNativeLibraries();
        onCreateNative(getActivity(), filePath);
//        getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
//        getWindow().requestFeature(Window.FEATURE_NO_TITLE);
//        setTheme(R.style.Theme_AppCompat_Light_NoActionBar);
//        super.onCreate(savedInstanceState);

        GlobalObject.setActivity(getActivity());
        CocosHelper.registerBatteryLevelReceiver(getActivity());
        CocosHelper.init(getActivity());
        CocosAudioFocusManager.registerAudioFocusListener(getActivity());
        CanvasRenderingContext2DImpl.init(getActivity());
//        this.setVolumeControlStream(AudioManager.STREAM_MUSIC);
//        getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_ADJUST_NOTHING);
        initView();

        String libname = "cocos";
        if (null != getActivity().getIntent().getStringExtra(META_DATA_LIB_NAME)) {
            libname = getActivity().getIntent().getStringExtra(META_DATA_LIB_NAME);
        }
        String funcname = "GameActivity_onCreate";
        ActivityInfo ai;
        try {
            ai = getActivity().getPackageManager().getActivityInfo(getActivity().getIntent().getComponent(), PackageManager.GET_META_DATA);
            if (ai.metaData != null) {
                String ln = ai.metaData.getString(META_DATA_LIB_NAME);
                if (ln != null) libname = ln;
                ln = ai.metaData.getString(META_DATA_FUNC_NAME);
                if (ln != null) funcname = ln;
            }
        } catch (PackageManager.NameNotFoundException e) {
            throw new RuntimeException("Error getting activity info", e);
        }

        BaseDexClassLoader classLoader = (BaseDexClassLoader) getActivity().getClassLoader();
        String path = classLoader.findLibrary(libname);

        if (path == null) {
            throw new IllegalArgumentException("Unable to find native library " + libname + " using classloader: " + classLoader.toString());
        }

        byte[] nativeSavedState = savedInstanceState != null ? savedInstanceState.getByteArray(KEY_NATIVE_SAVED_STATE) : null;
        Log.e("huangtao", "onCreate: path--->"+path+"  funcname--->"+funcname+"  nativeSavedState--->"+ Arrays.toString(nativeSavedState));
        mNativeHandle = loadNativeCode(path, funcname, getAbsolutePath(getActivity().getFilesDir()), getAbsolutePath(getActivity().getObbDir()), getAbsolutePath(getActivity().getExternalFilesDir(null)), getActivity().getAssets(), nativeSavedState);

        mSensorHandler = new CocosSensorHandler(getActivity());

//        setImmersiveMode();
//
//        Utils.hideVirtualButton();

        int orientation = getActivity().getRequestedOrientation();
        if (orientation == ActivityInfo.SCREEN_ORIENTATION_SENSOR || orientation == ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE || orientation == ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT) {
            mOrientationHelper = new CocosOrientationHelper(getActivity());
        }
    }

    private static String getAbsolutePath(File file) {
        return (file != null) ? file.getAbsolutePath() : null;
    }

    @RequiresApi(api = Build.VERSION_CODES.KITKAT)
    private void setImmersiveMode() {
        WindowManager.LayoutParams lp = getActivity().getWindow().getAttributes();
        try {
            Field field = lp.getClass().getField("layoutInDisplayCutoutMode");
            //Field constValue = lp.getClass().getDeclaredField("LAYOUT_IN_DISPLAY_CUTOUT_MODE_NEVER");
            Field constValue = lp.getClass().getDeclaredField("LAYOUT_IN_DISPLAY_CUTOUT_MODE_SHORT_EDGES");
            field.setInt(lp, constValue.getInt(null));

            // https://developer.android.com/training/system-ui/immersive
            int flag = View.SYSTEM_UI_FLAG_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_FULLSCREEN | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN | View.SYSTEM_UI_FLAG_LAYOUT_STABLE;

            flag |= View.class.getDeclaredField("SYSTEM_UI_FLAG_IMMERSIVE_STICKY").getInt(null);
            View view = getActivity().getWindow().getDecorView();
            view.setSystemUiVisibility(flag);

        } catch (NoSuchFieldException | IllegalAccessException e) {
            e.printStackTrace();
        }
    }

    protected void initView() {
//        FrameLayout frameLayout = findViewById(contentViewId);

        ViewGroup.LayoutParams frameLayoutParams = new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
        mFrameLayout = new FrameLayout(getActivity());
        mFrameLayout.setLayoutParams(frameLayoutParams);

        mSurfaceView = new SurfaceView(getActivity());
        mSurfaceView.getHolder().addCallback(this);
        mFrameLayout.addView(mSurfaceView);

        if (mWebViewHelper == null) {
            mWebViewHelper = new CocosWebViewHelper(mFrameLayout);
        }

        if (mVideoHelper == null) {
            mVideoHelper = new CocosVideoHelper(getActivity(), mFrameLayout);
        }
    }

    public SurfaceView getSurfaceView() {
        return this.mSurfaceView;
    }

    public void onDestroy() {
        mDestroyed = true;
        CocosHelper.unregisterBatteryLevelReceiver(getActivity());
        CocosAudioFocusManager.unregisterAudioFocusListener(getActivity());
        CanvasRenderingContext2DImpl.destroy();
        if (mCurSurfaceHolder != null) {
            onSurfaceDestroyedNative(mNativeHandle);
            mCurSurfaceHolder = null;
        }
        unloadNativeCode(mNativeHandle);
    }

    public void onPause() {
        mSensorHandler.onPause();
        if (null != mOrientationHelper) {
            mOrientationHelper.onPause();
        }
        onPauseNative(mNativeHandle);
    }

    public void onResume() {
        mSensorHandler.onResume();
        if (null != mOrientationHelper) {
            mOrientationHelper.onResume();
        }
        Utils.hideVirtualButton();
        if (CocosAudioFocusManager.isAudioFocusLoss()) {
            CocosAudioFocusManager.registerAudioFocusListener(getActivity());
        }
        onResumeNative(mNativeHandle);
    }

    public void onStop() {
        mSurfaceView.setVisibility(View.INVISIBLE);
        onStopNative(mNativeHandle);
    }

    public void onStart() {
        mSurfaceView.setVisibility(View.VISIBLE);
        onStartNative(mNativeHandle);
    }

    public void onWindowFocusChanged(boolean hasFocus) {
        if (hasFocus && CocosAudioFocusManager.isAudioFocusLoss()) {
            CocosAudioFocusManager.registerAudioFocusListener(getActivity());
        }
        onWindowFocusChangedNative(mNativeHandle, hasFocus);
    }

    private void onLoadNativeLibraries() {
        try {
            ApplicationInfo ai = getActivity().getPackageManager().getApplicationInfo(getActivity().getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = ai.metaData;
            String libName = bundle.getString("android.app.lib_name");
            System.loadLibrary(libName);
        } catch (Exception e) {
            e.printStackTrace();
            Log.e("huangtao", "onLoadNativeLibraries: load exception" );
        }
    }

    @Override
    public void surfaceCreated(@NonNull SurfaceHolder surfaceHolder) {
        if (!mDestroyed) {
            mCurSurfaceHolder = surfaceHolder;
            onSurfaceCreatedNative(mNativeHandle, surfaceHolder.getSurface());
        }
    }

    @Override
    public void surfaceChanged(@NonNull SurfaceHolder surfaceHolder, int i, int i1, int i2) {
        if (!mDestroyed) {
            mCurSurfaceHolder = surfaceHolder;
            onSurfaceChangedNative(mNativeHandle, surfaceHolder.getSurface(), i, i1, i2);
        }
    }

    @Override
    public void surfaceDestroyed(@NonNull SurfaceHolder surfaceHolder) {
        mCurSurfaceHolder = null;
        if (!mDestroyed) {
            onSurfaceDestroyedNative(mNativeHandle);
        }
    }

    @Override
    public void surfaceRedrawNeeded(SurfaceHolder holder) {
        if (!mDestroyed) {
            mCurSurfaceHolder = holder;
            onSurfaceRedrawNeededNative(mNativeHandle, holder.getSurface());
        }
    }
}
