import { _decorator, Component, Node, Mask, Vec3, Vec2, log, Mat4 } from 'cc';
const { ccclass, property } = _decorator;

const _circlePoints: Vec3[] = [];
function _calculateCircle (center: Vec3, radius: Vec3, segments: number) {
    _circlePoints.length = 0;
    const anglePerStep = Math.PI * 2 / segments;
    for (let step = 0; step < segments; ++step) {
        _circlePoints.push(new Vec3(radius.x * Math.cos(anglePerStep * step) + center.x,
            radius.y * Math.sin(anglePerStep * step) + center.y, 0));
    }

    return _circlePoints;
}

const _vec3_stencil = new Vec3();

const _worldMatrix = new Mat4();
const _vec2_temp = new Vec2();
const _mat4_temp = new Mat4();

@ccclass('MaskEx')
export class MaskEx extends Mask {
    private _stencilNode:Node | null = null;

    @property(Node)
    set stencilNode(value:Node){
        if( this._stencilNode == value ){
            return;
        }
        if( this._stencilNode ){
            this._stencilNode.off(Node.EventType.TRANSFORM_CHANGED,this._updateGraphics,this);
        }
        this._stencilNode = value;
        if( this._stencilNode ){
            this._stencilNode.on(Node.EventType.TRANSFORM_CHANGED,this._updateGraphics,this);
        }
        this._updateGraphics();
    }

    get stencilNode(){
        return this._stencilNode;
    }

    onDisable(){
        super.onDisable();
        if( this._stencilNode ){
            this._stencilNode.off(Node.EventType.TRANSFORM_CHANGED,this._updateGraphics,this);
        }
    }

    /**
     * setStenstencil:stencil
     */
    public setStencil(stencil:Node) {
        this.stencilNode = stencil;
    }

    /**
     * Hit test with point in World Space.
     *
     * @param worldPt point in World Space.
     */
     public isHit (worldPt: Vec2) {
        const node = this._stencilNode || this.node;
        const uiTrans = node._uiProps.uiTransformComp!;
        const size = uiTrans.contentSize;
        const w = size.width;
        const h = size.height;
        const testPt = _vec2_temp;

        node.getWorldMatrix(_worldMatrix);
        Mat4.invert(_mat4_temp, _worldMatrix);
        Vec2.transformMat4(testPt, worldPt, _mat4_temp);
        const ap = uiTrans.anchorPoint;
        testPt.x += ap.x * w;
        testPt.y += ap.y * h;

        let result = false;
        if (this.type === Mask.Type.GRAPHICS_RECT || this.type === Mask.Type.GRAPHICS_STENCIL || this.type === Mask.Type.SPRITE_STENCIL) {
            result = testPt.x >= 0 && testPt.y >= 0 && testPt.x <= w && testPt.y <= h;
        } else if (this.type === Mask.Type.GRAPHICS_ELLIPSE) {
            const rx = w / 2;
            const ry = h / 2;
            const px = testPt.x - 0.5 * w;
            const py = testPt.y - 0.5 * h;
            result = px * px / (rx * rx) + py * py / (ry * ry) < 1;
        }

        if (this._inverted) {
            result = !result;
        }

        return result;
    }

    protected _updateGraphics () {
        if (!this._graphics || (this._type !== Mask.Type.GRAPHICS_RECT && this._type !== Mask.Type.GRAPHICS_ELLIPSE)) {
            return;
        }

        log("_updateGraphics")
        const node = this._stencilNode || this.node;    //stencilNode为null，则使用当前节点
        const uiTrans = node._uiProps.uiTransformComp!;
        const graphics = this._graphics;
        // Share render data with graphics content
        graphics.clear();
        const size = uiTrans.contentSize;
        const width = size.width;
        const height = size.height;
        const ap = uiTrans.anchorPoint;
        _vec3_stencil.x = -width * ap.x;
        _vec3_stencil.y = -height * ap.y;

        uiTrans.convertToWorldSpaceAR(_vec3_stencil,_vec3_stencil);
        this.node._uiProps.uiTransformComp.convertToNodeSpaceAR(_vec3_stencil,_vec3_stencil);

        if (this._type === Mask.Type.GRAPHICS_RECT) {
            graphics.rect(_vec3_stencil.x, _vec3_stencil.y, width, height);
        } else if (this._type === Mask.Type.GRAPHICS_ELLIPSE) {
            const center = new Vec3(_vec3_stencil.x + width / 2, _vec3_stencil.y + height / 2, 0);
            const radius = new Vec3(width / 2, height / 2, 0);
            const points = _calculateCircle(center, radius, this._segments);
            for (let i = 0; i < points.length; ++i) {
                const point = points[i];
                if (i === 0) {
                    graphics.moveTo(point.x, point.y);
                } else {
                    graphics.lineTo(point.x, point.y);
                }
            }
            graphics.close();
        }

        graphics.fill();
    }
}

