
import { WebHttpRequest } from "./http/WebHttpRequest";
import { HttpResponseType } from "./http/HttpResponseType";
import { HttpMethod } from "./http/HttpMethod";
import App from "../App";
import AssetConst from "../../module/const/AssetConst";

const { ccclass, property } = cc._decorator;

/**
 * http通讯
 * @author chenkai 2020.7.7
 */
@ccclass
export default class Http {
    //单例
    private static _ins: Http;
    public static get ins(): Http {
        if (this._ins == null) {
            this._ins = new Http();
        }
        return this._ins;
    }


    /**Request*/
    public request: WebHttpRequest = null;
    /**当前发送内容*/
    private curSend = null;
    /**发送时锁屏UI */
    private lockUI: cc.Node;


    public constructor() {
        this.request = new WebHttpRequest();
        this.request.responseType = HttpResponseType.TEXT as any;
        this.request.on(WebHttpRequest.COMPLETE, this.onSendComplete, this);
        this.request.on(WebHttpRequest.IO_ERROR, this.onSendIOError, this);
    }

    /**
     * POST请求
     * @url 访问地址
     * @json 消息字符串 (json格式)
     * @cb 回调
     * @thisObj 回调执行对象
     * @bShowLoading 是否显示加载动画
     */
    public doPost(url: string, json: any, cb: Function, thisObj: any, bShowLoading: boolean = true) {
        this.send(url, json, cb, thisObj, HttpMethod.POST, bShowLoading);
    }

    /**
     * Get请求
     * @url 访问地址
     * @cb 回调
     * @thisObj 回调执行对象
     * @bShowLoading 是否显示加载动画
     */
    public doGet(url: string, cb: Function, thisObj: any, bShowLoading: boolean = true) {
        this.send(url, null, cb, thisObj, HttpMethod.GET, bShowLoading);
    }

    /**
     * Put请求
     * @url 访问地址
     * @json 传递数据
     * @cb 回调
     * @thisObj 回调执行对象
     * @bShowLoading 是否显示加载动画
     */
    public doPut(url: string, json: any, cb: Function, thisObj: any, bShowLoading: boolean = true) {
        this.send(url, json, cb, thisObj, "PUT", bShowLoading);
    }

    /**
     * 发送
     * @url 访问地址
     * @json 消息字符串 (json格式)
     * @callBack 回调
     * @thisObject 回调执行对象
     * @httpMethod 请求方式
     * @bShowLoading 是否显示加载动画
     */
    private send(url: string, json: any, cb: Function, thisObj: any, httpMethod: any, bShowLoading: boolean = false) {
        console.log("[Http] 请求Url:",url, " 方式:", httpMethod, " 数据:",json);
        //保存请求数据
        this.curSend = { json: json, cb: cb, obj: thisObj, url: url, httpMethod: httpMethod };
        //显示加载动画
        bShowLoading && this.showLoading();
        //发送数据
        this.request.open(url, httpMethod);
        //设置编码 JSON:application/json  application/x-www-form-urlencoded
        // this.request.setRequestHeader("Content-type","application/json");
        //根据POST和GET方式，选择是否发送msg数据
        if (httpMethod == HttpMethod.POST) {
            this.request.send(JSON.stringify(json));
        } else if (httpMethod == HttpMethod.GET) {
            this.request.send();
        } else {
            this.request.send(JSON.stringify(json));
        }
    }

    /**发送完成*/
    private onSendComplete(): void {
        console.log("[Http] 返回", JSON.parse(this.request.response));
        //隐藏遮罩
        this.hideLoading();

        //重置数据
        let cb = this.curSend.cb;
        let obj = this.curSend.obj;
        this.curSend = null;

        //执行回调
        if (cb && obj) {
            cb.call(obj, JSON.parse(this.request.response));
        }


    }

    /**发送失败*/
    private onSendIOError(res): void {
        console.log("[Http] 请求失败:", res);
        this.hideLoading();
    }

    /**发送失败时，继续发送失败的请求 */
    public resume() {
        //console.log("重新发送:", this.curSend);
        if (this.curSend != null) {
            this.send(this.curSend.url, this.curSend.json, this.curSend.cb, this.curSend.thisObj, this.curSend.httpMethod);
        }
    }

    /**删除请求*/
    public abort() {
        this.request.abort();
        this.curSend = null;
        this.hideLoading();
    }

    /**显示加载等待动画 */
    private showLoading() {
        //todo
    }

    /**隐藏加载等待动画 */
    private hideLoading() {
        //todo
    }

}
