#!/usr/bin/env python3
import argparse
import contextlib
import os
import re
import shutil
import subprocess
import sys
import tempfile
import zipfile
from pathlib import Path

RED = "\033[31m"
GREEN = "\033[32m"
ENDCOLOR = "\033[0m"

ALIGNMENT_REGEX = re.compile(r"2\*\*(1[4-9]|[2-9][0-9]|[1-9][0-9]{2,})")
ZIPALIGN_P_OPTION = r"-P <pagesize_kb>"


def run_command(cmd, *, check=False, capture_output=False, text=True, quiet=False):
    try:
        result = subprocess.run(
            cmd,
            check=check,
            capture_output=capture_output,
            text=text,
        )
        if capture_output and not quiet:
            if result.stdout:
                print(result.stdout, end="")
            if result.stderr:
                print(result.stderr, end="", file=sys.stderr)
        return result
    except FileNotFoundError:
        print(f"{RED}命令未找到：{' '.join(cmd)}{ENDCOLOR}", file=sys.stderr)
        sys.exit(1)


def ensure_path_exists(target: Path):
    if not target.exists():
        print(f"{RED}输入路径不存在：{target}{ENDCOLOR}", file=sys.stderr)
        sys.exit(1)


def has_zipalign():
    return shutil.which("zipalign") is not None


def zipalign_supports_pagesize():
    if not has_zipalign():
        return False
    result = run_command(["zipalign", "--help"], capture_output=True, quiet=True)
    help_text = (result.stdout or "") + (result.stderr or "")
    return ZIPALIGN_P_OPTION in help_text


def run_zipalign_check(apk_path: Path, indent=""):
    if not has_zipalign():
        print(f"{indent}NOTICE: 未找到 zipalign，可跳过 APK 对齐检查。")
        return
    if not zipalign_supports_pagesize():
        print(f"{indent}NOTICE: 需要 Android build-tools 35.0.0-rc3+ 才支持 16KB 校验。")
        print(f"{indent}       请通过 sdkmanager 更新：sdkmanager \"build-tools;35.0.0-rc3\"")
        return

    print(f"{indent}=== APK zip 对齐检查: {apk_path.name} ===")
    result = run_command(
        [
            "zipalign",
            "-v",
            "-c",
            "-P",
            "16",
            "4",
            str(apk_path),
        ],
        capture_output=True,
        quiet=True,
    )
    filtered_lines = []
    for line in (result.stdout or "").splitlines():
        if (
            "lib/arm64-v8a" in line
            or "lib/x86_64" in line
            or "Verification" in line
        ):
            filtered_lines.append(line)
    if filtered_lines:
        for line in filtered_lines:
            print(f"{indent}{line}")
    else:
        print(f"{indent}(zipalign 未输出匹配行，完整结果如下)")
        print(result.stdout or "", end="")
    if result.returncode != 0:
        print(result.stderr or "", end="")
        print(f"{indent}{RED}zipalign 返回非零状态，表示存在未对齐条目。{ENDCOLOR}")
    print(f"{indent}{'=' * 40}")


def extract_libs_from_apk(apk_path: Path, target_dir: Path):
    with zipfile.ZipFile(apk_path) as zf:
        members = [name for name in zf.namelist() if name.startswith("lib/")]
        if not members:
            # 若无 lib/ 目录，退化为完整解压，至少可分析其他 ELF。
            members = zf.namelist()
        for member in members:
            zf.extract(member, target_dir)


def unzip_archive(archive_path: Path, target_dir: Path):
    with zipfile.ZipFile(archive_path) as zf:
        zf.extractall(target_dir)


def build_universal_apks(aab_path: Path, tmp_dir: Path):
    if shutil.which("bundletool") is None:
        print("NOTICE: 未找到 bundletool，跳过派生 APK 的 zip 对齐检查。")
        return []

    derived_apks_path = tmp_dir / "derived.apks"
    cmd = [
        "bundletool",
        "build-apks",
        "--bundle",
        str(aab_path),
        "--output",
        str(derived_apks_path),
        "--mode",
        "universal",
    ]
    result = run_command(cmd, capture_output=True, quiet=True)
    if result.returncode != 0:
        print(
            "WARNING: bundletool build-apks 执行失败，"
            "跳过派生 APK 的 zip 对齐检查。"
        )
        print(result.stdout or "", end="")
        print(result.stderr or "", end="")
        return []

    derived_dir = tmp_dir / "derived_apks"
    derived_dir.mkdir(parents=True, exist_ok=True)
    apk_paths = []
    with zipfile.ZipFile(derived_apks_path) as zf:
        for member in zf.namelist():
            if member.endswith(".apk"):
                zf.extract(member, derived_dir)
                apk_paths.append(derived_dir / member)

    return apk_paths


def is_elf(path: Path):
    result = run_command(["file", str(path)], capture_output=True, quiet=True)
    if result.returncode != 0:
        return False
    return "ELF" in (result.stdout or "")


def read_alignment(path: Path):
    result = run_command(
        ["objdump", "-p", str(path)],
        capture_output=True,
        quiet=True,
    )
    if result.returncode != 0:
        return None

    for line in (result.stdout or "").splitlines():
        stripped = line.strip()
        if stripped.startswith("LOAD "):
            align = stripped.split()[-1]
            return align
    return None


def analyze_elves(root: Path):
    unaligned = []
    print("\n=== ELF 对齐检查 ===")

    for file_path in sorted(root.rglob("*")):
        if file_path.is_dir():
            continue

        suffix = file_path.suffix.lower()
        if suffix == ".apk":
            print(f"WARNING: 脚本不会递归处理嵌套 APK：{file_path}")
        elif suffix == ".apex":
            print(f"WARNING: 脚本不会递归处理嵌套 APEX：{file_path}")

        if not is_elf(file_path):
            continue

        alignment = read_alignment(file_path)
        if not alignment:
            print(f"{file_path}: {RED}无法解析对齐信息{ENDCOLOR}")
            unaligned.append(file_path)
            continue

        if ALIGNMENT_REGEX.fullmatch(alignment):
            print(f"{file_path}: {GREEN}ALIGNED{ENDCOLOR} ({alignment})")
        else:
            print(f"{file_path}: {RED}UNALIGNED{ENDCOLOR} ({alignment})")
            unaligned.append(file_path)

    if unaligned:
        print(
            f"{RED}共发现 {len(unaligned)} 个未对齐的库"
            "（仅 arm64-v8a / x86_64 必须满足 16KB 对齐）。"
            f"{ENDCOLOR}"
        )
    else:
        print(f"{GREEN}ELF Verification Successful{ENDCOLOR}")
    print("=====================")

    return unaligned


def ensure_command_availability():
    for cmd in ("file", "objdump"):
        if shutil.which(cmd) is None:
            print(f"{RED}缺少必要命令：{cmd}{ENDCOLOR}")
            sys.exit(1)


def handle_apex(apex_path: Path, stack: contextlib.ExitStack):
    tmp_dir = Path(
        stack.enter_context(
            tempfile.TemporaryDirectory(
                prefix=f"{apex_path.stem}_out_"
            )
        )
    )
    cmd = ["deapexer", "extract", str(apex_path), str(tmp_dir)]
    result = run_command(cmd, capture_output=True, quiet=True)
    if result.returncode != 0:
        print("Failed to deapex.", file=sys.stderr)
        sys.exit(1)
    print("\n递归分析 APEX:", apex_path)
    return tmp_dir


def handle_apk(apk_path: Path, stack: contextlib.ExitStack):
    tmp_dir = Path(
        stack.enter_context(
            tempfile.TemporaryDirectory(
                prefix=f"{apk_path.stem}_out_"
            )
        )
    )
    print("\n递归分析 APK:", apk_path)
    run_zipalign_check(apk_path)
    extract_libs_from_apk(apk_path, tmp_dir)
    return tmp_dir


def handle_aab(aab_path: Path, stack: contextlib.ExitStack):
    tmp_dir = Path(
        stack.enter_context(
            tempfile.TemporaryDirectory(
                prefix=f"{aab_path.stem}_out_"
            )
        )
    )
    bundle_contents = tmp_dir / "bundle_contents"
    bundle_contents.mkdir(parents=True, exist_ok=True)

    print("\n递归分析 AAB:", aab_path)
    try:
        unzip_archive(aab_path, bundle_contents)
    except zipfile.BadZipFile:
        print(f"{RED}无法解压 AAB：{aab_path}{ENDCOLOR}", file=sys.stderr)
        sys.exit(1)

    derived_apks = build_universal_apks(aab_path, tmp_dir)
    for apk in derived_apks:
        run_zipalign_check(apk, indent="  ")

    return bundle_contents


def parse_args():
    parser = argparse.ArgumentParser(
        description="检查 ELF 对齐 (16KB/64KB)。支持目录、APK、APEX、AAB。"
    )
    parser.add_argument(
        "input_path",
        help="待检测的 .so 所在路径、APK、APEX 或 AAB 文件",
    )
    return parser.parse_args()


def main():
    args = parse_args()
    target = Path(args.input_path).resolve()
    ensure_path_exists(target)
    ensure_command_availability()

    analysis_root = target
    with contextlib.ExitStack() as stack:
        suffix = target.suffix.lower()
        if target.is_file():
            if suffix == ".apk":
                analysis_root = handle_apk(target, stack)
            elif suffix == ".aab":
                analysis_root = handle_aab(target, stack)
            elif suffix == ".apex":
                analysis_root = handle_apex(target, stack)
        elif not target.is_dir():
            print(f"{RED}输入既非文件也非目录：{target}{ENDCOLOR}", file=sys.stderr)
            sys.exit(1)

        unaligned = analyze_elves(analysis_root)
        sys.exit(1 if unaligned else 0)


if __name__ == "__main__":
    main()