"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_extra_1 = require("fs-extra");
const path_1 = require("path");
// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore
const package_json_1 = __importDefault(require("../../../package.json"));
// 使用闭包变量保存处理器引用
let prefabSelectedHandler = null;
// 保存面板数据
const panelData = {
    prefabJson: null,
    nodes: [],
    selectedNodeId: null,
    filterType: '',
    sortType: 'none',
    showMainNodesOnly: false,
    deletedNodes: new Set(),
    originalFilePath: null,
};
/**
 * @zh 如果希望兼容 3.3 之前的版本可以使用下方的代码
 * @en You can add the code below if you want compatibility with versions prior to 3.3
 */
// Editor.Panel.define = Editor.Panel.define || function(options: any) { return options }
module.exports = Editor.Panel.define({
    listeners: {
        show() { console.log('show'); },
        hide() { console.log('hide'); },
    },
    template: (0, fs_extra_1.readFileSync)((0, path_1.join)(__dirname, '../../../static/template/default/index.html'), 'utf-8'),
    style: (0, fs_extra_1.readFileSync)((0, path_1.join)(__dirname, '../../../static/style/default/index.css'), 'utf-8'),
    $: {
        app: '#app',
        nodeList: '#node-list',
        nodeDetails: '#node-details',
        panelDivider: '#panel-divider',
        filterSelect: '#filter-select',
        sortButton: '#sort-button',
        typeHeader: '.node-type .header-text',
        mainNodesToggle: '#main-nodes-toggle',
        deleteSelectedButton: '#delete-selected-button',
        saveButton: '#save-button',
        saveAsButton: '#save-as-button',
    },
    methods: {
        hello() {
            if (this.$.app) {
                this.$.app.innerHTML = 'hello';
                console.log('[cocos-panel-html.default]: hello');
            }
        },
        /**
         * 解析 prefab 文件并显示 node 列表
         * @param {string} uuid - prefab 资源的 uuid
         */
        async parsePrefab(uuid) {
            var _a;
            try {
                // 获取资源信息
                const assetInfo = await Editor.Message.request('asset-db', 'query-asset-info', uuid);
                if (!(assetInfo === null || assetInfo === void 0 ? void 0 : assetInfo.file)) {
                    this.updateNodeList([]);
                    if (this.$.app) {
                        this.$.app.innerHTML = '无法读取 prefab 文件';
                    }
                    panelData.prefabJson = null;
                    panelData.nodes = [];
                    panelData.selectedNodeId = null;
                    panelData.deletedNodes.clear();
                    panelData.originalFilePath = null;
                    this.updateNodeDetails(null);
                    return;
                }
                // 读取 prefab 文件
                const prefabContent = (0, fs_extra_1.readFileSync)(assetInfo.file, 'utf-8');
                const prefabJson = JSON.parse(prefabContent);
                if (!Array.isArray(prefabJson)) {
                    this.updateNodeList([]);
                    if (this.$.app) {
                        this.$.app.innerHTML = 'prefab 文件格式错误';
                    }
                    panelData.prefabJson = null;
                    panelData.nodes = [];
                    panelData.selectedNodeId = null;
                    panelData.deletedNodes.clear();
                    panelData.originalFilePath = null;
                    this.updateNodeDetails(null);
                    return;
                }
                // 保存原始 prefab 数据
                panelData.prefabJson = prefabJson;
                panelData.selectedNodeId = null;
                panelData.filterType = '';
                panelData.sortType = 'none';
                panelData.deletedNodes.clear();
                panelData.originalFilePath = assetInfo.file;
                // 保持 showMainNodesOnly 的设置，不重置
                // 同步主要节点开关状态到数据（保持当前设置）
                if (this.$.mainNodesToggle) {
                    this.$.mainNodesToggle.checked = panelData.showMainNodesOnly;
                }
                // 解析 node 信息（处理所有类型的节点）
                const nodes = [];
                for (let i = 0; i < prefabJson.length; i++) {
                    const item = prefabJson[i];
                    const references = [];
                    const nodeInfo = {
                        id: i,
                        type: item.__type__ || '',
                        name: item._name || '',
                        parentId: null,
                        references: [],
                    };
                    // 字段名到标签类型的映射表（已知字段使用对应类型，未知字段使用 'other'）
                    // eslint-disable-next-line @typescript-eslint/naming-convention
                    const fieldToTypeMap = {
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _parent: 'parent',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _children: 'child',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _prefab: 'prefab',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        __prefab: 'prefab',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _components: 'component',
                        node: 'node',
                        data: 'data',
                        root: 'root',
                        asset: 'asset',
                        instance: 'instance',
                        prefabRootNode: 'prefabRootNode',
                        source: 'source',
                        sourceInfo: 'sourceInfo',
                        target: 'target',
                        propertyOverrides: 'propertyOverrides',
                        targetOverrides: 'targetOverrides',
                        mountedChildren: 'mountedChildren',
                        mountedComponents: 'mountedComponents',
                        removedComponents: 'removedComponents',
                        nestedPrefabInstanceRoots: 'nestedPrefabInstanceRoots',
                        clickEvents: 'clickEvents',
                        eventHandlers: 'clickEvents',
                        targetInfo: 'targetInfo',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _content: 'content',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _target: 'target',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _checkMark: 'checkMark',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _cameraComponent: 'cameraComponent',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _globals: 'globals',
                    };
                    // Helper 函数：添加引用
                    const addReference = (type, ref) => {
                        let id;
                        if (typeof ref === 'object' && ref !== null && ref.__id__ !== undefined) {
                            id = ref.__id__;
                        }
                        else if (typeof ref === 'number') {
                            id = ref;
                        }
                        if (id !== undefined) {
                            references.push({ type, id });
                        }
                    };
                    // Helper 函数：递归遍历对象的所有字段，查找包含 __id__ 的引用
                    // fieldPath: 字段路径，用于嵌套字段（如 "__editorExtras__.mountedRoot"）
                    // currentFieldName: 当前字段名，用于确定标签类型
                    const traverseFields = (obj, fieldPath = '', currentFieldName = '') => {
                        if (obj === null || obj === undefined || typeof obj !== 'object') {
                            return;
                        }
                        // 如果是数组，遍历每个元素
                        if (Array.isArray(obj)) {
                            for (const arrayItem of obj) {
                                traverseFields(arrayItem, fieldPath, currentFieldName);
                            }
                            return;
                        }
                        // 检查当前对象是否包含 __id__（这是引用节点）
                        if (obj.__id__ !== undefined) {
                            // 如果 currentFieldName 为空，说明这是嵌套对象中的引用，使用 'other' 类型
                            const tagType = currentFieldName ? (fieldToTypeMap[currentFieldName] || 'other') : 'other';
                            addReference(tagType, obj);
                            // 找到引用后，不再继续遍历其内部（避免重复）
                            return;
                        }
                        // 遍历对象的所有字段
                        for (const fieldName in obj) {
                            // 跳过系统字段
                            if (fieldName === '__type__' || fieldName === '__id__' || fieldName === '_name') {
                                continue;
                            }
                            const fieldValue = obj[fieldName];
                            // 跳过 null、undefined
                            if (fieldValue === null || fieldValue === undefined) {
                                continue;
                            }
                            // 构建字段路径和当前字段名
                            const newFieldPath = fieldPath ? `${fieldPath}.${fieldName}` : fieldName;
                            const newCurrentFieldName = fieldPath ? currentFieldName : fieldName;
                            // 如果字段值是对象或数组，递归遍历
                            if (typeof fieldValue === 'object') {
                                traverseFields(fieldValue, newFieldPath, newCurrentFieldName);
                            }
                        }
                    };
                    // 处理 _parent（特殊处理，需要设置 parentId）
                    if (item._parent) {
                        const parentType = fieldToTypeMap._parent || 'other';
                        addReference(parentType, item._parent);
                        if (typeof item._parent === 'object' && item._parent.__id__ !== undefined) {
                            nodeInfo.parentId = item._parent.__id__;
                        }
                    }
                    // 递归遍历所有字段（跳过 _parent，因为已经特殊处理）
                    for (const fieldName in item) {
                        // 跳过特殊字段
                        if (fieldName === '__type__' ||
                            fieldName === '__id__' ||
                            fieldName === '_name' ||
                            fieldName === '_parent' // 已特殊处理
                        ) {
                            continue;
                        }
                        const fieldValue = item[fieldName];
                        // 跳过 null、undefined
                        if (fieldValue === null || fieldValue === undefined) {
                            continue;
                        }
                        // 如果字段值是对象或数组，递归遍历
                        if (typeof fieldValue === 'object') {
                            traverseFields(fieldValue, '', fieldName);
                        }
                    }
                    nodeInfo.references = references;
                    nodes.push(nodeInfo);
                }
                panelData.nodes = nodes;
                // 更新显示
                const fileName = assetInfo.name || ((_a = assetInfo.url) === null || _a === void 0 ? void 0 : _a.split('/').pop()) || '未知';
                // 统计节点数量
                const totalCount = nodes.length;
                const sceneCount = nodes.filter(node => node.type === 'cc.Scene').length;
                const shouldHideCount = nodes.filter(node => node.name === 'should_hide_in_hierarchy').length;
                if (this.$.app) {
                    let statsText = `(共 ${totalCount} 个节点`;
                    if (sceneCount > 0) {
                        statsText += `，cc.Scene: ${sceneCount}`;
                    }
                    if (shouldHideCount > 0) {
                        statsText += `，should_hide_in_hierarchy: ${shouldHideCount}`;
                    }
                    statsText += ')';
                    this.$.app.innerHTML = `选中的 Prefab: <strong>${fileName}</strong> ${statsText}`;
                }
                this.updateNodeList(nodes);
                this.updateNodeDetails(null);
                this.updateDeleteButtonState();
                // 更新筛选下拉列表选项
                this.updateFilterSelect(nodes);
                // 重置筛选为"全部"
                panelData.filterType = '';
                if (this.$.filterSelect) {
                    this.$.filterSelect.value = '';
                }
                // 更新排序按钮文本
                this.updateSortButtonText();
                // 重新初始化列宽拖拽功能（因为 DOM 可能已经更新）
                this.initColumnResize();
                // 重新初始化 Type 列标题排序功能
                setTimeout(() => {
                    this.initTypeHeaderSort();
                }, 150);
            }
            catch (error) {
                console.error('[prefab-analysis] 解析 prefab 失败:', error);
                panelData.prefabJson = null;
                panelData.nodes = [];
                panelData.selectedNodeId = null;
                panelData.deletedNodes.clear();
                panelData.originalFilePath = null;
                this.updateNodeList([]);
                this.updateNodeDetails(null);
                if (this.$.app) {
                    this.$.app.innerHTML = `解析失败: ${error instanceof Error ? error.message : String(error)}`;
                }
            }
        },
        /**
         * 获取主要节点（cc.Prefab、cc.Node 以及没有被这些节点引用或间接引用的节点）
         * @param {NodeInfo[]} nodes - 所有节点数组
         * @returns {Set<number>} 主要节点的 ID 集合
         */
        getMainNodeIds(nodes) {
            // 第一步：找出所有 cc.Prefab 和 cc.Node 类型的节点
            const mainNodeIds = new Set();
            for (const node of nodes) {
                if (node.type === 'cc.Prefab' || node.type === 'cc.Node' || node.type === 'cc.PrefabInstance') {
                    mainNodeIds.add(node.id);
                }
            }
            // 第二步：找出所有被 cc.Prefab 和 cc.Node 直接或间接引用的节点
            const visitedIds = new Set();
            const toVisit = Array.from(mainNodeIds);
            while (toVisit.length > 0) {
                const currentId = toVisit.pop();
                if (!currentId || visitedIds.has(currentId)) {
                    continue;
                }
                visitedIds.add(currentId);
                const currentNode = nodes.find(n => n.id === currentId);
                if (currentNode) {
                    // 遍历当前节点的所有引用
                    for (const ref of currentNode.references) {
                        if (!visitedIds.has(ref.id)) {
                            toVisit.push(ref.id);
                        }
                    }
                }
            }
            // 第三步：找出所有没有被引用的节点（孤立节点）
            for (const node of nodes) {
                if (!visitedIds.has(node.id)) {
                    mainNodeIds.add(node.id);
                }
            }
            return mainNodeIds;
        },
        /**
         * 获取筛选和排序后的节点列表
         * @param {NodeInfo[]} nodes - 原始节点数组
         * @returns {NodeInfo[]} 筛选和排序后的节点数组
         */
        getFilteredAndSortedNodes(nodes) {
            var _a;
            let result = [...nodes];
            // 应用主要节点过滤
            if (panelData.showMainNodesOnly) {
                const mainNodeIds = this.getMainNodeIds(nodes);
                result = result.filter((node) => {
                    return mainNodeIds.has(node.id);
                });
            }
            // 应用类型筛选（精确匹配）
            if ((_a = panelData.filterType) === null || _a === void 0 ? void 0 : _a.trim()) {
                result = result.filter((node) => {
                    return node.type === panelData.filterType;
                });
            }
            // 应用排序
            if (panelData.sortType !== 'none') {
                result.sort((a, b) => {
                    const typeA = (a.type || '').toLowerCase();
                    const typeB = (b.type || '').toLowerCase();
                    if (panelData.sortType === 'asc') {
                        return typeA.localeCompare(typeB);
                    }
                    else {
                        return typeB.localeCompare(typeA);
                    }
                });
            }
            return result;
        },
        /**
         * 更新 node 列表显示
         * @param {NodeInfo[]} nodes - node 信息数组（如果传入，则更新原始节点数据；否则使用已保存的节点数据）
         */
        updateNodeList(nodes) {
            if (!this.$.nodeList) {
                return;
            }
            // 如果传入了 nodes，更新原始节点数据；否则使用已保存的节点数据
            if (nodes) {
                panelData.nodes = nodes;
            }
            // 如果没有节点数据，显示空状态
            if (!panelData.nodes || panelData.nodes.length === 0) {
                this.$.nodeList.innerHTML = '<div class="node-item">暂无节点</div>';
                return;
            }
            // 获取筛选和排序后的节点
            const filteredNodes = this.getFilteredAndSortedNodes(panelData.nodes);
            if (filteredNodes.length === 0) {
                this.$.nodeList.innerHTML = '<div class="node-item">暂无节点（已筛选）</div>';
                return;
            }
            let html = '';
            for (const node of filteredNodes) {
                // 如果字段为空，显示占位符
                const idText = node.id.toString();
                const typeText = node.type || '-';
                const nameText = node.name || '-';
                const selectedClass = node.id === panelData.selectedNodeId ? 'selected' : '';
                const deletedClass = panelData.deletedNodes.has(node.id) ? 'deleted' : '';
                // 检查 tag 中是否有被标记删除的节点
                let hasDeletedRef = false;
                if (node.references && node.references.length > 0) {
                    for (const ref of node.references) {
                        if (panelData.deletedNodes.has(ref.id)) {
                            hasDeletedRef = true;
                            break;
                        }
                    }
                }
                const hasDeletedRefClass = hasDeletedRef ? 'has-deleted-ref' : '';
                // 生成 tags HTML
                let tagsHtml = '';
                if (node.references && node.references.length > 0) {
                    for (const ref of node.references) {
                        const tagClass = `tag-${ref.type}`;
                        const refDeletedClass = panelData.deletedNodes.has(ref.id) ? 'ref-deleted' : '';
                        tagsHtml += `<span class="node-tag ${tagClass} ${refDeletedClass}" data-ref-id="${ref.id}" data-ref-type="${ref.type}">${ref.id}</span>`;
                    }
                }
                else {
                    tagsHtml = '-';
                }
                html += `
                    <div class="node-item ${selectedClass}" data-node-id="${node.id}">
                        <span class="node-id ${deletedClass} ${hasDeletedRefClass}">${idText}</span>
                        <span class="node-type">${typeText}</span>
                        <span class="node-name">${nameText}</span>
                        <span class="node-tags">${tagsHtml}</span>
                    </div>
                `;
            }
            this.$.nodeList.innerHTML = html;
            // 添加点击事件监听器
            const nodeItems = this.$.nodeList.querySelectorAll('.node-item');
            nodeItems.forEach((item) => {
                item.addEventListener('click', (e) => {
                    const target = e.currentTarget;
                    const clickedTag = e.target.closest('.node-tag');
                    // 如果点击的是 tag，处理 tag 点击
                    if (clickedTag) {
                        e.stopPropagation();
                        const refId = parseInt(clickedTag.getAttribute('data-ref-id') || '-1', 10);
                        if (refId >= 0) {
                            this.selectNode(refId);
                        }
                        return;
                    }
                    // 否则处理节点点击
                    const nodeId = parseInt(target.getAttribute('data-node-id') || '-1', 10);
                    if (nodeId >= 0) {
                        this.selectNode(nodeId);
                    }
                });
            });
        },
        /**
         * 选中节点
         * @param {number} nodeId - 节点 ID
         */
        selectNode(nodeId) {
            panelData.selectedNodeId = nodeId;
            this.updateNodeList(panelData.nodes);
            this.updateNodeDetails(nodeId);
        },
        /**
         * 更新节点详细信息显示
         * @param {number | null} nodeId - 节点 ID，null 表示清空
         */
        updateNodeDetails(nodeId) {
            if (!this.$.nodeDetails) {
                return;
            }
            if (nodeId === null || !panelData.prefabJson || nodeId < 0 || nodeId >= panelData.prefabJson.length) {
                this.$.nodeDetails.innerHTML = '<div class="details-placeholder">点击左侧节点查看详细信息</div>';
                return;
            }
            const nodeData = panelData.prefabJson[nodeId];
            const nodeInfo = panelData.nodes[nodeId];
            if (!nodeData) {
                this.$.nodeDetails.innerHTML = '<div class="details-placeholder">节点数据不存在</div>';
                return;
            }
            // 构建详细信息 HTML
            let html = '<div class="details-content">';
            // 基本信息部分
            html += '<div class="details-section">';
            html += '<div class="details-section-title">基本信息</div>';
            if (nodeInfo.id !== undefined) {
                html += `<div class="details-property">
                    <span class="details-property-label">ID:</span>
                    <span class="details-property-value">${nodeInfo.id}</span>
                </div>`;
            }
            if (nodeInfo.type) {
                html += `<div class="details-property">
                    <span class="details-property-label">Type:</span>
                    <span class="details-property-value">${nodeInfo.type}</span>
                </div>`;
            }
            if (nodeInfo.name) {
                html += `<div class="details-property">
                    <span class="details-property-label">Name:</span>
                    <span class="details-property-value">${nodeInfo.name}</span>
                </div>`;
            }
            // 显示 Tags（按类型分组）
            if (nodeInfo.references && nodeInfo.references.length > 0) {
                // 按类型分组
                const tagsByType = {};
                for (const ref of nodeInfo.references) {
                    if (!tagsByType[ref.type]) {
                        tagsByType[ref.type] = [];
                    }
                    tagsByType[ref.type].push(ref);
                }
                // 定义类型显示顺序和中文名称
                const typeOrder = [
                    'parent', 'child', 'prefab', 'component', 'node',
                    'root', 'asset', 'instance', 'prefabRootNode',
                    'source', 'sourceInfo', 'target', 'propertyOverrides',
                    'targetOverrides', 'mountedChildren', 'mountedComponents',
                    'removedComponents', 'nestedPrefabInstanceRoots', 'clickEvents',
                    'targetInfo', 'data', 'content', 'checkMark', 'cameraComponent',
                    'globals', 'other',
                ];
                const typeNames = {
                    parent: 'Parent',
                    child: 'Child',
                    prefab: 'Prefab',
                    component: 'Component',
                    node: 'Node',
                    root: 'Root',
                    asset: 'Asset',
                    instance: 'Instance',
                    prefabRootNode: 'PrefabRootNode',
                    source: 'Source',
                    sourceInfo: 'SourceInfo',
                    target: 'Target',
                    propertyOverrides: 'PropertyOverrides',
                    targetOverrides: 'TargetOverrides',
                    mountedChildren: 'MountedChildren',
                    mountedComponents: 'MountedComponents',
                    removedComponents: 'RemovedComponents',
                    nestedPrefabInstanceRoots: 'NestedPrefabInstanceRoots',
                    clickEvents: 'ClickEvents',
                    targetInfo: 'TargetInfo',
                    data: 'Data',
                    content: 'Content',
                    checkMark: 'CheckMark',
                    cameraComponent: 'CameraComponent',
                    globals: 'Globals',
                    other: 'Other',
                };
                let tagsHtml = '<div class="details-tags-grouped">';
                for (const type of typeOrder) {
                    if (tagsByType[type] && tagsByType[type].length > 0) {
                        const typeName = typeNames[type] || type;
                        tagsHtml += `<div class="details-tag-group">`;
                        tagsHtml += `<div class="details-tag-group-label">${typeName}:</div>`;
                        tagsHtml += `<div class="details-tag-group-items">`;
                        for (const ref of tagsByType[type]) {
                            const tagClass = `tag-${ref.type}`;
                            const refDeletedClass = panelData.deletedNodes.has(ref.id) ? 'ref-deleted' : '';
                            tagsHtml += `<span class="node-tag ${tagClass} ${refDeletedClass}" data-ref-id="${ref.id}" data-ref-type="${ref.type}">${ref.id}</span>`;
                        }
                        tagsHtml += `</div></div>`;
                    }
                }
                tagsHtml += '</div>';
                html += `<div class="details-property">
                    <span class="details-property-label">Tags:</span>
                    <span class="details-property-value">${tagsHtml}</span>
                </div>`;
            }
            else {
                html += `<div class="details-property">
                    <span class="details-property-label">Tags:</span>
                    <span class="details-property-value">-</span>
                </div>`;
            }
            html += '</div>';
            // 完整 JSON 数据部分
            html += '<div class="details-section">';
            html += '<div class="details-section-title">完整数据</div>';
            html += `<div class="details-json">${JSON.stringify(nodeData, null, 2)}</div>`;
            html += '</div>';
            // 标记删除/清除标记按钮
            const isDeleted = panelData.deletedNodes.has(nodeId);
            const deleteButtonText = isDeleted ? '清除标记（递归）' : '标记删除（递归）';
            const deleteSingleButtonText = isDeleted ? '清除标记（单节点）' : '标记删除（单节点）';
            html += '<div class="details-section">';
            html += `<button id="delete-mark-recursive-button" class="delete-mark-btn">${deleteButtonText}</button>`;
            html += `<button id="delete-mark-single-button" class="delete-mark-btn">${deleteSingleButtonText}</button>`;
            html += '</div>';
            html += '</div>';
            this.$.nodeDetails.innerHTML = html;
            // 为详细信息中的 tags 添加点击事件
            const detailTags = this.$.nodeDetails.querySelectorAll('.node-tag');
            detailTags.forEach((tag) => {
                tag.addEventListener('click', (e) => {
                    e.stopPropagation();
                    const clickedTag = e.currentTarget;
                    const refId = parseInt(clickedTag.getAttribute('data-ref-id') || '-1', 10);
                    if (refId >= 0) {
                        this.selectNode(refId);
                    }
                });
            });
            // 为标记删除（递归）按钮添加点击事件
            const deleteMarkRecursiveButton = this.$.nodeDetails.querySelector('#delete-mark-recursive-button');
            if (deleteMarkRecursiveButton) {
                deleteMarkRecursiveButton.addEventListener('click', () => {
                    this.toggleDeleteMark(nodeId);
                });
            }
            // 为标记删除（单节点）按钮添加点击事件
            const deleteMarkSingleButton = this.$.nodeDetails.querySelector('#delete-mark-single-button');
            if (deleteMarkSingleButton) {
                deleteMarkSingleButton.addEventListener('click', () => {
                    this.toggleDeleteMarkSingle(nodeId);
                });
            }
        },
        /**
         * 递归标记节点及其引用的所有节点为删除
         * @param {number} nodeId - 节点 ID
         */
        markNodeAndRefsAsDeleted(nodeId) {
            // 如果节点已经被标记删除，跳过（避免无限递归）
            if (panelData.deletedNodes.has(nodeId)) {
                return;
            }
            // 标记当前节点为删除
            panelData.deletedNodes.add(nodeId);
            // 获取当前节点的信息
            const nodeInfo = panelData.nodes[nodeId];
            if (!(nodeInfo === null || nodeInfo === void 0 ? void 0 : nodeInfo.references)) {
                return;
            }
            // 递归标记所有被引用的节点
            for (const ref of nodeInfo.references) {
                if (ref.id >= 0 && ref.id < panelData.nodes.length) {
                    this.markNodeAndRefsAsDeleted(ref.id);
                }
            }
        },
        /**
         * 递归清除节点及其引用的所有节点的删除标记
         * @param {number} nodeId - 节点 ID
         */
        unmarkNodeAndRefsAsDeleted(nodeId) {
            // 如果节点没有被标记删除，跳过（避免无限递归）
            if (!panelData.deletedNodes.has(nodeId)) {
                return;
            }
            // 清除当前节点的删除标记
            panelData.deletedNodes.delete(nodeId);
            // 获取当前节点的信息
            const nodeInfo = panelData.nodes[nodeId];
            if (!(nodeInfo === null || nodeInfo === void 0 ? void 0 : nodeInfo.references)) {
                return;
            }
            // 递归清除所有被引用节点的删除标记
            for (const ref of nodeInfo.references) {
                if (ref.id >= 0 && ref.id < panelData.nodes.length) {
                    this.unmarkNodeAndRefsAsDeleted(ref.id);
                }
            }
        },
        /**
         * 切换节点的删除标记（递归标记/清除引用的节点）
         * @param {number} nodeId - 节点 ID
         */
        toggleDeleteMark(nodeId) {
            if (panelData.deletedNodes.has(nodeId)) {
                // 清除标记（递归清除所有被引用节点的删除标记）
                this.unmarkNodeAndRefsAsDeleted(nodeId);
            }
            else {
                // 标记删除（递归标记所有被引用的节点）
                this.markNodeAndRefsAsDeleted(nodeId);
            }
            // 更新列表和详情显示
            this.updateNodeList(panelData.nodes);
            this.updateNodeDetails(nodeId);
            // 更新删除按钮状态
            this.updateDeleteButtonState();
        },
        /**
         * 切换节点的删除标记（只标记当前节点，不递归）
         * @param {number} nodeId - 节点 ID
         */
        toggleDeleteMarkSingle(nodeId) {
            if (panelData.deletedNodes.has(nodeId)) {
                // 清除标记
                panelData.deletedNodes.delete(nodeId);
            }
            else {
                // 只标记当前节点
                panelData.deletedNodes.add(nodeId);
            }
            // 更新列表和详情显示
            this.updateNodeList(panelData.nodes);
            this.updateNodeDetails(nodeId);
            // 更新删除按钮状态
            this.updateDeleteButtonState();
        },
        /**
         * 更新删除按钮的状态和文本
         */
        updateDeleteButtonState() {
            if (!this.$.deleteSelectedButton) {
                return;
            }
            const button = this.$.deleteSelectedButton;
            const deletedCount = panelData.deletedNodes.size;
            if (deletedCount > 0) {
                button.disabled = false;
                button.textContent = `删除选中节点(${deletedCount})`;
            }
            else {
                button.disabled = true;
                button.textContent = '删除选中节点';
            }
        },
        /**
         * 删除所有标记为删除的节点
         */
        deleteSelectedNodes() {
            if (!panelData.prefabJson || !panelData.nodes || panelData.deletedNodes.size === 0) {
                return;
            }
            // 将被删除的节点ID转换为数组并排序（从大到小）
            const deletedIds = Array.from(panelData.deletedNodes).sort((a, b) => b - a);
            // 创建索引映射表：旧索引 -> 新索引（只包含保留的节点）
            const indexMap = [];
            let newIndex = 0;
            for (let oldIndex = 0; oldIndex < panelData.nodes.length; oldIndex++) {
                if (!panelData.deletedNodes.has(oldIndex)) {
                    indexMap[oldIndex] = newIndex;
                    newIndex++;
                }
            }
            // 从大到小删除节点（避免索引变化影响删除）
            for (const deletedId of deletedIds) {
                panelData.prefabJson.splice(deletedId, 1);
                panelData.nodes.splice(deletedId, 1);
            }
            // 创建要删除的节点ID的Set，用于快速查找
            const deletedIdsSet = new Set(deletedIds);
            // 递归函数：更新对象中的所有 __id__ 引用
            const updateReferences = (obj) => {
                if (obj === null || obj === undefined) {
                    return;
                }
                if (typeof obj !== 'object') {
                    return;
                }
                // 如果是数组，遍历每个元素
                if (Array.isArray(obj)) {
                    for (let i = obj.length - 1; i >= 0; i--) {
                        const item = obj[i];
                        // 如果是包含 __id__ 的对象，检查是否需要删除或更新
                        if (item && typeof item === 'object' && item.__id__ !== undefined) {
                            const oldId = item.__id__;
                            if (deletedIdsSet.has(oldId)) {
                                // 引用已被删除，从数组中移除
                                obj.splice(i, 1);
                            }
                            else if (indexMap[oldId] !== undefined) {
                                // 更新引用ID
                                item.__id__ = indexMap[oldId];
                                // 递归处理该对象的其他字段
                                updateReferences(item);
                            }
                            else {
                                // 递归处理该对象的其他字段
                                updateReferences(item);
                            }
                        }
                        else {
                            // 递归处理数组元素
                            updateReferences(item);
                        }
                    }
                }
                else {
                    // 如果是对象，遍历所有字段
                    for (const key in obj) {
                        if (key === '__id__') {
                            // 跳过 __id__ 字段本身，因为节点的 __id__ 不需要更新（它们对应数组索引）
                            continue;
                        }
                        const value = obj[key];
                        // 如果值是包含 __id__ 的对象，更新或删除
                        if (value && typeof value === 'object' && value.__id__ !== undefined) {
                            const oldId = value.__id__;
                            if (deletedIdsSet.has(oldId)) {
                                // 引用已被删除，设置为 null
                                obj[key] = null;
                            }
                            else if (indexMap[oldId] !== undefined) {
                                // 更新引用ID
                                value.__id__ = indexMap[oldId];
                                // 递归处理该对象
                                updateReferences(value);
                            }
                            else {
                                // 递归处理该对象
                                updateReferences(value);
                            }
                        }
                        else {
                            // 递归处理其他类型的值
                            updateReferences(value);
                        }
                    }
                }
            };
            // 更新 prefabJson 中所有节点的引用
            for (let i = 0; i < panelData.prefabJson.length; i++) {
                // // 更新节点本身的 __id__ 为数组索引
                // if (panelData.prefabJson[i] && typeof panelData.prefabJson[i] === 'object') {
                //     panelData.prefabJson[i].__id__ = i;
                // }
                // 更新节点内部的所有引用
                updateReferences(panelData.prefabJson[i]);
            }
            // 更新所有节点的引用（将旧的节点ID映射到新的节点ID）
            for (const node of panelData.nodes) {
                if (node.references) {
                    for (const ref of node.references) {
                        const oldRefId = ref.id;
                        if (deletedIdsSet.has(oldRefId)) {
                            // 引用已被删除，标记为需要移除
                            ref.id = -1;
                        }
                        else if (indexMap[oldRefId] !== undefined) {
                            // 更新引用ID为新的索引
                            ref.id = indexMap[oldRefId];
                        }
                    }
                    // 移除标记为-1的引用（已被删除的节点）
                    node.references = node.references.filter(ref => ref.id >= 0);
                }
            }
            // 更新节点的ID（因为数组索引就是节点ID）
            for (let i = 0; i < panelData.nodes.length; i++) {
                panelData.nodes[i].id = i;
            }
            // 清空删除标记
            panelData.deletedNodes.clear();
            // 更新选中节点ID
            if (panelData.selectedNodeId !== null) {
                if (deletedIds.includes(panelData.selectedNodeId)) {
                    // 当前选中的节点被删除，清空选中状态
                    panelData.selectedNodeId = null;
                }
                else if (indexMap[panelData.selectedNodeId] !== undefined) {
                    // 更新选中节点的ID
                    panelData.selectedNodeId = indexMap[panelData.selectedNodeId];
                }
            }
            // 更新显示
            this.updateNodeList(panelData.nodes);
            this.updateNodeDetails(panelData.selectedNodeId);
            this.updateDeleteButtonState();
            // 更新筛选下拉列表
            this.updateFilterSelect(panelData.nodes);
        },
        /**
         * 保存修改后的 prefab 文件
         */
        async savePrefab(isSaveAs) {
            if (!panelData.originalFilePath) {
                console.error('[prefab-analysis] 没有原始文件路径');
                return;
            }
            if (!panelData.prefabJson) {
                console.error('[prefab-analysis] 没有可保存的数据');
                return;
            }
            try {
                // 获取原始文件的目录和文件名
                const dir = (0, path_1.dirname)(panelData.originalFilePath);
                const fileName = (0, path_1.basename)(panelData.originalFilePath);
                const ext = (0, path_1.extname)(fileName);
                const nameWithoutExt = fileName.substring(0, fileName.length - ext.length);
                // 生成新文件名：原名+"_new"
                const newFileName = isSaveAs ? `${nameWithoutExt}_new${ext}` : `${nameWithoutExt}${ext}`;
                const newFilePath = (0, path_1.join)(dir, newFileName);
                // 将 prefabJson 转换为 JSON 字符串并保存
                const jsonString = JSON.stringify(panelData.prefabJson, null, 2);
                (0, fs_extra_1.writeFileSync)(newFilePath, jsonString, 'utf-8');
                // 提示保存成功
                if (this.$.app) {
                    this.$.app.innerHTML = `保存成功: <strong>${newFileName}</strong>`;
                }
                console.log(`[prefab-analysis] 文件已保存到: ${newFilePath}`);
            }
            catch (error) {
                console.error('[prefab-analysis] 保存文件失败:', error);
                if (this.$.app) {
                    this.$.app.innerHTML = `保存失败: ${error instanceof Error ? error.message : String(error)}`;
                }
            }
        },
        /**
         * 处理 prefab 选择事件
         * @param {string} uuid - prefab 资源的 uuid
         */
        onPrefabSelected(uuid) {
            if (uuid === null || uuid === void 0 ? void 0 : uuid.trim()) {
                this.parsePrefab(uuid);
            }
            else {
                // 清空显示
                if (this.$.app) {
                    this.$.app.innerHTML = '未选中 Prefab';
                }
                panelData.prefabJson = null;
                panelData.nodes = [];
                panelData.selectedNodeId = null;
                panelData.filterType = '';
                panelData.sortType = 'none';
                panelData.deletedNodes.clear();
                panelData.originalFilePath = null;
                this.updateNodeList([]);
                this.updateNodeDetails(null);
            }
        },
        /**
         * 更新排序按钮文本
         */
        updateSortButtonText() {
            if (this.$.sortButton) {
                const sortTexts = {
                    none: '排序: 无',
                    asc: '排序: 升序',
                    desc: '排序: 降序',
                };
                this.$.sortButton.textContent = sortTexts[panelData.sortType] || '排序: 无';
            }
        },
        /**
         * 切换排序
         */
        toggleSort() {
            if (panelData.sortType === 'none') {
                panelData.sortType = 'asc';
            }
            else if (panelData.sortType === 'asc') {
                panelData.sortType = 'desc';
            }
            else {
                panelData.sortType = 'none';
            }
            this.updateSortButtonText();
            this.updateNodeList(panelData.nodes);
        },
        /**
         * 应用筛选
         * @param {string} filterValue - 筛选值
         */
        applyFilter(filterValue) {
            panelData.filterType = filterValue;
            this.updateNodeList();
        },
        /**
         * 获取所有唯一的 type 列表
         * @param {NodeInfo[]} nodes - 节点数组
         * @returns {string[]} 唯一的 type 列表
         */
        getUniqueTypes(nodes) {
            var _a;
            const typeSet = new Set();
            for (const node of nodes) {
                if ((_a = node.type) === null || _a === void 0 ? void 0 : _a.trim()) {
                    typeSet.add(node.type);
                }
            }
            return Array.from(typeSet).sort();
        },
        /**
         * 更新筛选下拉列表选项
         * @param {NodeInfo[]} nodes - 节点数组
         */
        updateFilterSelect(nodes) {
            if (!this.$.filterSelect) {
                return;
            }
            const select = this.$.filterSelect;
            const uniqueTypes = this.getUniqueTypes(nodes);
            // 清空现有选项（保留"全部"选项）
            select.innerHTML = '<option value="">全部</option>';
            // 添加所有唯一的 type
            for (const type of uniqueTypes) {
                const option = document.createElement('option');
                option.value = type;
                option.textContent = type;
                select.appendChild(option);
            }
            // 如果当前筛选的 type 存在，保持选中状态
            if (panelData.filterType) {
                select.value = panelData.filterType;
            }
        },
        /**
         * 初始化左右分栏拖拽功能
         */
        initPanelDividerDrag() {
            const divider = this.$.panelDivider;
            if (!divider) {
                return;
            }
            let isDragging = false;
            let startX = 0;
            let startLeftWidth = 0;
            const handleMouseDown = (e) => {
                const mouseEvent = e;
                isDragging = true;
                startX = mouseEvent.clientX;
                const leftPanel = divider.previousElementSibling;
                if (leftPanel) {
                    const rect = leftPanel.getBoundingClientRect();
                    startLeftWidth = rect.width;
                    divider.classList.add('dragging');
                }
                document.addEventListener('mousemove', handleMouseMove);
                document.addEventListener('mouseup', handleMouseUp);
                mouseEvent.preventDefault();
            };
            const handleMouseMove = (e) => {
                if (!isDragging) {
                    return;
                }
                const mouseEvent = e;
                const deltaX = mouseEvent.clientX - startX;
                const leftPanel = divider.previousElementSibling;
                if (leftPanel) {
                    const container = leftPanel.parentElement;
                    if (container) {
                        const containerWidth = container.getBoundingClientRect().width;
                        const newLeftWidth = startLeftWidth + deltaX;
                        const minWidth = 200;
                        const maxWidth = containerWidth - 200;
                        if (newLeftWidth >= minWidth && newLeftWidth <= maxWidth) {
                            const percentage = (newLeftWidth / containerWidth) * 100;
                            leftPanel.style.setProperty('--left-panel-width', `${percentage}%`);
                            leftPanel.style.flex = `0 0 ${percentage}%`;
                        }
                    }
                }
            };
            const handleMouseUp = () => {
                isDragging = false;
                divider.classList.remove('dragging');
                document.removeEventListener('mousemove', handleMouseMove);
                document.removeEventListener('mouseup', handleMouseUp);
            };
            divider.addEventListener('mousedown', handleMouseDown);
        },
        /**
         * 初始化列宽拖拽功能
         */
        initColumnResize() {
            // 延迟执行，确保 DOM 已渲染
            setTimeout(() => {
                // 获取包含 node-header 和 node-list 的容器
                const container = document.querySelector('#node-list-container');
                if (!container) {
                    return;
                }
                const header = container.querySelector('.node-header');
                if (!header) {
                    return;
                }
                const resizeHandles = header.querySelectorAll('.resize-handle');
                resizeHandles.forEach((handle) => {
                    let isDragging = false;
                    let startX = 0;
                    let startWidth = 0;
                    let columnIndex = 0;
                    const handleMouseDown = (e) => {
                        const mouseEvent = e;
                        isDragging = true;
                        startX = mouseEvent.clientX;
                        handle.classList.add('dragging');
                        const column = handle.getAttribute('data-column');
                        const columns = ['id', 'type', 'name', 'tags'];
                        columnIndex = columns.indexOf(column || '');
                        if (columnIndex >= 0) {
                            // 获取实际计算的宽度 - 使用直接子元素
                            const columnElements = Array.from(header.children);
                            if (columnElements[columnIndex]) {
                                const computedWidth = columnElements[columnIndex].getBoundingClientRect().width;
                                startWidth = computedWidth || 60;
                            }
                            else {
                                // 默认值
                                const defaults = { id: 60, type: 150, name: 200, parentId: 100, tags: 200 };
                                startWidth = defaults[columns[columnIndex]] || 60;
                            }
                        }
                        document.addEventListener('mousemove', handleMouseMove);
                        document.addEventListener('mouseup', handleMouseUp);
                        mouseEvent.preventDefault();
                        mouseEvent.stopPropagation();
                    };
                    const handleMouseMove = (e) => {
                        if (!isDragging || columnIndex < 0) {
                            return;
                        }
                        const mouseEvent = e;
                        const deltaX = mouseEvent.clientX - startX;
                        const columns = ['id', 'type', 'name', 'tags'];
                        const columnName = columns[columnIndex];
                        const cssVarName = `--col-${columnName}-width`;
                        let newWidth = startWidth + deltaX;
                        const minWidth = 30;
                        if (newWidth < minWidth) {
                            newWidth = minWidth;
                        }
                        // 更新 CSS 变量到容器上，这样 header 和 items 都能通过 CSS 继承访问
                        // CSS 变量会作用在容器及其所有子元素上
                        container.style.setProperty(cssVarName, `${newWidth}px`);
                    };
                    const handleMouseUp = () => {
                        isDragging = false;
                        handle.classList.remove('dragging');
                        document.removeEventListener('mousemove', handleMouseMove);
                        document.removeEventListener('mouseup', handleMouseUp);
                    };
                    handle.addEventListener('mousedown', handleMouseDown);
                });
            }, 100);
        },
        /**
         * 初始化筛选和排序功能
         */
        initFilterAndSort() {
            // 筛选下拉列表事件
            if (this.$.filterSelect) {
                this.$.filterSelect.addEventListener('change', (e) => {
                    const target = e.target;
                    const filterValue = target.value || '';
                    this.applyFilter(filterValue);
                });
            }
            // 排序按钮事件
            if (this.$.sortButton) {
                this.$.sortButton.addEventListener('click', () => {
                    this.toggleSort();
                });
            }
            // 主要节点开关事件
            if (this.$.mainNodesToggle) {
                const toggleElement = this.$.mainNodesToggle;
                toggleElement.addEventListener('change', (e) => {
                    const target = e.target;
                    panelData.showMainNodesOnly = target.checked;
                    this.updateNodeList();
                });
            }
            else {
                console.error('[prefab-analysis] 未找到 #main-nodes-toggle 元素');
            }
            // 初始化 Type 列标题排序功能
            this.initTypeHeaderSort();
        },
        /**
         * 初始化 Type 列标题排序功能
         */
        initTypeHeaderSort() {
            // 移除旧的监听器（如果存在）
            const oldHeader = document.querySelector('.node-type .header-text.sortable');
            if (oldHeader === null || oldHeader === void 0 ? void 0 : oldHeader.dataset.hasListener) {
                // 已经添加过监听器，跳过
                return;
            }
            // 延迟执行，确保 DOM 已渲染
            setTimeout(() => {
                const typeHeader = document.querySelector('.node-type .header-text.sortable');
                if (typeHeader && !typeHeader.dataset.hasListener) {
                    typeHeader.addEventListener('click', (e) => {
                        e.stopPropagation();
                        this.toggleSort();
                    });
                    typeHeader.style.cursor = 'pointer';
                    typeHeader.dataset.hasListener = 'true';
                }
            }, 100);
        },
    },
    ready() {
        if (this.$.app) {
            this.$.app.innerHTML = '未选中 Prefab';
        }
        if (this.$.nodeList) {
            this.$.nodeList.innerHTML = '<div class="node-item">暂无节点</div>';
        }
        // 初始化左右分栏拖拽功能
        this.initPanelDividerDrag();
        // 初始化列宽拖拽功能
        this.initColumnResize();
        // 初始化筛选和排序功能
        this.initFilterAndSort();
        // 更新排序按钮文本
        this.updateSortButtonText();
        // 更新删除按钮状态
        this.updateDeleteButtonState();
        // 初始化删除按钮事件监听器
        if (this.$.deleteSelectedButton) {
            this.$.deleteSelectedButton.addEventListener('click', () => {
                this.deleteSelectedNodes();
            });
        }
        // 初始化保存按钮事件监听器
        if (this.$.saveButton) {
            this.$.saveButton.addEventListener('click', () => {
                this.savePrefab(false);
            });
        }
        // 初始化另存按钮事件监听器
        if (this.$.saveAsButton) {
            this.$.saveAsButton.addEventListener('click', () => {
                this.savePrefab(true);
            });
        }
        // 绑定方法并保存引用，以便后续移除监听器
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore - onPrefabSelected 方法在 methods 中定义，运行时可用
        prefabSelectedHandler = this.onPrefabSelected.bind(this);
        // 监听 prefab 选择事件
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore - addBroadcastListener 方法在运行时存在，但类型定义中可能缺失
        if (Editor.Message.addBroadcastListener && prefabSelectedHandler) {
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            Editor.Message.addBroadcastListener(`${package_json_1.default.name}:prefab-selected`, prefabSelectedHandler);
        }
    },
    beforeClose() {
        // 移除监听器
        if (prefabSelectedHandler) {
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore - removeBroadcastListener 方法在运行时存在，但类型定义中可能缺失
            if (Editor.Message.removeBroadcastListener) {
                // eslint-disable-next-line @typescript-eslint/ban-ts-comment
                // @ts-ignore
                Editor.Message.removeBroadcastListener(`${package_json_1.default.name}:prefab-selected`, prefabSelectedHandler);
            }
        }
    },
    close() {
        // 移除监听器
        if (prefabSelectedHandler) {
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore - removeBroadcastListener 方法在运行时存在，但类型定义中可能缺失
            if (Editor.Message.removeBroadcastListener) {
                // eslint-disable-next-line @typescript-eslint/ban-ts-comment
                // @ts-ignore
                Editor.Message.removeBroadcastListener(`${package_json_1.default.name}:prefab-selected`, prefabSelectedHandler);
            }
            prefabSelectedHandler = null;
        }
    },
});
//# sourceMappingURL=data:application/json;base64,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