import { readFileSync, writeFileSync } from 'fs-extra';
import { join, dirname, basename, extname } from 'path';

// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore
import packageJSON from '../../../package.json';

// 使用闭包变量保存处理器引用
let prefabSelectedHandler: ((uuid: string) => void) | null = null;

// 节点引用接口
interface NodeReference {
    type: 'parent' | 'child' | 'component' | 'prefab' | 'instance' | 'propertyOverrides' | 'mountedChildren' | 'mountedComponents' | 'removedComponents' | 'clickEvents' | 'targetInfo' | 'targetOverrides' | 'nestedPrefabInstanceRoots' | 'node' | 'root' | 'asset' | 'source' | 'sourceInfo' | 'target' | 'prefabRootNode' | 'data' | 'content' | 'checkMark' | 'cameraComponent' | 'globals' | 'other';
    id: number;
}

// Node 信息接口
interface NodeInfo {
    id: number;
    type: string;
    name: string;
    parentId: number | null;
    references: NodeReference[];
}

// 面板数据接口
interface PanelData {
    prefabJson: any[] | null;
    nodes: NodeInfo[];
    selectedNodeId: number | null;
    filterType: string;
    sortType: 'none' | 'asc' | 'desc';
    showMainNodesOnly: boolean;
    deletedNodes: Set<number>;
    originalFilePath: string | null;
}

// 保存面板数据
const panelData: PanelData = {
    prefabJson: null,
    nodes: [],
    selectedNodeId: null,
    filterType: '',
    sortType: 'none',
    showMainNodesOnly: false,
    deletedNodes: new Set<number>(),
    originalFilePath: null,
};

/**
 * @zh 如果希望兼容 3.3 之前的版本可以使用下方的代码
 * @en You can add the code below if you want compatibility with versions prior to 3.3
 */

// Editor.Panel.define = Editor.Panel.define || function(options: any) { return options }
module.exports = Editor.Panel.define({
    listeners: {
        show() { console.log('show'); },
        hide() { console.log('hide'); },
    },
    template: readFileSync(join(__dirname, '../../../static/template/default/index.html'), 'utf-8'),
    style: readFileSync(join(__dirname, '../../../static/style/default/index.css'), 'utf-8'),
    $: {
        app: '#app',
        nodeList: '#node-list',
        nodeDetails: '#node-details',
        panelDivider: '#panel-divider',
        filterSelect: '#filter-select',
        sortButton: '#sort-button',
        typeHeader: '.node-type .header-text',
        mainNodesToggle: '#main-nodes-toggle',
        deleteSelectedButton: '#delete-selected-button',
        saveButton: '#save-button',
        saveAsButton: '#save-as-button',
    },
    methods: {
        hello() {
            if (this.$.app) {
                this.$.app.innerHTML = 'hello';
                console.log('[cocos-panel-html.default]: hello');
            }
        },
        /**
         * 解析 prefab 文件并显示 node 列表
         * @param {string} uuid - prefab 资源的 uuid
         */
        async parsePrefab(uuid: string) {
            try {
                // 获取资源信息
                const assetInfo = await Editor.Message.request('asset-db', 'query-asset-info', uuid);
                
                if (!assetInfo?.file) {
                    this.updateNodeList([]);

                    if (this.$.app) {
                        this.$.app.innerHTML = '无法读取 prefab 文件';
                    }

                    panelData.prefabJson = null;
                    panelData.nodes = [];
                    panelData.selectedNodeId = null;
                    panelData.deletedNodes.clear();
                    panelData.originalFilePath = null;
                    this.updateNodeDetails(null);

                    return;
                }

                // 读取 prefab 文件
                const prefabContent = readFileSync(assetInfo.file, 'utf-8');
                const prefabJson = JSON.parse(prefabContent);

                if (!Array.isArray(prefabJson)) {
                    this.updateNodeList([]);

                    if (this.$.app) {
                        this.$.app.innerHTML = 'prefab 文件格式错误';
                    }

                    panelData.prefabJson = null;
                    panelData.nodes = [];
                    panelData.selectedNodeId = null;
                    panelData.deletedNodes.clear();
                    panelData.originalFilePath = null;
                    this.updateNodeDetails(null);

                    return;
                }

                // 保存原始 prefab 数据
                panelData.prefabJson = prefabJson;
                panelData.selectedNodeId = null;
                panelData.filterType = '';
                panelData.sortType = 'none';
                panelData.deletedNodes.clear();
                panelData.originalFilePath = assetInfo.file;
                // 保持 showMainNodesOnly 的设置，不重置
                
                // 同步主要节点开关状态到数据（保持当前设置）
                if (this.$.mainNodesToggle) {
                    (this.$.mainNodesToggle as HTMLInputElement).checked = panelData.showMainNodesOnly;
                }

                // 解析 node 信息（处理所有类型的节点）
                const nodes: NodeInfo[] = [];
                
                for (let i = 0; i < prefabJson.length; i++) {
                    const item = prefabJson[i];
                    const references: NodeReference[] = [];
                    
                    const nodeInfo: NodeInfo = {
                        id: i,
                        type: item.__type__ || '',
                        name: item._name || '',
                        parentId: null,
                        references: [],
                    };

                    // 字段名到标签类型的映射表（已知字段使用对应类型，未知字段使用 'other'）
                    // eslint-disable-next-line @typescript-eslint/naming-convention
                    const fieldToTypeMap: Record<string, NodeReference['type']> = {
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _parent: 'parent',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _children: 'child',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _prefab: 'prefab',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        __prefab: 'prefab',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _components: 'component',
                        node: 'node',
                        data: 'data',
                        root: 'root',
                        asset: 'asset',
                        instance: 'instance',
                        prefabRootNode: 'prefabRootNode',
                        source: 'source',
                        sourceInfo: 'sourceInfo',
                        target: 'target',
                        propertyOverrides: 'propertyOverrides',
                        targetOverrides: 'targetOverrides',
                        mountedChildren: 'mountedChildren',
                        mountedComponents: 'mountedComponents',
                        removedComponents: 'removedComponents',
                        nestedPrefabInstanceRoots: 'nestedPrefabInstanceRoots',
                        clickEvents: 'clickEvents',
                        eventHandlers: 'clickEvents',
                        targetInfo: 'targetInfo',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _content: 'content',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _target: 'target',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _checkMark: 'checkMark',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _cameraComponent: 'cameraComponent',
                        // eslint-disable-next-line @typescript-eslint/naming-convention
                        _globals: 'globals',
                    };

                    // Helper 函数：添加引用
                    const addReference = (type: NodeReference['type'], ref: any): void => {
                        let id: number | undefined;

                        if (typeof ref === 'object' && ref !== null && ref.__id__ !== undefined) {
                            id = ref.__id__;
                        } else if (typeof ref === 'number') {
                            id = ref;
                        }

                        if (id !== undefined) {
                            references.push({ type, id });
                        }
                    };

                    // Helper 函数：递归遍历对象的所有字段，查找包含 __id__ 的引用
                    // fieldPath: 字段路径，用于嵌套字段（如 "__editorExtras__.mountedRoot"）
                    // currentFieldName: 当前字段名，用于确定标签类型
                    const traverseFields = (obj: any, fieldPath = '', currentFieldName = ''): void => {
                        if (obj === null || obj === undefined || typeof obj !== 'object') {
                            return;
                        }

                        // 如果是数组，遍历每个元素
                        if (Array.isArray(obj)) {
                            for (const arrayItem of obj) {
                                traverseFields(arrayItem, fieldPath, currentFieldName);
                            }

                            return;
                        }

                        // 检查当前对象是否包含 __id__（这是引用节点）
                        if (obj.__id__ !== undefined) {
                            // 如果 currentFieldName 为空，说明这是嵌套对象中的引用，使用 'other' 类型
                            const tagType = currentFieldName ? (fieldToTypeMap[currentFieldName] || 'other') : 'other';

                            addReference(tagType, obj);

                            // 找到引用后，不再继续遍历其内部（避免重复）
                            return;
                        }

                        // 遍历对象的所有字段
                        for (const fieldName in obj) {
                            // 跳过系统字段
                            if (fieldName === '__type__' || fieldName === '__id__' || fieldName === '_name') {
                                continue;
                            }

                            const fieldValue = obj[fieldName];

                            // 跳过 null、undefined
                            if (fieldValue === null || fieldValue === undefined) {
                                continue;
                            }

                            // 构建字段路径和当前字段名
                            const newFieldPath = fieldPath ? `${fieldPath}.${fieldName}` : fieldName;
                            const newCurrentFieldName = fieldPath ? currentFieldName : fieldName;

                            // 如果字段值是对象或数组，递归遍历
                            if (typeof fieldValue === 'object') {
                                traverseFields(fieldValue, newFieldPath, newCurrentFieldName);
                            }
                        }
                    };

                    // 处理 _parent（特殊处理，需要设置 parentId）
                    if (item._parent) {
                        const parentType = fieldToTypeMap._parent || 'other';

                        addReference(parentType, item._parent);
                        if (typeof item._parent === 'object' && item._parent.__id__ !== undefined) {
                            nodeInfo.parentId = item._parent.__id__;
                        }
                    }

                    // 递归遍历所有字段（跳过 _parent，因为已经特殊处理）
                    for (const fieldName in item) {
                        // 跳过特殊字段
                        if (
                            fieldName === '__type__' ||
                            fieldName === '__id__' ||
                            fieldName === '_name' ||
                            fieldName === '_parent' // 已特殊处理
                        ) {
                            continue;
                        }

                        const fieldValue = item[fieldName];

                        // 跳过 null、undefined
                        if (fieldValue === null || fieldValue === undefined) {
                            continue;
                        }

                        // 如果字段值是对象或数组，递归遍历
                        if (typeof fieldValue === 'object') {
                            traverseFields(fieldValue, '', fieldName);
                        }
                    }

                    nodeInfo.references = references;
                    nodes.push(nodeInfo);
                }

                panelData.nodes = nodes;

                // 更新显示
                const fileName = assetInfo.name || assetInfo.url?.split('/').pop() || '未知';
                // 统计节点数量
                const totalCount = nodes.length;
                const sceneCount = nodes.filter(node => node.type === 'cc.Scene').length;
                const shouldHideCount = nodes.filter(node => node.name === 'should_hide_in_hierarchy').length;

                if (this.$.app) {
                    let statsText = `(共 ${totalCount} 个节点`;

                    if (sceneCount > 0) {
                        statsText += `，cc.Scene: ${sceneCount}`;
                    }

                    if (shouldHideCount > 0) {
                        statsText += `，should_hide_in_hierarchy: ${shouldHideCount}`;
                    }

                    statsText += ')';
                    this.$.app.innerHTML = `选中的 Prefab: <strong>${fileName}</strong> ${statsText}`;
                }
                
                this.updateNodeList(nodes);
                this.updateNodeDetails(null);
                this.updateDeleteButtonState();
                
                // 更新筛选下拉列表选项
                this.updateFilterSelect(nodes);
                
                // 重置筛选为"全部"
                panelData.filterType = '';
                if (this.$.filterSelect) {
                    (this.$.filterSelect as HTMLSelectElement).value = '';
                }
                
                // 更新排序按钮文本
                this.updateSortButtonText();
                
                // 重新初始化列宽拖拽功能（因为 DOM 可能已经更新）
                this.initColumnResize();
                
                // 重新初始化 Type 列标题排序功能
                setTimeout(() => {
                    this.initTypeHeaderSort();
                }, 150);
            } catch (error) {
                console.error('[prefab-analysis] 解析 prefab 失败:', error);
                panelData.prefabJson = null;
                panelData.nodes = [];
                panelData.selectedNodeId = null;
                panelData.deletedNodes.clear();
                panelData.originalFilePath = null;
                this.updateNodeList([]);
                this.updateNodeDetails(null);

                if (this.$.app) {
                    this.$.app.innerHTML = `解析失败: ${error instanceof Error ? error.message : String(error)}`;
                }
            }
        },
        /**
         * 获取主要节点（cc.Prefab、cc.Node 以及没有被这些节点引用或间接引用的节点）
         * @param {NodeInfo[]} nodes - 所有节点数组
         * @returns {Set<number>} 主要节点的 ID 集合
         */
        getMainNodeIds(nodes: NodeInfo[]): Set<number> {
            // 第一步：找出所有 cc.Prefab 和 cc.Node 类型的节点
            const mainNodeIds = new Set<number>();
            
            for (const node of nodes) {
                if (node.type === 'cc.Prefab' || node.type === 'cc.Node'|| node.type === 'cc.PrefabInstance') {
                    mainNodeIds.add(node.id);
                }
            }

            // 第二步：找出所有被 cc.Prefab 和 cc.Node 直接或间接引用的节点
            const visitedIds = new Set<number>();
            const toVisit = Array.from(mainNodeIds);

            while (toVisit.length > 0) {
                const currentId = toVisit.pop();

                if (!currentId || visitedIds.has(currentId)) {
                    continue;
                }
                
                visitedIds.add(currentId);
                const currentNode = nodes.find(n => n.id === currentId);

                if (currentNode) {
                    // 遍历当前节点的所有引用
                    for (const ref of currentNode.references) {
                        if (!visitedIds.has(ref.id)) {
                            toVisit.push(ref.id);
                        }
                    }
                }
            }

            // 第三步：找出所有没有被引用的节点（孤立节点）
            for (const node of nodes) {
                if (!visitedIds.has(node.id)) {
                    mainNodeIds.add(node.id);
                }
            }

            return mainNodeIds;
        },
        /**
         * 获取筛选和排序后的节点列表
         * @param {NodeInfo[]} nodes - 原始节点数组
         * @returns {NodeInfo[]} 筛选和排序后的节点数组
         */
        getFilteredAndSortedNodes(nodes: NodeInfo[]): NodeInfo[] {
            let result = [...nodes];

            // 应用主要节点过滤
            if (panelData.showMainNodesOnly) {
                const mainNodeIds = this.getMainNodeIds(nodes);

                result = result.filter((node) => {
                    return mainNodeIds.has(node.id);
                });
            }

            // 应用类型筛选（精确匹配）
            if (panelData.filterType?.trim()) {
                result = result.filter((node) => {
                    return node.type === panelData.filterType;
                });
            }

            // 应用排序
            if (panelData.sortType !== 'none') {
                result.sort((a, b) => {
                    const typeA = (a.type || '').toLowerCase();
                    const typeB = (b.type || '').toLowerCase();

                    if (panelData.sortType === 'asc') {
                        return typeA.localeCompare(typeB);
                    } else {
                        return typeB.localeCompare(typeA);
                    }
                });
            }

            return result;
        },
        /**
         * 更新 node 列表显示
         * @param {NodeInfo[]} nodes - node 信息数组（如果传入，则更新原始节点数据；否则使用已保存的节点数据）
         */
        updateNodeList(nodes?: NodeInfo[]) {
            if (!this.$.nodeList) {
                return;
            }

            // 如果传入了 nodes，更新原始节点数据；否则使用已保存的节点数据
            if (nodes) {
                panelData.nodes = nodes;
            }

            // 如果没有节点数据，显示空状态
            if (!panelData.nodes || panelData.nodes.length === 0) {
                this.$.nodeList.innerHTML = '<div class="node-item">暂无节点</div>';

                return;
            }

            // 获取筛选和排序后的节点
            const filteredNodes = this.getFilteredAndSortedNodes(panelData.nodes);

            if (filteredNodes.length === 0) {
                this.$.nodeList.innerHTML = '<div class="node-item">暂无节点（已筛选）</div>';

                return;
            }

            let html = '';

            for (const node of filteredNodes) {
                // 如果字段为空，显示占位符
                const idText = node.id.toString();
                const typeText = node.type || '-';
                const nameText = node.name || '-';
                const selectedClass = node.id === panelData.selectedNodeId ? 'selected' : '';
                const deletedClass = panelData.deletedNodes.has(node.id) ? 'deleted' : '';
                // 检查 tag 中是否有被标记删除的节点
                let hasDeletedRef = false;

                if (node.references && node.references.length > 0) {
                    for (const ref of node.references) {
                        if (panelData.deletedNodes.has(ref.id)) {
                            hasDeletedRef = true;
                            break;
                        }
                    }
                }

                const hasDeletedRefClass = hasDeletedRef ? 'has-deleted-ref' : '';
                // 生成 tags HTML
                let tagsHtml = '';

                if (node.references && node.references.length > 0) {
                    for (const ref of node.references) {
                        const tagClass = `tag-${ref.type}`;
                        const refDeletedClass = panelData.deletedNodes.has(ref.id) ? 'ref-deleted' : '';

                        tagsHtml += `<span class="node-tag ${tagClass} ${refDeletedClass}" data-ref-id="${ref.id}" data-ref-type="${ref.type}">${ref.id}</span>`;
                    }
                } else {
                    tagsHtml = '-';
                }

                html += `
                    <div class="node-item ${selectedClass}" data-node-id="${node.id}">
                        <span class="node-id ${deletedClass} ${hasDeletedRefClass}">${idText}</span>
                        <span class="node-type">${typeText}</span>
                        <span class="node-name">${nameText}</span>
                        <span class="node-tags">${tagsHtml}</span>
                    </div>
                `;
            }

            this.$.nodeList.innerHTML = html;

            // 添加点击事件监听器
            const nodeItems = this.$.nodeList.querySelectorAll('.node-item');

            nodeItems.forEach((item) => {
                item.addEventListener('click', (e) => {
                    const target = e.currentTarget as HTMLElement;
                    const clickedTag = (e.target as HTMLElement).closest('.node-tag');

                    // 如果点击的是 tag，处理 tag 点击
                    if (clickedTag) {
                        e.stopPropagation();

                        const refId = parseInt(clickedTag.getAttribute('data-ref-id') || '-1', 10);

                        if (refId >= 0) {
                            this.selectNode(refId);
                        }

                        return;
                    }

                    // 否则处理节点点击
                    const nodeId = parseInt(target.getAttribute('data-node-id') || '-1', 10);

                    if (nodeId >= 0) {
                        this.selectNode(nodeId);
                    }
                });
            });
        },
        /**
         * 选中节点
         * @param {number} nodeId - 节点 ID
         */
        selectNode(nodeId: number) {
            panelData.selectedNodeId = nodeId;
            this.updateNodeList(panelData.nodes);
            this.updateNodeDetails(nodeId);
        },
        /**
         * 更新节点详细信息显示
         * @param {number | null} nodeId - 节点 ID，null 表示清空
         */
        updateNodeDetails(nodeId: number | null) {
            if (!this.$.nodeDetails) {
                return;
            }

            if (nodeId === null || !panelData.prefabJson || nodeId < 0 || nodeId >= panelData.prefabJson.length) {
                this.$.nodeDetails.innerHTML = '<div class="details-placeholder">点击左侧节点查看详细信息</div>';

                return;
            }

            const nodeData = panelData.prefabJson[nodeId];
            const nodeInfo = panelData.nodes[nodeId];

            if (!nodeData) {
                this.$.nodeDetails.innerHTML = '<div class="details-placeholder">节点数据不存在</div>';

                return;
            }

            // 构建详细信息 HTML
            let html = '<div class="details-content">';

            // 基本信息部分
            html += '<div class="details-section">';
            html += '<div class="details-section-title">基本信息</div>';

            if (nodeInfo.id !== undefined) {
                html += `<div class="details-property">
                    <span class="details-property-label">ID:</span>
                    <span class="details-property-value">${nodeInfo.id}</span>
                </div>`;
            }

            if (nodeInfo.type) {
                html += `<div class="details-property">
                    <span class="details-property-label">Type:</span>
                    <span class="details-property-value">${nodeInfo.type}</span>
                </div>`;
            }

            if (nodeInfo.name) {
                html += `<div class="details-property">
                    <span class="details-property-label">Name:</span>
                    <span class="details-property-value">${nodeInfo.name}</span>
                </div>`;
            }

            // 显示 Tags（按类型分组）
            if (nodeInfo.references && nodeInfo.references.length > 0) {
                // 按类型分组
                const tagsByType: Record<string, NodeReference[]> = {};

                for (const ref of nodeInfo.references) {
                    if (!tagsByType[ref.type]) {
                        tagsByType[ref.type] = [];
                    }

                    tagsByType[ref.type].push(ref);
                }

                // 定义类型显示顺序和中文名称
                const typeOrder: NodeReference['type'][] = [
                    'parent', 'child', 'prefab', 'component', 'node',
                    'root', 'asset', 'instance', 'prefabRootNode',
                    'source', 'sourceInfo', 'target', 'propertyOverrides',
                    'targetOverrides', 'mountedChildren', 'mountedComponents',
                    'removedComponents', 'nestedPrefabInstanceRoots', 'clickEvents',
                    'targetInfo', 'data', 'content', 'checkMark', 'cameraComponent',
                    'globals', 'other',
                ];

                const typeNames: Record<string, string> = {
                    parent: 'Parent',
                    child: 'Child',
                    prefab: 'Prefab',
                    component: 'Component',
                    node: 'Node',
                    root: 'Root',
                    asset: 'Asset',
                    instance: 'Instance',
                    prefabRootNode: 'PrefabRootNode',
                    source: 'Source',
                    sourceInfo: 'SourceInfo',
                    target: 'Target',
                    propertyOverrides: 'PropertyOverrides',
                    targetOverrides: 'TargetOverrides',
                    mountedChildren: 'MountedChildren',
                    mountedComponents: 'MountedComponents',
                    removedComponents: 'RemovedComponents',
                    nestedPrefabInstanceRoots: 'NestedPrefabInstanceRoots',
                    clickEvents: 'ClickEvents',
                    targetInfo: 'TargetInfo',
                    data: 'Data',
                    content: 'Content',
                    checkMark: 'CheckMark',
                    cameraComponent: 'CameraComponent',
                    globals: 'Globals',
                    other: 'Other',
                };

                let tagsHtml = '<div class="details-tags-grouped">';

                for (const type of typeOrder) {
                    if (tagsByType[type] && tagsByType[type].length > 0) {
                        const typeName = typeNames[type] || type;

                        tagsHtml += `<div class="details-tag-group">`;
                        tagsHtml += `<div class="details-tag-group-label">${typeName}:</div>`;
                        tagsHtml += `<div class="details-tag-group-items">`;

                        for (const ref of tagsByType[type]) {
                            const tagClass = `tag-${ref.type}`;
                            const refDeletedClass = panelData.deletedNodes.has(ref.id) ? 'ref-deleted' : '';

                            tagsHtml += `<span class="node-tag ${tagClass} ${refDeletedClass}" data-ref-id="${ref.id}" data-ref-type="${ref.type}">${ref.id}</span>`;
                        }

                        tagsHtml += `</div></div>`;
                    }
                }

                tagsHtml += '</div>';

                html += `<div class="details-property">
                    <span class="details-property-label">Tags:</span>
                    <span class="details-property-value">${tagsHtml}</span>
                </div>`;
            } else {
                html += `<div class="details-property">
                    <span class="details-property-label">Tags:</span>
                    <span class="details-property-value">-</span>
                </div>`;
            }


            html += '</div>';

            // 完整 JSON 数据部分
            html += '<div class="details-section">';
            html += '<div class="details-section-title">完整数据</div>';
            html += `<div class="details-json">${JSON.stringify(nodeData, null, 2)}</div>`;
            html += '</div>';

            // 标记删除/清除标记按钮
            const isDeleted = panelData.deletedNodes.has(nodeId);
            const deleteButtonText = isDeleted ? '清除标记（递归）' : '标记删除（递归）';
            const deleteSingleButtonText = isDeleted ? '清除标记（单节点）' : '标记删除（单节点）';

            html += '<div class="details-section">';
            html += `<button id="delete-mark-recursive-button" class="delete-mark-btn">${deleteButtonText}</button>`;
            html += `<button id="delete-mark-single-button" class="delete-mark-btn">${deleteSingleButtonText}</button>`;
            html += '</div>';

            html += '</div>';

            this.$.nodeDetails.innerHTML = html;

            // 为详细信息中的 tags 添加点击事件
            const detailTags = this.$.nodeDetails.querySelectorAll('.node-tag');

            detailTags.forEach((tag) => {
                tag.addEventListener('click', (e) => {
                    e.stopPropagation();

                    const clickedTag = e.currentTarget as HTMLElement;
                    const refId = parseInt(clickedTag.getAttribute('data-ref-id') || '-1', 10);

                    if (refId >= 0) {
                        this.selectNode(refId);
                    }
                });
            });

            // 为标记删除（递归）按钮添加点击事件
            const deleteMarkRecursiveButton = this.$.nodeDetails.querySelector('#delete-mark-recursive-button');

            if (deleteMarkRecursiveButton) {
                deleteMarkRecursiveButton.addEventListener('click', () => {
                    this.toggleDeleteMark(nodeId);
                });
            }

            // 为标记删除（单节点）按钮添加点击事件
            const deleteMarkSingleButton = this.$.nodeDetails.querySelector('#delete-mark-single-button');

            if (deleteMarkSingleButton) {
                deleteMarkSingleButton.addEventListener('click', () => {
                    this.toggleDeleteMarkSingle(nodeId);
                });
            }
        },
        /**
         * 递归标记节点及其引用的所有节点为删除
         * @param {number} nodeId - 节点 ID
         */
        markNodeAndRefsAsDeleted(nodeId: number) {
            // 如果节点已经被标记删除，跳过（避免无限递归）
            if (panelData.deletedNodes.has(nodeId)) {
                return;
            }

            // 标记当前节点为删除
            panelData.deletedNodes.add(nodeId);

            // 获取当前节点的信息
            const nodeInfo = panelData.nodes[nodeId];

            if (!nodeInfo?.references) {
                return;
            }

            // 递归标记所有被引用的节点
            for (const ref of nodeInfo.references) {
                if (ref.id >= 0 && ref.id < panelData.nodes.length) {
                    this.markNodeAndRefsAsDeleted(ref.id);
                }
            }
        },
        /**
         * 递归清除节点及其引用的所有节点的删除标记
         * @param {number} nodeId - 节点 ID
         */
        unmarkNodeAndRefsAsDeleted(nodeId: number) {
            // 如果节点没有被标记删除，跳过（避免无限递归）
            if (!panelData.deletedNodes.has(nodeId)) {
                return;
            }

            // 清除当前节点的删除标记
            panelData.deletedNodes.delete(nodeId);

            // 获取当前节点的信息
            const nodeInfo = panelData.nodes[nodeId];

            if (!nodeInfo?.references) {
                return;
            }

            // 递归清除所有被引用节点的删除标记
            for (const ref of nodeInfo.references) {
                if (ref.id >= 0 && ref.id < panelData.nodes.length) {
                    this.unmarkNodeAndRefsAsDeleted(ref.id);
                }
            }
        },
        /**
         * 切换节点的删除标记（递归标记/清除引用的节点）
         * @param {number} nodeId - 节点 ID
         */
        toggleDeleteMark(nodeId: number) {
            if (panelData.deletedNodes.has(nodeId)) {
                // 清除标记（递归清除所有被引用节点的删除标记）
                this.unmarkNodeAndRefsAsDeleted(nodeId);
            } else {
                // 标记删除（递归标记所有被引用的节点）
                this.markNodeAndRefsAsDeleted(nodeId);
            }

            // 更新列表和详情显示
            this.updateNodeList(panelData.nodes);
            this.updateNodeDetails(nodeId);
            // 更新删除按钮状态
            this.updateDeleteButtonState();
        },
        /**
         * 切换节点的删除标记（只标记当前节点，不递归）
         * @param {number} nodeId - 节点 ID
         */
        toggleDeleteMarkSingle(nodeId: number) {
            if (panelData.deletedNodes.has(nodeId)) {
                // 清除标记
                panelData.deletedNodes.delete(nodeId);
            } else {
                // 只标记当前节点
                panelData.deletedNodes.add(nodeId);
            }

            // 更新列表和详情显示
            this.updateNodeList(panelData.nodes);
            this.updateNodeDetails(nodeId);
            // 更新删除按钮状态
            this.updateDeleteButtonState();
        },
        /**
         * 更新删除按钮的状态和文本
         */
        updateDeleteButtonState() {
            if (!this.$.deleteSelectedButton) {
                return;
            }

            const button = this.$.deleteSelectedButton as HTMLButtonElement;
            const deletedCount = panelData.deletedNodes.size;

            if (deletedCount > 0) {
                button.disabled = false;
                button.textContent = `删除选中节点(${deletedCount})`;
            } else {
                button.disabled = true;
                button.textContent = '删除选中节点';
            }
        },
        /**
         * 删除所有标记为删除的节点
         */
        deleteSelectedNodes() {
            if (!panelData.prefabJson || !panelData.nodes || panelData.deletedNodes.size === 0) {
                return;
            }

            // 将被删除的节点ID转换为数组并排序（从大到小）
            const deletedIds = Array.from(panelData.deletedNodes).sort((a, b) => b - a);
            // 创建索引映射表：旧索引 -> 新索引（只包含保留的节点）
            const indexMap: number[] = [];
            let newIndex = 0;

            for (let oldIndex = 0; oldIndex < panelData.nodes.length; oldIndex++) {
                if (!panelData.deletedNodes.has(oldIndex)) {
                    indexMap[oldIndex] = newIndex;
                    newIndex++;
                }
            }

            // 从大到小删除节点（避免索引变化影响删除）
            for (const deletedId of deletedIds) {
                panelData.prefabJson.splice(deletedId, 1);
                panelData.nodes.splice(deletedId, 1);
            }

            // 创建要删除的节点ID的Set，用于快速查找
            const deletedIdsSet = new Set(deletedIds);

            // 递归函数：更新对象中的所有 __id__ 引用
            const updateReferences = (obj: any): void => {
                if (obj === null || obj === undefined) {
                    return;
                }

                if (typeof obj !== 'object') {
                    return;
                }

                // 如果是数组，遍历每个元素
                if (Array.isArray(obj)) {
                    for (let i = obj.length - 1; i >= 0; i--) {
                        const item = obj[i];

                        // 如果是包含 __id__ 的对象，检查是否需要删除或更新
                        if (item && typeof item === 'object' && item.__id__ !== undefined) {
                            const oldId = item.__id__;

                            if (deletedIdsSet.has(oldId)) {
                                // 引用已被删除，从数组中移除
                                obj.splice(i, 1);
                            } else if (indexMap[oldId] !== undefined) {
                                // 更新引用ID
                                item.__id__ = indexMap[oldId];
                                // 递归处理该对象的其他字段
                                updateReferences(item);
                            } else {
                                // 递归处理该对象的其他字段
                                updateReferences(item);
                            }
                        } else {
                            // 递归处理数组元素
                            updateReferences(item);
                        }
                    }
                } else {
                    // 如果是对象，遍历所有字段
                    for (const key in obj) {
                        if (key === '__id__') {
                            // 跳过 __id__ 字段本身，因为节点的 __id__ 不需要更新（它们对应数组索引）
                            continue;
                        }

                        const value = obj[key];

                        // 如果值是包含 __id__ 的对象，更新或删除
                        if (value && typeof value === 'object' && value.__id__ !== undefined) {
                            const oldId = value.__id__;

                            if (deletedIdsSet.has(oldId)) {
                                // 引用已被删除，设置为 null
                                obj[key] = null;
                            } else if (indexMap[oldId] !== undefined) {
                                // 更新引用ID
                                value.__id__ = indexMap[oldId];
                                // 递归处理该对象
                                updateReferences(value);
                            } else {
                                // 递归处理该对象
                                updateReferences(value);
                            }
                        } else {
                            // 递归处理其他类型的值
                            updateReferences(value);
                        }
                    }
                }
            };

            // 更新 prefabJson 中所有节点的引用
            for (let i = 0; i < panelData.prefabJson.length; i++) {
                // // 更新节点本身的 __id__ 为数组索引
                // if (panelData.prefabJson[i] && typeof panelData.prefabJson[i] === 'object') {
                //     panelData.prefabJson[i].__id__ = i;
                // }

                // 更新节点内部的所有引用
                updateReferences(panelData.prefabJson[i]);
            }

            // 更新所有节点的引用（将旧的节点ID映射到新的节点ID）
            for (const node of panelData.nodes) {
                if (node.references) {
                    for (const ref of node.references) {
                        const oldRefId = ref.id;

                        if (deletedIdsSet.has(oldRefId)) {
                            // 引用已被删除，标记为需要移除
                            ref.id = -1;
                        } else if (indexMap[oldRefId] !== undefined) {
                            // 更新引用ID为新的索引
                            ref.id = indexMap[oldRefId];
                        }
                    }

                    // 移除标记为-1的引用（已被删除的节点）
                    node.references = node.references.filter(ref => ref.id >= 0);
                }
            }

            // 更新节点的ID（因为数组索引就是节点ID）
            for (let i = 0; i < panelData.nodes.length; i++) {
                panelData.nodes[i].id = i;
            }

            // 清空删除标记
            panelData.deletedNodes.clear();

            // 更新选中节点ID
            if (panelData.selectedNodeId !== null) {
                if (deletedIds.includes(panelData.selectedNodeId)) {
                    // 当前选中的节点被删除，清空选中状态
                    panelData.selectedNodeId = null;
                } else if (indexMap[panelData.selectedNodeId] !== undefined) {
                    // 更新选中节点的ID
                    panelData.selectedNodeId = indexMap[panelData.selectedNodeId];
                }
            }

            // 更新显示
            this.updateNodeList(panelData.nodes);
            this.updateNodeDetails(panelData.selectedNodeId);
            this.updateDeleteButtonState();

            // 更新筛选下拉列表
            this.updateFilterSelect(panelData.nodes);
        },
        /**
         * 保存修改后的 prefab 文件
         */
        async savePrefab(isSaveAs: boolean) {
            if (!panelData.originalFilePath) {
                console.error('[prefab-analysis] 没有原始文件路径');

                return;
            }

            if (!panelData.prefabJson) {
                console.error('[prefab-analysis] 没有可保存的数据');

                return;
            }

            try {
                // 获取原始文件的目录和文件名
                const dir = dirname(panelData.originalFilePath);
                const fileName = basename(panelData.originalFilePath);
                const ext = extname(fileName);
                const nameWithoutExt = fileName.substring(0, fileName.length - ext.length);
                // 生成新文件名：原名+"_new"
                const newFileName = isSaveAs ? `${nameWithoutExt}_new${ext}` : `${nameWithoutExt}${ext}`;
                const newFilePath = join(dir, newFileName);
                // 将 prefabJson 转换为 JSON 字符串并保存
                const jsonString = JSON.stringify(panelData.prefabJson, null, 2);

                writeFileSync(newFilePath, jsonString, 'utf-8');

                // 提示保存成功
                if (this.$.app) {
                    this.$.app.innerHTML = `保存成功: <strong>${newFileName}</strong>`;
                }

                console.log(`[prefab-analysis] 文件已保存到: ${newFilePath}`);
            } catch (error) {
                console.error('[prefab-analysis] 保存文件失败:', error);

                if (this.$.app) {
                    this.$.app.innerHTML = `保存失败: ${error instanceof Error ? error.message : String(error)}`;
                }
            }
        },
        /**
         * 处理 prefab 选择事件
         * @param {string} uuid - prefab 资源的 uuid
         */
        onPrefabSelected(uuid: string) {
            if (uuid?.trim()) {
                this.parsePrefab(uuid);
            } else {
                // 清空显示
                if (this.$.app) {
                    this.$.app.innerHTML = '未选中 Prefab';
                }

                panelData.prefabJson = null;
                panelData.nodes = [];
                panelData.selectedNodeId = null;
                panelData.filterType = '';
                panelData.sortType = 'none';
                panelData.deletedNodes.clear();
                panelData.originalFilePath = null;
                this.updateNodeList([]);
                this.updateNodeDetails(null);
            }
        },
        /**
         * 更新排序按钮文本
         */
        updateSortButtonText() {
            if (this.$.sortButton) {
                const sortTexts = {
                    none: '排序: 无',
                    asc: '排序: 升序',
                    desc: '排序: 降序',
                };

                this.$.sortButton.textContent = sortTexts[panelData.sortType] || '排序: 无';
            }
        },
        /**
         * 切换排序
         */
        toggleSort() {
            if (panelData.sortType === 'none') {
                panelData.sortType = 'asc';
            } else if (panelData.sortType === 'asc') {
                panelData.sortType = 'desc';
            } else {
                panelData.sortType = 'none';
            }

            this.updateSortButtonText();
            this.updateNodeList(panelData.nodes);
        },
        /**
         * 应用筛选
         * @param {string} filterValue - 筛选值
         */
        applyFilter(filterValue: string) {
            panelData.filterType = filterValue;
            this.updateNodeList();
        },
        /**
         * 获取所有唯一的 type 列表
         * @param {NodeInfo[]} nodes - 节点数组
         * @returns {string[]} 唯一的 type 列表
         */
        getUniqueTypes(nodes: NodeInfo[]): string[] {
            const typeSet = new Set<string>();

            for (const node of nodes) {
                if (node.type?.trim()) {
                    typeSet.add(node.type);
                }
            }

            return Array.from(typeSet).sort();
        },
        /**
         * 更新筛选下拉列表选项
         * @param {NodeInfo[]} nodes - 节点数组
         */
        updateFilterSelect(nodes: NodeInfo[]) {
            if (!this.$.filterSelect) {
                return;
            }

            const select = this.$.filterSelect as HTMLSelectElement;
            const uniqueTypes = this.getUniqueTypes(nodes);
            
            // 清空现有选项（保留"全部"选项）
            select.innerHTML = '<option value="">全部</option>';


            // 添加所有唯一的 type
            for (const type of uniqueTypes) {
                const option = document.createElement('option');

                option.value = type;
                option.textContent = type;
                select.appendChild(option);
            }

            // 如果当前筛选的 type 存在，保持选中状态
            if (panelData.filterType) {
                select.value = panelData.filterType;
            }
        },
        /**
         * 初始化左右分栏拖拽功能
         */
        initPanelDividerDrag(): void {
            const divider = this.$.panelDivider;

            if (!divider) {
                return;
            }

            let isDragging = false;
            let startX = 0;
            let startLeftWidth = 0;

            const handleMouseDown = (e: Event): void => {
                const mouseEvent = e as MouseEvent;

                isDragging = true;
                startX = mouseEvent.clientX;
                
                const leftPanel = divider.previousElementSibling as HTMLElement;
                
                if (leftPanel) {
                    const rect = leftPanel.getBoundingClientRect();

                    startLeftWidth = rect.width;
                    divider.classList.add('dragging');
                }

                document.addEventListener('mousemove', handleMouseMove);
                document.addEventListener('mouseup', handleMouseUp);
                mouseEvent.preventDefault();
            };

            const handleMouseMove = (e: Event): void => {
                if (!isDragging) {
                    return;
                }

                const mouseEvent = e as MouseEvent;
                const deltaX = mouseEvent.clientX - startX;
                const leftPanel = divider.previousElementSibling as HTMLElement;
                
                if (leftPanel) {
                    const container = leftPanel.parentElement;
                    
                    if (container) {
                        const containerWidth = container.getBoundingClientRect().width;
                        const newLeftWidth = startLeftWidth + deltaX;
                        const minWidth = 200;
                        const maxWidth = containerWidth - 200;

                        if (newLeftWidth >= minWidth && newLeftWidth <= maxWidth) {
                            const percentage = (newLeftWidth / containerWidth) * 100;

                            leftPanel.style.setProperty('--left-panel-width', `${percentage}%`);
                            leftPanel.style.flex = `0 0 ${percentage}%`;
                        }
                    }
                }
            };

            const handleMouseUp = (): void => {
                isDragging = false;
                divider.classList.remove('dragging');
                document.removeEventListener('mousemove', handleMouseMove);
                document.removeEventListener('mouseup', handleMouseUp);
            };

            divider.addEventListener('mousedown', handleMouseDown);
        },
        /**
         * 初始化列宽拖拽功能
         */
        initColumnResize(): void {
            // 延迟执行，确保 DOM 已渲染
            setTimeout(() => {
                // 获取包含 node-header 和 node-list 的容器
                const container = document.querySelector('#node-list-container') as HTMLElement;

                if (!container) {
                    return;
                }

                const header = container.querySelector('.node-header') as HTMLElement;

                if (!header) {
                    return;
                }

                const resizeHandles = header.querySelectorAll('.resize-handle');

                resizeHandles.forEach((handle) => {
                    let isDragging = false;
                    let startX = 0;
                    let startWidth = 0;
                    let columnIndex = 0;

                    const handleMouseDown = (e: Event): void => {
                        const mouseEvent = e as MouseEvent;

                        isDragging = true;
                        startX = mouseEvent.clientX;
                        handle.classList.add('dragging');

                        const column = handle.getAttribute('data-column');
                        const columns = ['id', 'type', 'name', 'tags'];

                        columnIndex = columns.indexOf(column || '');
                        
                        if (columnIndex >= 0) {
                            // 获取实际计算的宽度 - 使用直接子元素
                            const columnElements = Array.from(header.children) as HTMLElement[];

                            if (columnElements[columnIndex]) {
                                const computedWidth = columnElements[columnIndex].getBoundingClientRect().width;

                                startWidth = computedWidth || 60;
                            } else {
                                // 默认值
                                const defaults: { [key: string]: number } = { id: 60, type: 150, name: 200, parentId: 100, tags: 200 };

                                startWidth = defaults[columns[columnIndex]] || 60;
                            }
                        }

                        document.addEventListener('mousemove', handleMouseMove);
                        document.addEventListener('mouseup', handleMouseUp);
                        mouseEvent.preventDefault();
                        mouseEvent.stopPropagation();
                    };

                    const handleMouseMove = (e: Event): void => {
                        if (!isDragging || columnIndex < 0) {
                            return;
                        }

                        const mouseEvent = e as MouseEvent;
                    const deltaX = mouseEvent.clientX - startX;
                        const columns = ['id', 'type', 'name', 'tags'];
                        const columnName = columns[columnIndex];
                        const cssVarName = `--col-${columnName}-width`;
                        let newWidth = startWidth + deltaX;
                        const minWidth = 30;

                        if (newWidth < minWidth) {
                            newWidth = minWidth;
                        }

                        // 更新 CSS 变量到容器上，这样 header 和 items 都能通过 CSS 继承访问
                        // CSS 变量会作用在容器及其所有子元素上
                        container.style.setProperty(cssVarName, `${newWidth}px`);
                    };

                    const handleMouseUp = (): void => {
                        isDragging = false;
                        handle.classList.remove('dragging');
                        document.removeEventListener('mousemove', handleMouseMove);
                        document.removeEventListener('mouseup', handleMouseUp);
                    };

                    handle.addEventListener('mousedown', handleMouseDown);
                });
            }, 100);
        },
        /**
         * 初始化筛选和排序功能
         */
        initFilterAndSort(): void {
            // 筛选下拉列表事件
            if (this.$.filterSelect) {
                this.$.filterSelect.addEventListener('change', (e) => {
                    const target = e.target as HTMLSelectElement;
                    const filterValue = target.value || '';

                    this.applyFilter(filterValue);
                });
            }

            // 排序按钮事件
            if (this.$.sortButton) {
                this.$.sortButton.addEventListener('click', () => {
                    this.toggleSort();
                });
            }

            // 主要节点开关事件
            if (this.$.mainNodesToggle) {
                const toggleElement = this.$.mainNodesToggle as HTMLInputElement;

                toggleElement.addEventListener('change', (e: Event) => {
                    const target = e.target as HTMLInputElement;

                    panelData.showMainNodesOnly = target.checked;
                    this.updateNodeList();
                });
            } else {
                console.error('[prefab-analysis] 未找到 #main-nodes-toggle 元素');
            }

            // 初始化 Type 列标题排序功能
            this.initTypeHeaderSort();
        },
        /**
         * 初始化 Type 列标题排序功能
         */
        initTypeHeaderSort(): void {
            // 移除旧的监听器（如果存在）
            const oldHeader = document.querySelector('.node-type .header-text.sortable') as HTMLElement;

            if (oldHeader?.dataset.hasListener) {
                // 已经添加过监听器，跳过
                return;
            }

            // 延迟执行，确保 DOM 已渲染
            setTimeout(() => {
                const typeHeader = document.querySelector('.node-type .header-text.sortable') as HTMLElement;

                if (typeHeader && !typeHeader.dataset.hasListener) {
                    typeHeader.addEventListener('click', (e) => {
                        e.stopPropagation();
                        this.toggleSort();
                    });

                    typeHeader.style.cursor = 'pointer';
                    typeHeader.dataset.hasListener = 'true';
                }
            }, 100);
        },
    },
    ready() {
        if (this.$.app) {
            this.$.app.innerHTML = '未选中 Prefab';
        }
        
        if (this.$.nodeList) {
            this.$.nodeList.innerHTML = '<div class="node-item">暂无节点</div>';
        }
        
        // 初始化左右分栏拖拽功能
        this.initPanelDividerDrag();
        
        // 初始化列宽拖拽功能
        this.initColumnResize();
        
        // 初始化筛选和排序功能
        this.initFilterAndSort();
        
        // 更新排序按钮文本
        this.updateSortButtonText();
        
        // 更新删除按钮状态
        this.updateDeleteButtonState();
        
        // 初始化删除按钮事件监听器
        if (this.$.deleteSelectedButton) {
            this.$.deleteSelectedButton.addEventListener('click', () => {
                this.deleteSelectedNodes();
            });
        }

        // 初始化保存按钮事件监听器
        if (this.$.saveButton) {
            this.$.saveButton.addEventListener('click', () => {
                this.savePrefab(false);
            });
        }

        // 初始化另存按钮事件监听器
        if (this.$.saveAsButton) {
            this.$.saveAsButton.addEventListener('click', () => {
                this.savePrefab(true);
            });
        }
        
        // 绑定方法并保存引用，以便后续移除监听器
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore - onPrefabSelected 方法在 methods 中定义，运行时可用
        prefabSelectedHandler = this.onPrefabSelected.bind(this);
        
        // 监听 prefab 选择事件
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-ignore - addBroadcastListener 方法在运行时存在，但类型定义中可能缺失
        if (Editor.Message.addBroadcastListener && prefabSelectedHandler) {
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore
            Editor.Message.addBroadcastListener(`${packageJSON.name}:prefab-selected`, prefabSelectedHandler);
        }
    },
    beforeClose() {
        // 移除监听器
        if (prefabSelectedHandler) {
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore - removeBroadcastListener 方法在运行时存在，但类型定义中可能缺失
            if (Editor.Message.removeBroadcastListener) {
                // eslint-disable-next-line @typescript-eslint/ban-ts-comment
                // @ts-ignore
                Editor.Message.removeBroadcastListener(`${packageJSON.name}:prefab-selected`, prefabSelectedHandler);
            }
        }
    },
    close() {
        // 移除监听器
        if (prefabSelectedHandler) {
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore - removeBroadcastListener 方法在运行时存在，但类型定义中可能缺失
            if (Editor.Message.removeBroadcastListener) {
                // eslint-disable-next-line @typescript-eslint/ban-ts-comment
                // @ts-ignore
                Editor.Message.removeBroadcastListener(`${packageJSON.name}:prefab-selected`, prefabSelectedHandler);
            }

            prefabSelectedHandler = null;
        }
    },
});
