import { exit } from "process";
import FileList from "./FileList.js";
import * as fs from "fs";

export default class I18Texture {
    
    _artPath = "";
    _projectAssetsPath = "";
    _defaultLanguageDir = "";
    _languageDir = "";
    _allLanguageDir = [];
    _i18CmptType = "";
    _i18CmptData = null;

    _artFl = null;
    _projectFl = null;
    _artNames = [];
    _filesInProject = [];
    _repeatFileInProject = null;
    _notFoundArtResInProject = [];
    _mapUUID = null;
    _editorFilePath = [];

    constructor(){
    }

    init( data ){
        const self = this;

        self._artFl = null;
        self._projectFl = null;
        self._artNames = [];
        self._filesInProject = [];
        self._repeatFileInProject = null;
        self._notFoundArtResInProject = [];
        self._mapUUID = new Map();
        self._editorFilePath = [];

        self._artPath = data.artPath;
        self._projectAssetsPath = data.assetsPath;
        self._defaultLanguageDir = data.defaultDir;
        self._languageDir = data.languageDir;
        self._allLanguageDir = data.allLanguageDir;
        self._i18CmptType = data.i18CmptType;
        self._i18CmptData = data.i18CmptData;
    }

    //深度拷贝
    clone(obj) {
        if (typeof obj !== 'object' || obj === null) {
            return obj;
        }

        let newObj = Array.isArray(obj)? [] : {};

        for (let key in obj) {
            newObj[key] = this.clone(obj[key]);
        }

        return newObj;
    }

    //读取json
    readJsonInFile( fileName ){
        const str = fs.readFileSync( fileName, "utf8" );
        let jsonData = null;
        try {        
            jsonData = JSON.parse( str );
        } catch (error) {
            console.error( "JSON parse meta file error, file:" + fileName );
            console.error( error );
            return null;
        }
        return jsonData;
    }

    //获取meta文件里的uuid
    getUUID( metaFileName ){
        const jsonData = this.readJsonInFile( metaFileName );
        return jsonData ? jsonData["uuid"] : null;
    }

    //通过文件名找到project中文件位置	并记录位置
    findArtInProject(){
        console.log( "\n----------find art file in project----------" );
        const self = this;
        for( const name of self._artNames ){
            if( !self.checkRepeatNameInProject( name ) ){
                exit( 1 );
            }
            const pathTmp = self._projectFl.getFilePath( name );
            if( pathTmp != null ){
                self._filesInProject.push( pathTmp + "/" + name );
            }else{
                //console.warn( `not found file in project ! path:${self._artFl.getFilePath(name)} file:${name}` );
                const path1 = self._artFl.getFilePath(name);
                const fullPath = path1 ? (self._artFl.getFilePath( name ) + "/" + name) : name;
                self._notFoundArtResInProject.push( fullPath );
            }
        }
    }

    //记录uuid
    recordUUIDInProject(){
        console.log( "\n----------record uuid in project----------" );
        const self = this;
        self._filesInProject.forEach( (fileName)=>{
            const metaName = fileName + ".meta";
        
            const uuid = self.getUUID( self._projectAssetsPath + metaName );
            if( uuid != null ){
                self._mapUUID.set( uuid, fileName );
            }else{
                console.error( "no uuid ! file:" + self._projectAssetsPath + metaName );
            }
        } );
    }

    //找到所有的scene和prefab
    findAllScneAndPrefab(){
        const self = this;
        const allScene = self._projectFl.filterByType( "scene" );
        for( let name of allScene ){
            self._editorFilePath.push( self._projectAssetsPath + name );
        }
        const allPrefab = self._projectFl.filterByType( "prefab" );
        for( let name of allPrefab ){
            self._editorFilePath.push( self._projectAssetsPath + name );
        }
    }

    //找到Node的I18组件
    getI18CmptByNodeId( jsonData, nodeId ){
        const self = this;
        for( const data of jsonData ){
            if( data[ "__type__" ] != self._i18CmptType ) continue;
            if( data[ "node" ] == null ) continue;
            if( data[ "node" ][ "__id__" ] == nodeId )
                return data;
        }
        return null;
    }

    //修改所有的scene和prefab
    modifyProjectFile() {
        console.log( "\n----------modify scene and prefab in project----------" );

        const self = this;
        for( const scneOrPrefabPath of self._editorFilePath ){
            const jsonData = self.readJsonInFile( scneOrPrefabPath );
            if( jsonData == null ) continue;
        
            let bModify = false;
            const arrAddCmpt = [];
            for( const data of jsonData ){
                if( data[ "node" ] == null ) continue;
                if( data[ "node" ][ "__id__" ] == null ) continue;

                let uuid = null;
                if( data[ "__type__" ] == "cc.Sprite" ){
                    if( data["_spriteFrame"] == null || data["_spriteFrame"]["__uuid__"] == null ) continue;
                    const strUUID = data["_spriteFrame"]["__uuid__"];
                    uuid = strUUID.substr( 0, strUUID.lastIndexOf( "@" ) );
                }else if( data["__type__"] == "sp.Skeleton" ){
                    if( data["_skeletonData"] == null || data["_skeletonData"]["__uuid__"] == null ) continue;
                    uuid = data["_skeletonData"]["__uuid__"];
                }
                if( uuid == null ) continue;

                const resPath = self._mapUUID.get( uuid );
                if( resPath == null ) continue;

                let recordPath = "";
                if( resPath.indexOf( self._defaultLanguageDir ) == -1 ){
                    recordPath = resPath;
                }else{
                    recordPath = resPath.replace( self._defaultLanguageDir, "" );
                }
                recordPath = recordPath.substring( 0, recordPath.lastIndexOf(".") );

                const nodeId = data["node"][ "__id__" ];
                const i18Cmp = self.getI18CmptByNodeId( jsonData, nodeId );
                if( i18Cmp != null ){
                    if( i18Cmp[ "_key" ] != recordPath ){
                        i18Cmp[ "_key" ] = recordPath;
                        bModify = true;
                    }
                }else{
                    const cmptInfo = self.clone( self._i18CmptData );
                    cmptInfo["node"][ "__id__" ] = nodeId;
                    cmptInfo[ "id" ] = 0;
                    cmptInfo[ "_key" ] = recordPath;
                    arrAddCmpt.push( cmptInfo );
                    bModify = true;
                }
            }

            let cmptId = jsonData.length;
            for( const info of arrAddCmpt ){
                const nodeId = info["node"][ "__id__" ];
                const nodeInfo = jsonData[nodeId];
                if( nodeInfo == null ) continue;
                if( nodeInfo["_components"] == null )
                    nodeInfo[ "_components" ] = [];
                const components = nodeInfo[ "_components" ];

                let bExist = false;
                for( const infoTmp of components ){
                    if( infoTmp["__id__"] == cmptId ){
                        bExist = true;
                        break;
                    }
                }
                if( bExist ){
                    console.error( `I18Cmpt is exist already, scene: ${scneOrPrefabPath} nodeId: ${nodeId}` );
                    continue;
                }

                components.push( { "__id__": cmptId } );
                jsonData.push( info );
                ++cmptId;
            }

            if( bModify ){
                fs.writeFileSync( scneOrPrefabPath, JSON.stringify(jsonData, null, 2) );
                console.warn( "modify file:" + scneOrPrefabPath );
            }
        }
    }

    //创建文件夹
    createDir( dir ){
        if (!fs.existsSync(dir)) {
            fs.mkdirSync(dir, { recursive: true });
        }
    }

    //循环创建文件夹
    createDirDeep( basePath, arrDir ){
        let dir = basePath;
        for( const dirTmp of arrDir ){
            dir += dirTmp + "/";
            this.createDir( dir );
        }
    }

    //拷贝文件
    copyFile( src, dst ){
        fs.writeFileSync(dst, fs.readFileSync(src));
    }

    //将美术资源移动到多语言目录
    moveArtFileToProject(){
        console.log( "\n----------move art res to project----------" );
        const self = this;
        for( const fileName of self._filesInProject ){
            let enFileName = fileName;
            if( enFileName.indexOf(self._defaultLanguageDir) == 0 ){
                enFileName = enFileName.substr( self._defaultLanguageDir.length );
            }
            const arrDir = enFileName.split( "/" );
            if( arrDir.length > 1 ){
                arrDir.splice( arrDir.length - 1, 1 );
                self.createDirDeep( self._projectAssetsPath + self._languageDir, arrDir );
            }

            const shortName = fileName.substr( fileName.lastIndexOf("/") + 1 );
            const artDir = self._artFl.getFilePath( shortName );
            if( artDir != null ){
                self.copyFile( self._artPath + artDir + "/" + shortName, self._projectAssetsPath + self._languageDir + enFileName );
            }else{
                self.copyFile( self._artPath + shortName, self._projectAssetsPath + self._languageDir + enFileName );
            }
        }
    }

    //移动已存在的文件到中文目录
    moveFileInProject(){
        console.log( "\n----------move project res to cn----------" );
        const self = this;
        for( const fileName of self._filesInProject ){
            if( fileName.indexOf(self._defaultLanguageDir) == 0 ) continue;

            const arrDir = fileName.split( "/" );
            if( arrDir.length > 1 ){
                arrDir.splice( arrDir.length - 1, 1 );
                self.createDirDeep( self._projectAssetsPath + self._defaultLanguageDir, arrDir );
            }
            
            self.copyFile( self._projectAssetsPath + fileName, self._projectAssetsPath + self._defaultLanguageDir + fileName );
            self.copyFile( self._projectAssetsPath + fileName + ".meta", self._projectAssetsPath + self._defaultLanguageDir + fileName + ".meta" );

            fs.unlinkSync( self._projectAssetsPath + fileName );
            fs.unlinkSync( self._projectAssetsPath + fileName + ".meta" );
        }
    }

    //输出未处理的资源
    printNoDealRes(){
        const self = this
        if( self._notFoundArtResInProject.length == 0 ) return;

        console.log( "\n----------no deal art res----------" );
        for( const name of self._notFoundArtResInProject ){
            console.log( name );
        }
    }

    //检查名字重复
    checkFileNameSingle( fl ){
        console.log( "\n----------check repeat name----------" );
        const self = this;
        const mapRepeat = fl.getAllRepeatPath();
        if( mapRepeat.size > 0 ){
            console.log( "repeat path" );
            mapRepeat.forEach((arrPath, name)=>{
                arrPath.forEach((pathTmp)=>{
                    console.warn( pathTmp );
                });
                console.log( "" );
            });
            return false;
        }
        return true;
    }

    //检查项目中名字重复
    checkRepeatNameInProject( name ){
        const self = this;
        const arrPath = self._repeatFileInProject.get( name );
        if( arrPath == null ) return true;
        if( arrPath.length <= 1 ) return true;

        //console.warn( "use repeat file name in project" );
        arrPath.forEach((pathTmp)=>{
            console.warn( pathTmp );
        });
        console.log( "" );
        return false;
    }

    //检查美术资源在项目中有多个文件
    checkMultiFileInProject(){
        console.log( "\n----------check multi file in project----------" );
        const self = this
        let bRepeat = false;
        for( const name of self._artNames ){
            if( !self.checkRepeatNameInProject( name ) ){
                bRepeat = true;
            }
        }

        if( bRepeat ){
            console.error( "pls modify file name" );
        }

        return !bRepeat;
    }

    run(){
        console.log( "=========start handle texture=========" );

        console.log( "----------init art dir----------" );
        const self = this;
        self._artFl = new FileList();
        self._artFl.initData( self._artPath );
        if( !self.checkFileNameSingle( self._artFl ) ) return;

        console.log( "\n----------init project dir----------" );
        self._projectFl = new FileList();
        self._projectFl.initData( self._projectAssetsPath, self._allLanguageDir );
        self._repeatFileInProject = self._projectFl.getAllRepeatPath();

        console.log( "\n----------get art files name----------" );
        self._artNames = self._artFl.getShortName();

        if( !self.checkMultiFileInProject() ) return;

        self.findArtInProject();
        self.recordUUIDInProject();

        self.findAllScneAndPrefab();
        self.modifyProjectFile();

        self.moveArtFileToProject();
        self.moveFileInProject();

        self.printNoDealRes();
                
        console.log( "=========handle texture finished=========\n\n" );
    }

}
