/**
 * BoardComp 组件的描述
 */

import { _decorator, tween } from 'cc';
import { CompBase, ICompBase } from '../Base/CompBase';
import EventMgr from '../manager/EventMgr';
import { EventConst } from './EventConst';
import { IReelCompArgs, ReelComp } from './ReelComp';
const { ccclass, property } = _decorator;

export interface IBoardCompArgs extends ICompBase {

}

@ccclass('BoardComp')
export class BoardComp extends CompBase {

    private _reelCompArr: Array<ReelComp> = [];
    private _rollCount: number = 0;
    private _rollEndCount: number = 0;
    private _rolling: boolean = false;

    /**
    * 在节点首次激活时触发，比如所在的场景被载入，或者所在节点被激活的情况下。
    * 在 onLoad 阶段，保证了你可以获取到场景中的其他节点，以及节点关联的资源数据。
    * 生命周期函数顺序：onLoad => onEnable => start => update => lateUpdate
    */
    protected onLoad(): void {
        super.onLoad();
        this.node.children.forEach((n, idx) => {
            const param: IReelCompArgs = {
                idx: idx
            }
            const reelComp = n.getOrAddComponent(ReelComp)
            reelComp.init(param)
            this._reelCompArr.push(reelComp)
        })
    }


    public resetSymbols(){
        this._reelCompArr.forEach(v=>{
            v.resetSymbol()
        })
    }

    public startRoll() {


        this._rollCount = 0;
        this._rollEndCount = 0;
        this._rolling = true;
        const delay = 0.1;
        const reelLength = this._reelCompArr.length;
        for (let i = 0; i < reelLength; i++) {
            const reelComp = this._reelCompArr[i];
            tween(reelComp)
                .delay(delay * i)
                .call(() => {
                    reelComp.startRoll();
                    this._rollCount++;
                })
                .start()
        }
    }

    public stopRoll() {
        // const resultArr = this.getRandomResult();
        const resultArr = [
            [1, 2, 3, 4],
            [5, 6, 7, 8],
            [9, 10, 11, 12]
        ]
        for (let i = 0; i < this._reelCompArr.length; i++) {
            const reelComp = this._reelCompArr[i];
            if (!reelComp) {
                console.log('reelComp is null', i)
                break;
            }
            const _symbolArr = resultArr[this._reelCompArr.length - 1 - i];
            if (!_symbolArr) {
                continue
            }
            const delay = 0.1;
            tween(reelComp)
                .delay(delay * i)
                .call(() => {
                    reelComp.stopRoll(_symbolArr);
                })
                .start();

        }
    }

    private getRandomResult(): Array<Array<number>> {
        const result = []
        for (let i = 0; i < 3; i++) {
            const arr = []
            for (let j = 0; j < 4; j++) {
                arr.push(Math.floor(Math.random() * 10))
            }
            result.push(arr)
        }
        return result;
    }


    /**
    * 当组件的 enabled 属性从 false 变为 true 时，或者所在节点的 active 属性从 false 变为 true 时，会激活 onEnable 回调。
    * 倘若节点第一次被创建且 enabled 为 true，则会在 onLoad 之后，start 之前被调用。
    */
    protected onEnable(): void {
        super.onEnable();
    }

    /**
    * 当组件的 enabled 属性从 true 变为 false 时，或者所在节点的 active 属性从 true 变为 false 时，会激活 onDisable 回调。
    */
    protected onDisable(): void {
        super.onDisable();
    }

    /**
    * 在组件第一次激活前，也就是第一次执行 update 之前触发。
    */
    protected start(): void {
        super.start();
    }

    /**
    * 当组件或者所在节点调用了 destroy()，则会调用 onDestroy 回调，并在当帧结束时统一回收组件。
    */
    protected onDestroy(): void {
        super.onDestroy();
    }

    /**
 * 初始化参数，由 UIMgr 在打开界面时调用
 * @param uiArgs 
 */
    initArgs(uiArgs?: IBoardCompArgs): void {
        super.initArgs(uiArgs);
    };

    /**
     * 使用 SDK.msgMgr.addMessage 添加所有的监听事件。
     * onEnable 时自动调用，组件失效时自动移除通过 SDK.msgMgr.addMessage 添加的所有事件。
     * 注意：不是用 SDK.msgMgr 添加的事件，需要重载父类 removeAllMsgs 方法，并自己移除。
     */
    registerEvents(): void {
        super.registerEvents();
        EventMgr.on(EventConst.ROLL_STOP_DONE, this.onRollStopDone, this);

    }

    private onRollStopDone(param: number) {
        console.log(param);
        this._rollEndCount++;
        if (this._rollEndCount == this._rollCount) {
            this._rolling = false;
            //所以滚轴转动结束
            EventMgr.emit(EventConst.ALL_ROLL_DONE)
        }
    }

    protected update(dt: number): void {
        if (!this._rolling) return;
        this._reelCompArr.forEach((v) => {
            v.updateTick(dt);
        })
    }


    public eliminateSymbols(winRouteArr: Array<Array<number>>) {
        return new Promise(async (resolve, reject) => {
            let eli_Promises: Promise<boolean>[] = [];
            winRouteArr.forEach((winRoute, idx) => {
                const reelComp = this._reelCompArr[idx];
                if (!reelComp) return;
                const promise = reelComp.eliSymbols(winRoute[0])
                eli_Promises.push(promise)
            })
            await Promise.all(eli_Promises)
            resolve(true);
        })
    }

    public async dropSymbols() {
        let self = this;
        return new Promise(async (resolve, reject) => {
            let total = 0;
            let count = 0;
            for (let i = 0; i < this._reelCompArr.length; i++) {
                const reelComp = this._reelCompArr[i];
                if (!reelComp) return;
                const dropSymbolNum = Math.floor(Math.random() * 3) + 1;
                total++;
                reelComp.fillSymbols();
                reelComp.dropSymbols(dropSymbolNum).then(() => {
                    count++;
                    if (count == total) {
                        resolve(true);
                    }
                })
            }

        })
    }
}
