/**
 * ReelComp 组件的描述
 */

import { _decorator, Node, tween, UITransform, v3, Vec3 } from 'cc';
import { CompBase, ICompBase } from '../Base/CompBase';
import { quickBind } from '../extension/Decorator';
import EventMgr from '../manager/EventMgr';
import { EventConst } from './EventConst';
import { SymbolComp } from './SymbolComp';
const { ccclass, property } = _decorator;

export interface IReelCompArgs extends ICompBase {
    idx: number
}

enum RollState {
    STOP,
    ROLLING,
    RESULT,
}

const SYMBOL_SIZE = {
    width: 100,
    height: 100
}

const OffsetY = 0;


@ccclass('ReelComp')
export class ReelComp extends CompBase {
    @quickBind('symbolContainer', Node)
    private _symbolContainer!: Node;

    private _targetSymbol?: SymbolComp;
    private _symbolCompArr: SymbolComp[] = [];
    private _reelIdx: number = 0;
    private _speed: number = 0;
    private _rollState: RollState = RollState.STOP;
    private _initSpeed: number = 1000;
    private _rollSpeed: number = 2500;
    private _uiTrans!: UITransform
    private _resultSymbolIdx = 0;
    private _symbolNumArr: number[] = [];
    private __dropStartIdx: number = 0


    /**
    * 在节点首次激活时触发，比如所在的场景被载入，或者所在节点被激活的情况下。
    * 在 onLoad 阶段，保证了你可以获取到场景中的其他节点，以及节点关联的资源数据。
    * 生命周期函数顺序：onLoad => onEnable => start => update => lateUpdate
    */
    protected onLoad(): void {
        super.onLoad();

    }


    public init(args: IReelCompArgs): void {
        this._reelIdx = args.idx;
        this._symbolContainer.children.forEach((n) => {
            const symbolComp = n.getOrAddComponent(SymbolComp)
            this._symbolCompArr.push(symbolComp)
        })
        this._uiTrans = this.node.getComponent(UITransform);
    }



    /**
    * 当组件的 enabled 属性从 false 变为 true 时，或者所在节点的 active 属性从 false 变为 true 时，会激活 onEnable 回调。
    * 倘若节点第一次被创建且 enabled 为 true，则会在 onLoad 之后，start 之前被调用。
    */
    protected onEnable(): void {
        super.onEnable();
    }

    /**
    * 当组件的 enabled 属性从 true 变为 false 时，或者所在节点的 active 属性从 true 变为 false 时，会激活 onDisable 回调。
    */
    protected onDisable(): void {
        super.onDisable();
    }

    /**
    * 在组件第一次激活前，也就是第一次执行 update 之前触发。
    */
    protected start(): void {
        super.start();
    }

    /**
    * 当组件或者所在节点调用了 destroy()，则会调用 onDestroy 回调，并在当帧结束时统一回收组件。
    */
    protected onDestroy(): void {
        super.onDestroy();
    }

    /**
 * 初始化参数，由 UIMgr 在打开界面时调用
 * @param uiArgs 
 */
    initArgs(uiArgs?: IReelCompArgs): void {
        super.initArgs(uiArgs);
    };

    /**
     * 使用 SDK.msgMgr.addMessage 添加所有的监听事件。
     * onEnable 时自动调用，组件失效时自动移除通过 SDK.msgMgr.addMessage 添加的所有事件。
     * 注意：不是用 SDK.msgMgr 添加的事件，需要重载父类 removeAllMsgs 方法，并自己移除。
     */
    registerEvents(): void {
        super.registerEvents();
    }

    public resetSymbol() {
        this._symbolCompArr.forEach((symbolComp) => {
            symbolComp.isEmpty = false;
        })
    }



    public startRoll() {
        this._speed = 1000;
        this._rollState = RollState.ROLLING;
        this.speedUp();
    }

    public stopRoll(symbolNumArr: number[]) {
        this._symbolNumArr = symbolNumArr;
        if (this._rollState == RollState.ROLLING) {
            this._stopRoll();
        }
    }

    private _stopRoll() {
        this._rollState = RollState.RESULT;
        this._resultSymbolIdx = 0;
        this._targetSymbol = undefined;
    }


    private speedUp() {
        let speedObj = { speed: this._initSpeed }
        let rollSpeed = this._rollSpeed
        tween(speedObj)
            .to(0.3, { speed: rollSpeed },
                {
                    //换箭头函数写
                    progress: (start, end, current, ratio) => {
                        this._speed = start - Math.ceil((start - end) * ratio);
                        return this._speed;
                    }
                }
            )
            .start();

    }

    updateTick(dt) {
        const state = this._rollState//获取当前状态
        if (state == RollState.STOP) return;//如果是停止状态，直接返回
        dt = Math.min(dt, 0.03);//限制更新间隔，防止帧率太高导致速度过快
        let delta = this._speed * dt;//计算本次滚动的距离
        const pos = this._symbolContainer.position;//获取当前容器的位置
        delta = Math.min(delta, SYMBOL_SIZE.height);//获取最小滚动距离，防止超出边界
        pos.set(pos.x, pos.y - delta, pos.z);//计算新的位置
        this._symbolContainer.setPosition(pos);//设置新的位置
        if (state == RollState.ROLLING) {//如果是滚动状态，则继续滚动
            this._updateFirstSymbol();
        } else if (state == RollState.RESULT) {//如果是结果状态，则更新符号并判断是否停止
            //结果符号已经设置完成了
            if (this._resultSymbolIdx >= this._symbolNumArr.length) {
                this._updateFirstSymbol();
                //如果第一个设置的元素已经从最末尾到底部了，则停止滚动并重置位置
                const symbolComp = this._symbolCompArr[0];
                if (symbolComp == this._targetSymbol) {
                    this._rollState = RollState.STOP;
                    this._resetPos();
                }
            } else {
                this._updateResultSymbol();
            }
        }
    }

    private _resetPos() {
        this._symbolCompArr.forEach((comp, idx) => {
            this._updatePos(comp, idx);
        })
        this._symbolContainer.setPosition(v3(0, 0, 0));
        // this._rollState = RollState.STOP;
        EventMgr.emit(EventConst.ROLL_STOP_DONE, this._reelIdx);
    }

    protected _updateResultSymbol() {
        const comp = this._symbolCompArr[0];
        const isOut = this._checkOut(comp);
        if (!isOut) return;
        this._symbolCompArr.shift();
        this._symbolCompArr.push(comp);

        if (this._resultSymbolIdx == 0) {//从0开始设置模板符号
            let idx = this._symbolCompArr.length - 1;//获取最后一个符号的索引
            this._targetSymbol = this._symbolCompArr[idx];//获取目标符号的组件
            const symbolComp = this._symbolCompArr[idx];//获取最后一个符号的组件，其实就是上面的目标组件
            //设置结果【1,2,3,4】。将1赋值给最后，从上往下滚动也及时第一个元素在最下面
            const symbolNum = this._symbolNumArr[this._resultSymbolIdx];
            if (symbolComp && symbolNum) {
                symbolComp.updateSymbolLbl(symbolNum);
            }
            this._updatePos(symbolComp, idx);
            this._resultSymbolIdx++;//更新结果符号索引
        } else {
            //设置结果符号的数字，开始设置 2 ,1已经在resultSymbolIdx==0设置了(【1,2,3,4】)
            const symbolNum = this._symbolNumArr[this._resultSymbolIdx];
            comp.updateSymbolLbl(symbolNum);
            //已经出了屏幕out，设置到最末尾的位置
            const symIdx = this._symbolCompArr.length - 1;
            this._updatePos(comp, symIdx);
            this._resultSymbolIdx++;
        }
    }



    private _updateFirstSymbol() {
        const comp = this._symbolCompArr[0];
        const isOut = this._checkOut(comp);
        if (!isOut) return;
        this._symbolCompArr.shift();
        this._symbolCompArr.push(comp);
        const symbolIdx = this._symbolCompArr.length - 1;
        const random_num = Math.floor(Math.random() * 100);
        comp.updateSymbolLbl(random_num);
        comp.node.setSiblingIndex(this._symbolCompArr.length - 1 - symbolIdx);

        // 更新符号位置
        this._updatePos(comp, symbolIdx);
    }


    private _updatePos(comp: SymbolComp, idx: number) {
        let pox_y = 50;
        let endPos = v3(0, 0, 0);
        const preComp = this._symbolCompArr[idx - 1];
        if (preComp) {
            const pos = preComp.node.getPosition();
            pos.y += SYMBOL_SIZE.height + OffsetY;
            endPos = pos;
        } else {
            const newPos = new Vec3(0, pox_y, 0)
            endPos = newPos;
        }
        comp.node.setPosition(endPos);
    }


    private _checkOut(comp: SymbolComp) {
        const pos = this._uiTrans.convertToWorldSpaceAR(Vec3.ZERO);
        let symbolPos = comp.getBoundaryWorldPos();
        return pos.y > symbolPos.y;
    }


    public async eliSymbols(idx: number): Promise<boolean> {
        return new Promise(async (resolve, reject) => {
            const symbolComp = this._symbolCompArr[idx];
            if (!symbolComp) return
            symbolComp.isEmpty = true
            resolve(true);
        })
    }

    public fillSymbols() {
        this.__dropStartIdx = 0;
        const topSymbol = this._symbolCompArr.pop();
        let moveNum = 0;
        let otherNum = 0;
        const rowNum = 4;
        //填充符号
        for (let idx = 0, endIdx = this._symbolCompArr.length; idx < endIdx; idx++) {
            const symbolComp = this._symbolCompArr[idx];
            if (symbolComp.isEmpty) {
                moveNum++;
                this._symbolCompArr.push(symbolComp);
                this._symbolCompArr.splice(idx, 1);
                idx--;
                endIdx--;
            } else {
                if (otherNum + moveNum == rowNum) {
                    break;
                }
                otherNum++;
                this.__dropStartIdx++;
                if (moveNum > 0) {
                    this.playDropAnim(symbolComp, moveNum);
                }
            }
        }
        this._symbolCompArr.push(topSymbol)

    }

    private async playDropAnim(SymbolComp: SymbolComp, moveNum: number, callBack?: Function) {
        let moveDis = 0;
        let moveV3 = v3(0, 0, 0);
        moveDis = SYMBOL_SIZE.height * moveNum;
        moveV3.y = -moveDis;
        SymbolComp.dropByPos(moveV3, moveDis / 1000, callBack)
    }

    public async dropSymbols(dropNumArr: number, callBack?: Function) {
        let moveNum = 1;
        // dropNumArr.forEach((v) => {
        //     moveNum += v;
        // })
        let count = 0;
        let dropCount = 0;
        for (let idx = this.__dropStartIdx, len = this._symbolCompArr.length; idx < len; idx++) {
            const symbolComp = this._symbolCompArr[idx];
            const dropNum = dropNumArr;
            if (dropNum) {
                symbolComp.updateSymbolLbl(dropNum)
            }
            if (symbolComp.isEmpty) {
                const randomNum = Math.floor(Math.random() * 10000);
                symbolComp.updateSymbolLbl(randomNum);
            }
            if (idx == this.__dropStartIdx) {
                symbolComp.node.position = this.getOutSymbolPos();
            }
            count++;
            await this.playDropAnim(symbolComp, moveNum, () => {
                dropCount++;
                if (count == dropCount) {
                    callBack && callBack();
                }
            })
        }
    }

    private getOutSymbolPos() {
        const rowCnt = 4;
        const lastPos = rowCnt * SYMBOL_SIZE.height;
        console.log("finalPos", SYMBOL_SIZE.height/2 + lastPos)
        return v3(0, SYMBOL_SIZE.height/2 + lastPos, 0);
    }
}
