/**
 * GameLayer 组件的描述
 */

import { _decorator, Node } from 'cc';
import { CompBase, ICompBase } from '../Base/CompBase';
import { quickBind } from '../extension/Decorator';
import EventMgr from '../manager/EventMgr';
import { CocosUtil } from '../Utils/CocosUtil';
import { BoardComp } from './BoardComp';
import { EventConst } from './EventConst';
const { ccclass, property } = _decorator;

export interface IGameLayerArgs extends ICompBase {

}

@ccclass('GameLayer')
export class GameLayer extends CompBase {
    @quickBind('BoardComp', BoardComp, true)
    private _boardComp!: BoardComp;
    @quickBind('startBtn', Node)
    private _startBtn!: Node;
    @quickBind('stopBtn', Node)
    private _stopBtn!: Node;

    private _canClick: boolean = true;

    /**
    * 在节点首次激活时触发，比如所在的场景被载入，或者所在节点被激活的情况下。
    * 在 onLoad 阶段，保证了你可以获取到场景中的其他节点，以及节点关联的资源数据。
    * 生命周期函数顺序：onLoad => onEnable => start => update => lateUpdate
    */
    protected onLoad(): void {
        super.onLoad();
        this._startBtn.on(Node.EventType.TOUCH_END, this.onStartBtnClick, this);
        this._stopBtn.on(Node.EventType.TOUCH_END, this.onStopBtnClick, this);
    }


    private onStartBtnClick() {
        if (!this._canClick) return;
        this._boardComp.resetSymbols();
        this._boardComp.startRoll();
        this._canClick = false;
    }

    private onStopBtnClick() {
        if (this._canClick) return;
        this._boardComp.stopRoll();
        this._canClick = true;

    }

    /**
    * 当组件的 enabled 属性从 false 变为 true 时，或者所在节点的 active 属性从 false 变为 true 时，会激活 onEnable 回调。
    * 倘若节点第一次被创建且 enabled 为 true，则会在 onLoad 之后，start 之前被调用。
    */
    protected onEnable(): void {
        super.onEnable();
    }

    /**
    * 当组件的 enabled 属性从 true 变为 false 时，或者所在节点的 active 属性从 true 变为 false 时，会激活 onDisable 回调。
    */
    protected onDisable(): void {
        super.onDisable();
    }

    /**
    * 在组件第一次激活前，也就是第一次执行 update 之前触发。
    */
    protected start(): void {
        super.start();
    }

    /**
    * 当组件或者所在节点调用了 destroy()，则会调用 onDestroy 回调，并在当帧结束时统一回收组件。
    */
    protected onDestroy(): void {
        super.onDestroy();
    }

    /**
 * 初始化参数，由 UIMgr 在打开界面时调用
 * @param uiArgs 
 */
    initArgs(uiArgs?: IGameLayerArgs): void {
        super.initArgs(uiArgs);
    };

    /**
     * 使用 SDK.msgMgr.addMessage 添加所有的监听事件。
     * onEnable 时自动调用，组件失效时自动移除通过 SDK.msgMgr.addMessage 添加的所有事件。
     * 注意：不是用 SDK.msgMgr 添加的事件，需要重载父类 removeAllMsgs 方法，并自己移除。
     */

    registerEvents(): void {
        super.registerEvents();
        EventMgr.on(EventConst.ALL_ROLL_DONE, this.onAllRollDone, this);

    }


    private onAllRollDone() {
        console.log('游戏停止转动！一局游戏结束')
        this._checkWin();
    }

    private _checkWin() {
        const isWinThisEli = true;//此处为示意，实际逻辑需要根据游戏规则来判断是否胜利
        if (isWinThisEli) {
            this._winEvent();
        } else {
            this._loseEvent();
        }

    }

    private async _winEvent() {
        //测试数据,第一列第0个，第二列第1个，第三列第2个是胜利的格子位置
        const winRouterArr = [
            [0], [1], [2]
        ]
        await this._boardComp.eliminateSymbols(winRouterArr)
        await CocosUtil.waitByComp(2,this);
        await this._boardComp.dropSymbols()
    }

    private _loseEvent() {
        //如果开启了自动模式，接着开启下一轮接着转
        //清空当前盘面数据，为下一轮开始初始化必要的数据，例如转到第几轮了之类的
    }


}
