import { _decorator } from 'cc';
import { UIWindow } from '../../core/ui/windows/UIWindow';
import { peanut } from '../../core/Peanut';
import { LogUtils } from '../../core/utils/LogUtils';
import { decorator } from '../../core/decorators/Decorator';
import { PromiseManager } from '../../core/task/promise/PromiseManager';
import { IPromiseCancellable } from '../../core/task/promise/PromiseDefines';

// const { ccclass, property } = _decorator;

/**
 * PromiseManager 单元测试类
 */
@decorator.class({
    ui: {
        url: 'tests/promise/PromiseTest',
    },
})
export class PromiseTest extends UIWindow {
    private testManager: PromiseManager = new PromiseManager();
    private testResults: Map<string, boolean> = new Map();
    private testObj = { status: 'pending', value: 0, flag: false };

    async enter() {
        LogUtils.log('=== PromiseManager 单元测试开始 ===');

        try {
            await this.runAllTests();
            this.printResults();
        } catch (error) {
            LogUtils.error('测试执行失败:', error);
        }
    }

    /**
     * 运行所有测试
     */
    private async runAllTests(): Promise<void> {
        const tests = [
            this.testBasicPromiseCreation,
            this.testPromiseResolution,
            this.testPromiseRejection,
            this.testPromiseCancellation,
            this.testPromiseTimeout,
            this.testTimeoutResolve,
            this.testTimeoutReject,
            this.testOnCancelCallback,
            this.testWaitForChangeValue,
            this.testWaitForChangeFunction,
            this.testWaitForChangeAnyChange,
            this.testWaitForChangeTimeout,
            this.testBatchCancelAll,
            this.testBatchCancelCategory,
            this.testPromiseManagerStats,
            this.testErrorHandling,
            this.testEdgeCases,
            this.testMemoryCleanup,
        ];

        for (const test of tests) {
            try {
                await test.call(this);
                await this.delay(10); // 小延迟确保异步操作完成
            } catch (error) {
                LogUtils.error(`测试失败: ${test.name}`, error);
                this.testResults.set(test.name, false);
            }
        }
    }

    /**
     * 测试基本 Promise 创建
     */
    private async testBasicPromiseCreation(): Promise<void> {
        const promise = this.testManager.create<string>((resolve) => {
            resolve('test');
        });

        const result = await promise.promise;
        this.assert(result === 'test', 'testBasicPromiseCreation', 'Promise 应该正确解析值');
        this.assert(typeof promise.id === 'string', 'testBasicPromiseCreation', 'Promise 应该有 ID');
        this.assert(promise.category === 'CancellablePromise', 'testBasicPromiseCreation', '默认分类应该正确');
    }

    /**
     * 测试 Promise 解决
     */
    private async testPromiseResolution(): Promise<void> {
        let resolvedValue: number | undefined;

        const promise = this.testManager.create<number>((resolve) => {
            setTimeout(() => resolve(42), 50);
        });

        resolvedValue = await promise.promise;
        this.assert(resolvedValue === 42, 'testPromiseResolution', 'Promise 应该正确解决');
    }

    /**
     * 测试 Promise 拒绝
     */
    private async testPromiseRejection(): Promise<void> {
        const promise = this.testManager.create<never>((_, reject) => {
            setTimeout(() => reject(new Error('test error')), 50);
        });

        try {
            await promise.promise;
            this.assert(false, 'testPromiseRejection', 'Promise 应该被拒绝');
        } catch (error) {
            this.assert(error.message === 'test error', 'testPromiseRejection', '错误信息应该正确');
        }
    }

    /**
     * 测试 Promise 取消
     */
    private async testPromiseCancellation(): Promise<void> {
        const promise = this.testManager.create<string>((resolve) => {
            setTimeout(() => resolve('should not reach'), 1000);
        });

        // 取消 Promise
        setTimeout(() => promise.cancel('test cancel'), 10);

        try {
            await promise.promise;
            this.assert(false, 'testPromiseCancellation', 'Promise 应该被取消');
        } catch (error) {
            this.assert(promise.isCancelled, 'testPromiseCancellation', 'Promise 应该标记为已取消');
        }
    }

    /**
     * 测试超时功能
     */
    private async testPromiseTimeout(): Promise<void> {
        const promise = this.testManager.create<string>(
            (_resolve) => {
                // 故意不解决，让它超时
            },
            {
                timeout: 100,
                text: 'timeout-test',
            },
        );

        try {
            await promise.promise;
            this.assert(false, 'testPromiseTimeout', 'Promise 应该超时');
        } catch (error) {
            this.assert(error.message.includes('超时'), 'testPromiseTimeout', '应该是超时错误');
        }
    }

    /**
     * 测试超时时解决
     */
    private async testTimeoutResolve(): Promise<void> {
        const promise = this.testManager.create<string>(
            (_resolve) => {
                // 故意不解决
            },
            {
                timeout: 100,
                timeoutAction: 'resolve',
                timeoutValue: 'timeout-resolved',
            },
        );

        const result = await promise.promise;
        this.assert(result === 'timeout-resolved', 'testTimeoutResolve', '超时应该解决为指定值');
    }

    /**
     * 测试超时时拒绝
     */
    private async testTimeoutReject(): Promise<void> {
        const promise = this.testManager.create<string>(
            (_resolve) => {
                // 故意不解决
            },
            {
                timeout: 100,
                timeoutAction: 'reject',
                timeoutReason: new Error('custom timeout error'),
            },
        );

        try {
            await promise.promise;
            this.assert(false, 'testTimeoutReject', 'Promise 应该超时拒绝');
        } catch (error) {
            this.assert(error.message === 'custom timeout error', 'testTimeoutReject', '应该是自定义错误');
        }
    }

    /**
     * 测试 onCancel 回调
     */
    private async testOnCancelCallback(): Promise<void> {
        let cancelCallbackExecuted = false;

        const promise = this.testManager.create<string>(
            (resolve) => {
                setTimeout(() => resolve('test'), 1000);
            },
            {
                onCancel: () => {
                    cancelCallbackExecuted = true;
                },
            },
        );

        setTimeout(() => promise.cancel(), 10);

        try {
            await promise.promise;
        } catch (error) {
            // 预期会抛出异常
        }

        this.assert(cancelCallbackExecuted, 'testOnCancelCallback', 'onCancel 回调应该被执行');
    }

    /**
     * 测试 waitForChange - 等待特定值
     */
    private async testWaitForChangeValue(): Promise<void> {
        this.testObj.value = 0;

        const waitPromise = this.testManager.waitForChange(this.testObj, 'value', {
            expectedValue: 100,
        });

        setTimeout(() => {
            this.testObj.value = 50;
        }, 10);

        setTimeout(() => {
            this.testObj.value = 100; // 触发条件
        }, 20);

        await waitPromise;
        this.assert(this.testObj.value === 100, 'testWaitForChangeValue', '应该等待到指定值');
    }

    /**
     * 测试 waitForChange - 等待函数返回 true
     */
    private async testWaitForChangeFunction(): Promise<void> {
        this.testObj.flag = false;

        const waitPromise = this.testManager.waitForChange(this.testObj, 'flag', {
            expectedValue: () => this.testObj.flag === true && this.testObj.value > 50,
        });

        setTimeout(() => {
            this.testObj.flag = true;
            this.testObj.value = 30; // 还不满足条件
        }, 10);

        setTimeout(() => {
            this.testObj.value = 60; // 满足条件
        }, 20);

        await waitPromise;
        this.assert(this.testObj.flag && this.testObj.value > 50, 'testWaitForChangeFunction', '应该等待函数条件满足');
    }

    /**
     * 测试 waitForChange - 等待任何变化
     */
    private async testWaitForChangeAnyChange(): Promise<void> {
        this.testObj.status = 'initial';

        const waitPromise = this.testManager.waitForChange(this.testObj, 'status');

        setTimeout(() => {
            this.testObj.status = 'changed';
        }, 10);

        await waitPromise;
        this.assert(this.testObj.status === 'changed', 'testWaitForChangeAnyChange', '应该检测到任何变化');
    }

    /**
     * 测试 waitForChange 超时
     */
    private async testWaitForChangeTimeout(): Promise<void> {
        this.testObj.value = 0;

        try {
            await this.testManager.waitForChange(this.testObj, 'value', {
                expectedValue: 999,
                timeout: 100,
            });
            this.assert(false, 'testWaitForChangeTimeout', 'waitForChange 应该超时');
        } catch (error) {
            this.assert(error.message.includes('超时'), 'testWaitForChangeTimeout', '应该是超时错误');
        }
    }

    /**
     * 测试批量取消所有 Promise
     */
    private async testBatchCancelAll(): Promise<void> {
        const promises: IPromiseCancellable<string>[] = [];

        // 创建多个 Promise
        for (let i = 0; i < 3; i++) {
            promises.push(
                this.testManager.create<string>((resolve) => {
                    setTimeout(() => resolve(`test-${i}`), 1000);
                }),
            );
        }

        const initialCount = this.testManager.getActivePromiseIds().length;

        // 批量取消
        await this.testManager.cancelAll();

        const finalCount = this.testManager.getActivePromiseIds().length;

        this.assert(initialCount >= 3, 'testBatchCancelAll', '应该有活跃的 Promise');
        this.assert(finalCount === 0, 'testBatchCancelAll', '所有 Promise 应该被取消');
    }

    /**
     * 测试按分类批量取消
     */
    private async testBatchCancelCategory(): Promise<void> {
        const testCategory = 'test-category';
        const otherCategory = 'other-category';

        // 创建不同分类的 Promise
        const testPromise = this.testManager.create<string>(
            (resolve) => {
                setTimeout(() => resolve('test'), 1000);
            },
            { category: testCategory },
        );

        const otherPromise = this.testManager.create<string>(
            (resolve) => {
                setTimeout(() => resolve('other'), 1000);
            },
            { category: otherCategory },
        );

        // 取消指定分类
        await this.testManager.cancelCategory(testCategory);

        this.assert(testPromise.isCancelled, 'testBatchCancelCategory', '指定分类的 Promise 应该被取消');
        this.assert(!otherPromise.isCancelled, 'testBatchCancelCategory', '其他分类的 Promise 不应该被取消');

        // 清理
        await otherPromise.cancel();
    }

    /**
     * 测试管理器统计信息
     */
    private async testPromiseManagerStats(): Promise<void> {
        // 确保开始时没有活跃的 Promise
        await this.testManager.cancelAll();

        this.testManager.create<string>(
            (resolve) => {
                setTimeout(resolve, 100);
            },
            { category: 'test-stats' },
        );

        this.testManager.create<string>(
            (resolve) => {
                setTimeout(resolve, 100);
            },
            { category: 'test-stats' },
        );

        const ids = this.testManager.getActivePromiseIds();
        this.assert(ids.length === 2, 'testPromiseManagerStats', '应该有 2 个活跃的 Promise');

        const promise = this.testManager.getCancellablePromise(ids[0]);
        this.assert(promise !== undefined, 'testPromiseManagerStats', '应该能获取到 Promise');

        // 清理
        await this.testManager.cancelAll();
    }

    /**
     * 测试错误处理
     */
    private async testErrorHandling(): Promise<void> {
        // 测试取消不存在的 Promise
        await this.testManager.cancel('non-existent-id'); // 应该不抛出错误

        // 测试重复取消
        const promise = this.testManager.create<string>((resolve) => {
            setTimeout(resolve, 100);
        });

        await promise.cancel();
        await promise.cancel(); // 重复取消应该安全

        this.assert(promise.isCancelled, 'testErrorHandling', 'Promise 应该被标记为已取消');
    }

    /**
     * 测试边界情况
     */
    private async testEdgeCases(): Promise<void> {
        // 测试立即解决的 Promise
        const immediatePromise = this.testManager.create<string>((resolve) => {
            resolve('immediate');
        });

        const result = await immediatePromise.promise;
        this.assert(result === 'immediate', 'testEdgeCases', '立即解决的 Promise 应该正常工作');

        // 测试 0 超时
        const zeroTimeoutPromise = this.testManager.create<string>(
            (resolve) => {
                setTimeout(() => resolve('zero-timeout'), 10);
            },
            { timeout: 0 },
        ); // 0 超时应该被忽略

        const zeroResult = await zeroTimeoutPromise.promise;
        this.assert(zeroResult === 'zero-timeout', 'testEdgeCases', '0 超时应该被忽略');
    }

    /**
     * 测试内存清理
     */
    private async testMemoryCleanup(): Promise<void> {
        const initialCount = this.testManager.getActivePromiseIds().length;

        // 创建并立即解决 Promise
        const promise = this.testManager.create<string>((resolve) => {
            resolve('cleanup-test');
        });

        await promise.promise;

        // Promise 应该自动清理
        const finalCount = this.testManager.getActivePromiseIds().length;
        this.assert(finalCount === initialCount, 'testMemoryCleanup', '已完成的 Promise 应该被自动清理');
    }

    /**
     * 断言辅助函数
     */
    private assert(condition: boolean, testName: string, message: string): void {
        if (condition) {
            this.testResults.set(testName, true);
            LogUtils.log(`✅ ${testName}: ${message}`);
        } else {
            this.testResults.set(testName, false);
            LogUtils.error(`❌ ${testName}: ${message}`);
        }
    }

    /**
     * 延迟函数
     */
    private delay(ms: number): Promise<void> {
        return new Promise((resolve) => setTimeout(resolve, ms));
    }

    /**
     * 打印测试结果
     */
    private printResults(): void {
        const total = this.testResults.size;
        const passed = Array.from(this.testResults.values()).filter((result) => result).length;
        const failed = total - passed;

        LogUtils.log('\n=== 测试结果统计 ===');
        LogUtils.log(`总测试数: ${total}`);
        LogUtils.log(`通过: ${passed}`);
        LogUtils.log(`失败: ${failed}`);
        LogUtils.log(`成功率: ${((passed / total) * 100).toFixed(1)}%`);

        if (failed > 0) {
            LogUtils.log('\n失败的测试:');
            for (const [testName, result] of this.testResults) {
                if (!result) {
                    LogUtils.error(`- ${testName}`);
                }
            }
        }

        LogUtils.log('\n=== PromiseManager 单元测试完成 ===');
    }

    // 保留原有的简单测试作为演示
    aaa: number = 0;
    async testWaitForChangeDemo(): Promise<void> {
        LogUtils.log('开始 waitForChange 演示...');
        setTimeout(() => {
            this.aaa = 5;
            LogUtils.log('设置 aaa = 5');
        }, 1000);

        await peanut.promise.waitForChange<PromiseTest>(this, 'aaa', { expectedValue: 5 });
        LogUtils.log('window.aaa changed done');
    }
}
