const { ccclass, property } = cc._decorator;

@ccclass
export default class BackgroundMusicManager extends cc.Component {

    @property
    musicPath: string = "audio/background"; // 背景音乐路径

    @property
    musicShortPath: string = "audio/bgm_short"; // 短背景音乐路径

    @property
    clickSoundPath: string = "audio/bullet"; // 音效路径

    private audioId: number = -1;
    private audioClip: cc.AudioClip = null;
    private clickAudioClip: cc.AudioClip = null;

    onLoad() {
        cc.game.addPersistRootNode(this.node); // 保证音乐管理节点在场景切换时不被销毁
        this.loadMusic();
        this.loadClickSound();
    }

    // 加载背景音乐
    loadMusic() {
        if (!this.musicPath) {
            cc.error("请设置正确的背景音乐路径");
            return;
        }
        cc.resources.load(this.musicPath, cc.AudioClip, (err, clip) => {
            if (err) {
                cc.error("加载背景音乐失败：", err);
                return;
            }
            this.audioClip = clip;
            cc.log("背景音乐加载成功");
        });
        cc.resources.load(this.musicShortPath, cc.AudioClip, (err, clip) => {
            if (err) {
                cc.error("加载短背景音乐失败：", err);
                return;
            }
            this.audioClip = clip;
            cc.log("短背景音乐加载成功");
        });
    }

    // 加载点击音效
    loadClickSound() {
        if (!this.clickSoundPath) {
            cc.error("请设置正确的点击音效路径");
            return;
        }
        cc.resources.load(this.clickSoundPath, cc.AudioClip, (err, clip) => {
            if (err) {
                cc.error("加载点击音效失败：", err);
                return;
            }
            this.clickAudioClip = clip;
            cc.log("点击音效加载成功");
        });
    }

    // 播放背景音乐
    startMusic() {
        if (this.audioClip) {
            this.audioId = cc.audioEngine.playMusic(this.audioClip, true);
            cc.audioEngine.setMusicVolume(1.0); // 设置音量
            cc.log("正常播放背景音乐")
        } else {
            cc.error("背景音乐未加载完成！");
        }
    }

    pauseMusic() {
        if (this.audioId !== -1) {
            cc.audioEngine.pauseMusic();
        }
    }

    resumeMusic() {
        if (this.audioId !== -1) {
            cc.audioEngine.resumeMusic();
        }
    }

    stopMusic() {
        if (this.audioId !== -1) {
            cc.audioEngine.stopMusic();
            this.audioId = -1;
        }
    }
    // 播放背景音乐
    startShortMusic() {
        if (this.audioClip) {
            this.audioId = cc.audioEngine.playMusic(this.audioClip, true);
            cc.audioEngine.setMusicVolume(1.0); // 设置音量
            cc.log("正常播放短背景音乐")
        } else {
            cc.error("短背景音乐未加载完成！");
        }
    }
    // 播放点击音效
    playClickSound() {
        if (this.clickAudioClip) {
            cc.audioEngine.playEffect(this.clickAudioClip, false);
            cc.log("正常播放点击音效")
        } else {
            cc.error("点击音效未加载完成！");
        }
    }
}
