/**
* @ author:gaoang
* @ email:455528347@qq.com
* @ date:2021-08-09 14:43
* @ description:
* @ changeLog:
*/

import { _decorator, Component, sp, Vec2, v2, v3, Vec3, Color, logID } from 'cc';

const { ccclass, property } = _decorator;

@ccclass('SpineObj')
export class SpineObj extends Component {

    /**
     * spine动画组件
     */
    @property(sp.Skeleton)
    _spineAnim: sp.Skeleton = null!;
    @property({ type: sp.Skeleton })
    set spineAnim(anim) {
        this._spineAnim = anim;
    }
    get spineAnim(): sp.Skeleton {
        if (this._spineAnim == null) {
            if (this.node && this.node.getComponent(sp.Skeleton)) {
                this._spineAnim = this.node.getComponent(sp.Skeleton) as sp.Skeleton;
            }
        }
        return this._spineAnim;
    }

    /** 动画监听事件的名称 **/
    /*  动画监听事件的触发顺序: Interrupt-->Start-->LoopComplete-->Complete-->Dispose
    */
    public static EventType = {
        Start: 'Start',                      //动画播放开始
        Interrupt: 'Interrupt',              //动画播放中切换其他动画，导致动画中断(clearTracks不会触发该回调)
        LoopComplete: 'LoopComplete',        //动画播放完一次的回调(只有loop为true的时候才会触发)
        Complete: 'Complete',                //动画播放完成回调(clearTracks会触发该回调)，动画播放完成可能无法触发该回调（原因可能与spine工程中该动画的配置有关）
        Dispose: 'Dispose',                  //动画被销毁的回调(clearTracks会触发该回调)
        FrameEvent: 'frameEvent',           //动画帧事件回调，包含帧事件参数
    };

    /**
     * 动画监听事件的回调--Start
     * 注册回调方法的参数列表：event:sp.spine.TrackEntry
     */
    private startEventListenerArray: Array<{ callback: Function, target: Object }> | null = null;
    //@ts-ignore
    private startEventListener(trackEntry: sp.spine.TrackEntry): void {
        if (!this.startEventListenerArray) {
            return;
        }
        if (this.startEventListenerArray.length <= 0) {
            return;
        }
        for (var i = 0; i < this.startEventListenerArray.length; i++) {
            if (this.startEventListenerArray[i].callback && this.startEventListenerArray[i].target) {
                this.startEventListenerArray[i].callback.call(this.startEventListenerArray[i].target, trackEntry);
            }
        }
    }

    /**
     * 动画监听事件的回调--Interrupt
     * 注册回调方法的参数列表：event:sp.spine.TrackEntry
     */
    private interruptEventListenerArray: Array<{ callback: Function, target: Object }> | null = null;
    //@ts-ignore
    private interruptEventListener(trackEntry: sp.spine.TrackEntry): void {
        if (!this.interruptEventListenerArray) {
            return;
        }
        if (this.interruptEventListenerArray.length <= 0) {
            return;
        }
        for (var i = 0; i < this.interruptEventListenerArray.length; i++) {
            if (this.interruptEventListenerArray[i].callback && this.interruptEventListenerArray[i].target) {
                this.interruptEventListenerArray[i].callback.call(this.interruptEventListenerArray[i].target, trackEntry);
            }
        }
    }

    /**
     * 动画监听事件的回调--LoopComplete
     * 注册回调方法的参数列表：event:sp.spine.TrackEntry
     */
    private loopCompleteEventListenerArray: Array<{ callback: Function, target: Object }> | null = null;
    //@ts-ignore
    private loopCompleteEventListener(trackEntry: sp.spine.TrackEntry): void {
        if (!this.loopCompleteEventListenerArray) {
            return;
        }
        if (this.loopCompleteEventListenerArray.length <= 0) {
            return;
        }
        for (var i = 0; i < this.loopCompleteEventListenerArray.length; i++) {
            if (this.loopCompleteEventListenerArray[i].callback && this.loopCompleteEventListenerArray[i].target) {
                this.loopCompleteEventListenerArray[i].callback.call(this.loopCompleteEventListenerArray[i].target, trackEntry);
            }
        }
    }

    /**
     * 动画监听事件的回调--Complete
     * 注册回调方法的参数列表：event:sp.spine.TrackEntry
     */
    private completeEventListenerArray: Array<{ callback: Function, target: Object }> | null = null;
    //@ts-ignore
    private completeEventListener(trackEntry: sp.spine.TrackEntry): void {
        if (!this.completeEventListenerArray) {
            return;
        }
        if (this.completeEventListenerArray.length <= 0) {
            return;
        }
        for (var i = 0; i < this.completeEventListenerArray.length; i++) {
            if (this.completeEventListenerArray[i].callback && this.completeEventListenerArray[i].target) {
                this.completeEventListenerArray[i].callback.call(this.completeEventListenerArray[i].target, trackEntry);
            }
        }
    }

    /**
     * 动画监听事件的回调--Dispose
     * 注册回调方法的参数列表：event:sp.spine.TrackEntry
     */
    private disposeEventListenerArray: Array<{ callback: Function, target: Object }> | null = null;
    //@ts-ignore
    private disposeEventListener(trackEntry: sp.spine.TrackEntry): void {
        if (!this.disposeEventListenerArray) {
            return;
        }
        if (this.disposeEventListenerArray.length <= 0) {
            return;
        }
        for (var i = 0; i < this.disposeEventListenerArray.length; i++) {
            if (this.disposeEventListenerArray[i].callback && this.disposeEventListenerArray[i].target) {
                this.disposeEventListenerArray[i].callback.call(this.disposeEventListenerArray[i].target, trackEntry);
            }
        }
    }

    /**
     * 动画监听事件的回调--FrameEvent
     * 注册回调方法的参数列表：trackEntry:sp.spine.TrackEntry , event:sp.spine.Event
     * 帧事件参数获取：事件名称 event.data.name , 事件参数 event.floatValue、event.intValue、event.stringValue
     */
    private frameEventListenerArray: Array<{ callback: Function, target: Object }> | null = null;
    //@ts-ignore
    private frameEventListener(trackEntry: sp.spine.TrackEntry, event: sp.spine.Event): void {
        if (!this.frameEventListenerArray) {
            return;
        }
        if (this.frameEventListenerArray.length <= 0) {
            return;
        }
        for (var i = 0; i < this.frameEventListenerArray.length; i++) {
            if (this.frameEventListenerArray[i].callback && this.frameEventListenerArray[i].target) {
                this.frameEventListenerArray[i].callback.call(this.frameEventListenerArray[i].target, trackEntry, event);
            }
        }
    }
    /****************** CocosCreator生命周期 *******************/
    onLoad() {
        
    }

    onEnable(){
      
    }

    onDisable(){
       
    }
    /****************** 功能方法封装 *******************/
    /**
     * 动态加载动画数据
     * @param skeletonData 
     */
    public loadSpineSkeleton(skeletonData: sp.SkeletonData): void {
        this.setSpineSkeletonData(skeletonData);
    }

    /**
     * 清空动画数据
     */
    public clearSpineSkeleton(): void {
        this.stopAnimation();
        this.setSpineSkeletonData(null);
    }

    /**
     * 是否显示骨骼的调试信息
     * 备注：
     * 1、3.x在Editor中可见调试信息，在运行窗口中不可见
     * @param visible 
     */
    public setDebugInfoVisible(visible: boolean): void {
        //骨骼
        this.spineAnim.debugBones = visible;
        //插槽
        this.spineAnim.debugSlots = visible;
        //Mesh网格
        this.spineAnim.debugMesh = visible;
    }

    /**
     * 获取Spine动画的Skeleton类
     * 备注：
     * 1、Skeleton类中包含了SkeletonData和AnimationState
     */
    public getSpineSkeleton(): sp.Skeleton {
        return this.spineAnim;
    }

    /**
     * 获取Spine动画的SkeletonData类
     */
    public getSpineSkeletonData(): sp.SkeletonData|null {
        if(this.spineAnim){
            return this.spineAnim.skeletonData;
        }
        else{
            return null;
        }
    }
    /**
     * 设置Spine动画的SkeletonData数据，类型是CocosCreator的Spine数据类型
     * 区别于 setSkeletonData()方法，该方法设置的是底层运行时skeletonData数据
     * @param data 
     */
    public setSpineSkeletonData(data: sp.SkeletonData|null): void {
        //@ts-ignore
        this.spineAnim.skeletonData = data;
    }

    /**
     * 获取Spine动画的状态类，状态类包含了Spine动画的全部状态
     * 备注：
     * 1、AnimationState类中包含了AnimationStateData
     * @returns 
     */
    //@ts-ignore
    public getSpineAnimationState(): sp.spine.AnimationState {
        return this.spineAnim.getState();
    }

    /**
     * 获取Spine动画的状态数据类
     * @returns 
     */
    //@ts-ignore
    public getSpineAnimationStateData(): sp.spine.AnimationStateData {
        return this.spineAnim.getState().data;
    }

    /**
     * 获取Spine中所有动画的名称
     */
    public getSpineAnimationNames():Array<string>{
        let animationNames:Array<string> = [];
        if(this.getSpineAnimationStateData()){
            let animations = this.getSpineAnimationStateData().skeletonData.animations;
            if(animations){
                for(let i=0; i<animations.length; i++){
                    animationNames.push(animations[i].name);
                }
            }
        }
        return animationNames;
    }

    /**
     * 获取Spine动画某个轨道的数据
     * animation 当前动画
     * trackIndex 轨道索引id
     */
    //@ts-ignore
    public getSpineTrackEntry(trackIndex: number = 0): sp.spine.TrackEntry {
        return this.spineAnim.getCurrent(trackIndex);
    }

    /**
     * 获取Spine某个轨道当前播放的动画
     * @param trackEntry 
     * @returns 
     */
    //@ts-ignore
    public getSpineTrackEntryAnimation(trackIndexOrEntry:sp.spine.TrackEntry|number = 0): sp.spine.Animation {
        let trackEntry = null;
        if(typeof trackIndexOrEntry == 'number'){
            trackEntry = this.getSpineTrackEntry(trackIndexOrEntry);
        }
        else{
            trackEntry = trackIndexOrEntry;
        }

        if (trackEntry) {
            return trackEntry.animation;
        }
        return null;
    }

    /**
     * 获取指定轨道当前播放的动画
     */
    //@ts-ignore
    public getCurAnimation(trackIndex: number = 0): sp.spine.Animation {
        if (this.getSpineTrackEntry(trackIndex)) {
            return this.getSpineTrackEntry(trackIndex).animation;
        }
        return null;
    }

    /**
     * 获取当前正在播放的动画clip名称
     */
    public getCurAnimationName(trackIndex:number = 0): string | null {
        if (this.getSpineTrackEntry(trackIndex)) {
            return this.getSpineTrackEntry(trackIndex).animation.name;
        }
        return null;
    }

    /**
     * 获取当前骨骼中所有动画的时间缩放率
     */
    public getTimeScale(): number {
        if(this.spineAnim){
            return this.spineAnim.timeScale;
        }
        return 0;
    }
    /**
     * 设置当前骨骼中所有动画的时间缩放率
     * 备注：可以实现慢动作的效果
     * @param timeScale 
     */
    public setTimeScale(timeScale: number): void {
        if (this.spineAnim) {
            this.spineAnim.timeScale = timeScale;  
        }
    }

    /**
     * 设置Spine动画的叠加颜色
     * @param color 
     */
    public setColor(color:Color):void{
        if(this.spineAnim){
            this.spineAnim.color = color;
        }
    }
    /**
     * 复位颜色为白色
     */
    public resetColor():void{
        if(this.spineAnim){
            this.spineAnim.color = Color.WHITE;
        }
    }

    /**
     * 还原到起始动作（spine工程中骨架装配的起始动作）
     * 备注：
     * 1、需要先调用stopAnimation停止播放，复位才有效，否则复位后会立即被动画数据覆盖
     * @param setBones 还原骨骼到起始动作
     * @param setSlots 还原插槽到原始动作
     */
    public setToSetupPose(setBones: boolean = true, setSlots: boolean = true): void {
        if (setBones && setSlots) {
            this.spineAnim.setToSetupPose();
        }
        else if (setBones) {
            this.spineAnim.setBonesToSetupPose();
        }
        else if (setSlots) {
            this.spineAnim.setSlotsToSetupPose();
        }
    }

    /**
     * 设置皮肤
     * 如果失败,将使用默认皮肤
     * @param skinName 皮肤名称,不传入将使用默认皮肤<default>
     */
    public setSkin(skinName:string = "default"): void {
        try {
            this.spineAnim.setSkin(skinName);
        } catch (error) {
            this.spineAnim.setSkin("default");
        }
    }

    /**
     * 设置动画的过渡融合时间
     * 说明：设置好过渡时间后会永久生效
     * @param fromAnimName  从某个动画
     * @param toAnimName    到某个动画
     * @param mixDuration   过渡融合的时长
     */
    public setAnimationMix(fromAnimName: string, toAnimName: string, mixDuration: number) {
        this.spineAnim.setMix(fromAnimName, toAnimName, mixDuration);
    }

    /**
     * 检查是否包含指定名称的动画数据
     * @param animationName 
     */
    public hasAnimation(animationName: string): boolean {
        return this.spineAnim.findAnimation(animationName) != null;
    }

    /**
     * 播放动画
     * @param animationName 动画名称 或 动画列表
     * @param loop 是否循环
     * @param trackIndex 动画轨道:不同轨道的动画会自动融合
     */
    //@ts-ignore
    public playAnimation(animationName: string | Array<string>, loop: boolean, trackIndex: number = 0): sp.spine.TrackEntry {
        //恢复动画播放
        this.resumeAnimation();
        //添加播放列表
        if (animationName instanceof Array) {
            for (var i = 0; i < animationName.length; i++) {
                //loop为true时，队列末端的动画会循环播放
                //loop为false时，队列动画全部播放完毕后自动停止
                if(this.hasAnimation(animationName[i])){
                    this.addAnimation(animationName[i], loop,trackIndex);
                }
                else{
                    continue;
                }
            }
            return this.getSpineTrackEntry(trackIndex);
        }
        else {
            if(this.hasAnimation(animationName)){
                return this.spineAnim.setAnimation(trackIndex, animationName, loop);
            }
            else{
                return this.getSpineTrackEntry(trackIndex);
            }
        }
    }

    /**
     * 添加动画到动画队列中
     * @param trackIndex 动画轨道
     * @param animationName 动画名称
     * @param loop 是否循环
     * @param delay 延迟播放
     */
    //@ts-ignore
    public addAnimation(animationName: string, loop: boolean, delay:number = 0,trackIndex:number = 0): sp.spine.TrackEntry {
        if(this.hasAnimation(animationName)){
            return this.spineAnim.addAnimation(trackIndex, animationName, loop, delay);
        }
        else{
            return this.getSpineTrackEntry(trackIndex);
        }
    }

    /**
     * 暂停当前动画的播放
     * @param animationName 
     */
    public pauseAnimation(): void {
        if(this.spineAnim){
            this.spineAnim.paused = true;
        }
    }

    /**
     * 恢复当前动画的播放
     */
    public resumeAnimation(): void {
        if(this.spineAnim){
            this.spineAnim.paused = false;
        }
    }

    /**
     * 停止播放动画(动画会定格在停止瞬间)
     * @param trackIndex 动画轨道：传入null或undefined则清空所有动画轨道
     */
    public stopAnimation(trackIndex:number = 0): void {
        if (trackIndex == null || trackIndex == undefined) {
            this.spineAnim.clearTracks();
        }
        else {
            this.spineAnim.clearTrack(trackIndex);
        }
        this.setToSetupPose();
    }

    /**
     * 当前是否有动画正在播放
     * 备注：
     * 1、如果所有动画都播放完成了则返回false，如果有动画在循环播放则返回true
     */
    public isAnimationPlaying(trackIndex: number = 0): boolean {
        let track = this.getSpineTrackEntry(trackIndex);
        if (track) {
            if (track.loop) {
                return true;
            }
            else {
                return !track.isComplete();
            }
        }
        return false;
    }

    /**
     * 获取骨骼对象
     * 找不到对应名称的骨骼则返回null
     * @param boneName 骨骼名称
     */
    //@ts-ignore
    public getBone(boneName: string): sp.spine.Bone {
        if(this.spineAnim){
            return this.spineAnim.findBone(boneName);
        }
        return null;
    }

    /**
     * 获取骨骼对象在CocosCreator中的世界缩放
     * @param boneOrName 
     */
    //@ts-ignore
    public getBoneWorldScale(boneOrName: string | sp.spine.Bone): Vec3 | null {
        let bone = null;
        if (typeof boneOrName == 'string') {
            bone = this.getBone(boneOrName);
        }
        else {
            bone = boneOrName;
        }

        if (bone) {
            //计算在Spine坐标系中的世界旋转
            let curBone = bone;
            let boneWorldScale: Vec2 = v2(1,1);
            while (curBone) {
                //手动更新骨骼位置信息，防止使用了上一帧的位置数据
                curBone.updateWorldTransform();
                boneWorldScale.x *= curBone.scaleX;
                boneWorldScale.y *= curBone.scaleY;
                curBone = curBone.parent;
            }
            //将骨骼坐标系转换为Cocos坐标系
            let cocosWorldScale: Vec3 = this.node.transformLocalToWorldScale(v3(boneWorldScale.x,boneWorldScale.y,1));
            return cocosWorldScale;
        }
        else {
            return null;
        }
    }

    /**
     * 获取插槽对象
     * 备注：
     * 1、slot.bone是该插槽附着的骨骼对象
     * 2、slot.attachment是该插槽上附着的附件(贴图、碰撞框)
     * @param slotName 插槽名称
     */
    //@ts-ignore
    public getSlot(slotName: string): sp.spine.Slot {
        if(this.spineAnim){
            return this.spineAnim.findSlot(slotName);
        }
        return null;
    }

    /**
     * 获取插槽中的Attachment附件
     * 类型有：RegionAttachment、BoundingBoxAttachment、
     * @param slotName 
     * @returns 
     */
    //@ts-ignore
    public getAttachment(slotName: string): sp.spine.Attachment {
        let slot = this.getSlot(slotName);
        if (slot) {
            return slot.getAttachment();
        }
        return null;
    }

    /**
     * 获取边界框(依附在插槽上)，返回边界框的顶点数据(CocosCreator中的世界坐标)
     * 备注：
     * 1、只有动画在播放状态下才能获取到世界坐标
     * 2、只有边界框插槽在动画中激活的时候才能获取到，否则是null
     * @param slotName 
     */
    public getBoundingBoxVertices(slotName: string): Array<Vec2> | null {
        let slot = this.getSlot(slotName);
        let attachment = this.getAttachment(slotName);
        let bone = slot ? slot.bone : null;
        if(bone){
            //更新骨骼位置
            bone.updateWorldTransform();
        }
        if (slot && attachment) {
            //BoundingBoxAttachment 继承 VertexAttachment
            //@ts-ignore
            if (attachment instanceof sp.spine.VertexAttachment) {
                if (attachment.vertices) {
                    //将顶点转换为Spine工程中的世界坐标(不会考虑Cocos中ScaleX翻转的情况，是Spine坐标系)
                    let verticesSpine: Array<number> = new Array<number>();
                    this.computeWorldVertices(slot, 0, attachment.worldVerticesLength,
                        attachment.vertices, verticesSpine,
                        attachment.bones,
                        0, 2);
                    //将Spine工程世界坐标转换为CocosCreator中的世界坐标(会考虑Cocos中ScaleX的翻转情况，是Cocos坐标系)
                    let verticesCocos: Array<Vec2> = new Array<Vec2>();
                    for (var i = 0; i < verticesSpine.length; i = i + 2) {
                        let vertice: Vec3 = v3(0,0,0);
                        //边界框顶点坐标转换为CocosCreator坐标系中的世界坐标
                        vertice = this.node.transformLocalToWorldPoint(v3(verticesSpine[i], verticesSpine[i + 1],0));
                        //保存该顶点
                        verticesCocos.push(v2(vertice.x,vertice.y));
                    }
                    return verticesCocos;
                }
            }
        }
        //插槽不存在
        return null;
    }

    /**
     * 安卓平台没有 attachment.computeWorldVertices 方法，使用该方法替代
     * @param matrix 
     * @param x 
     * @param y 
     * @param result 
     * @param delta 
     */
    //@ts-ignore
    private computeWorldVertices(slot: sp.spine.Slot, start: number, count: number,
        localVertices: ArrayLike<number>, worldVertices: Array<number>,
        boundboxBones: Array<number>,
        offset: number, stride: number) 
    {
        count = offset + (count >> 1) * stride;
        var skeleton = slot.bone.skeleton;
        var deformArray = slot.deform;
        var vertices = localVertices;
        var bones = boundboxBones;
        if (null == bones) {
            deformArray.length > 0 && (vertices = deformArray);
            var bone = slot.bone;
            var x = bone.worldX;
            var y = bone.worldY;
            var a = bone.a, b = bone.b, c = bone.c, d = bone.d;
            for (var v_1 = start, w = offset; w < count; v_1 += 2, w += stride) {
                var vx = vertices[v_1], vy = vertices[v_1 + 1];
                worldVertices[w] = vx * a + vy * b + x;
                worldVertices[w + 1] = vx * c + vy * d + y;
            }
            return;
        }
        var v = 0, skip = 0;
        for (var i = 0; i < start; i += 2) {
            var n = bones[v];
            v += n + 1;
            skip += n;
        }
        var skeletonBones = skeleton.bones;
        if (0 == deformArray.length) for (var w = offset, b = 3 * skip; w < count; w += stride) {
            var wx = 0, wy = 0;
            var n = bones[v++];
            n += v;
            for (; v < n; v++, b += 3) {
                var bone = skeletonBones[bones[v]];
                var vx = vertices[b], vy = vertices[b + 1], weight = vertices[b + 2];
                wx += (vx * bone.a + vy * bone.b + bone.worldX) * weight;
                wy += (vx * bone.c + vy * bone.d + bone.worldY) * weight;
            }
            worldVertices[w] = wx;
            worldVertices[w + 1] = wy;
        } else {
            var deform = deformArray;
            for (var w = offset, b = 3 * skip, f = skip << 1; w < count; w += stride) {
                var wx = 0, wy = 0;
                var n = bones[v++];
                n += v;
                for (; v < n; v++, b += 3, f += 2) {
                    var bone = skeletonBones[bones[v]];
                    var vx = vertices[b] + deform[f], vy = vertices[b + 1] + deform[f + 1], weight = vertices[b + 2];
                    wx += (vx * bone.a + vy * bone.b + bone.worldX) * weight;
                    wy += (vx * bone.c + vy * bone.d + bone.worldY) * weight;
                }
                worldVertices[w] = wx;
                worldVertices[w + 1] = wy;
            }
        }
    }

}