import { Camera, Component, Layers, Material, Node, RenderTexture, Size, Sprite, SpriteFrame, UITransform, Vec3, _decorator, director, isValid, v2, v3 } from "cc"
import Queue from "../../../Core/Base/Container/Queue"
import { NodeLayer } from "../../../Core/Base/Obj/NodeObjType"
import { ResConst } from "../../../Core/Res/ResConst"
import ResMgr from "../../../Core/Res/ResMgr"
import UIMgr from "../../UI/UIMgr"
import CameraController from "../CameraController"

const { ccclass, property, executeInEditMode } = _decorator



/**
 */
@ccclass
export default class KawaseBlur extends Component {

    private sprite: Sprite
    private materialDown: Material = null
    private materialUp: Material = null
    private offset: number = 5
    private iteration: number = 1
    private scale: number = 0.5
    private uiTrans: UITransform

    //对象池
    private srcRT: RenderTexture = new RenderTexture()
    private lastRT: RenderTexture = new RenderTexture()
    private spriteFramePool: Queue<SpriteFrame> = new Queue()
    private static renderCounter: number = 0
    private static renderCamera: Camera
    private static renderSprite: Sprite

    protected onLoad() {
        KawaseBlur.createRenderNodes()

        this.uiTrans = UIMgr.UIRoot.getComponent(UITransform)
        this.node.setScale(v3(1, -1, 1))
        this.sprite = this.getComponent(Sprite)
        this.srcRT.addRef()
        this.lastRT.addRef()
        const trans = this.uiTrans
        this.srcRT.reset({ width: Math.floor(trans.width), height: Math.floor(trans.height) })
        this.lastRT.reset({ width: Math.floor(trans.width), height: Math.floor(trans.height) })
    }

    protected onEnable(): void {
        this.blur(this.offset, this.iteration, this.scale)
    }

    protected onDisable(): void {
        this.node.setScale(Vec3.ONE)
        const sp = this.sprite.spriteFrame
        sp && this.spriteFramePool.enqueue(sp)
        //bug:移除后不显示
        // this.sprite.spriteFrame = null
    }

    protected onDestroy(): void {
        this.srcRT.decRef()
        this.lastRT.decRef()
        this.srcRT = null
        this.lastRT = null
    }

    /**
     * 模糊渲染
     * @param offset 模糊半径
     * @param iteration 模糊迭代次数
     * @param scale 降采样缩放比例
     */
    private async blur(offset: number, iteration: number, scale: number = 0.5) {
        if (!this.materialDown) {
            this.materialDown = await ResMgr.loadAssetAsync(ResConst.Mat, "sprite-kawaseblurdown", Material)
            this.materialUp = await ResMgr.loadAssetAsync(ResConst.Mat, "sprite-kawaseblurup", Material)
        }
        // 设置材质
        const trans = this.uiTrans
        this.materialDown.setProperty('resolution', v2(trans.width, trans.height))
        this.materialDown.setProperty('offset', offset)
        this.materialUp.setProperty('resolution', v2(trans.width, trans.height))
        this.materialUp.setProperty('offset', offset)
        // 创建临时 RenderTexture
        // let srcRT = this.renderTexPool.dequeue()??new RenderTexture()
        // let lastRT = this.renderTexPool.dequeue()??new RenderTexture()
        let srcRT = this.srcRT, lastRT = this.lastRT
        // 获取初始 RenderTexture
        await this.screenShotTexture(lastRT)
        // Downsample
        for (let i = 0; i < iteration; i++) {
            [lastRT, srcRT] = [srcRT, lastRT]
            await this.renderWithMaterial(srcRT, lastRT, this.materialDown)
        }
        // Upsample
        for (let i = 0; i < iteration; i++) {
            [lastRT, srcRT] = [srcRT, lastRT]
            await this.renderWithMaterial(srcRT, lastRT, this.materialUp)
        }
        
        const sp = this.spriteFramePool.dequeue() ?? new SpriteFrame()
        sp.reset({ texture: lastRT })
        sp.packable = false
        this.sprite.spriteFrame = sp
    }


    /**
     * 获取节点的 RenderTexture
     * @param node 节点
     * @param out 输出
     * @see RenderUtil.ts https://gitee.com/ifaswind/eazax-ccc/blob/master/utils/RenderUtil.ts
     */
    protected screenShotTexture(out?: RenderTexture) {
        return new Promise<void>((resolve)=>{
            const mainCamera = CameraController.Inst.MainCamera
            if(mainCamera?.isValid){
                mainCamera.targetTexture = out
            }
            director.root.frameMove(0)
            this.scheduleOnce(()=>{
                mainCamera&&(mainCamera.targetTexture = null)
                resolve()
            },0)
        })

    }

    /**
     * 使用指定材质来将 RenderTexture 渲染到另一个 RenderTexture
     * @param srcRT 来源
     * @param dstRT 目标
     * @param material 材质
     * @param size RenderTexture尺寸缩放比例
     * @see RenderUtil.ts https://gitee.com/ifaswind/eazax-ccc/blob/master/utils/RenderUtil.ts
     */
    protected async renderWithMaterial(srcRT: RenderTexture, dstRT: RenderTexture, material?: Material, size?: Size) {
        return new Promise<void>((resolve)=>{
            const { renderCamera, renderSprite } = KawaseBlur.showRenderNodes()
            renderCamera.node.active = true
            const sf = renderSprite
            const sp = this.spriteFramePool.dequeue() ?? new SpriteFrame()
            sp.reset({ texture: srcRT })
            sp.packable = false
            sf.spriteFrame = sp
            sf.customMaterial = material
            // 获取图像宽高
            const { width, height } = size ?? { width: srcRT.width, height: srcRT.height }
            dstRT.resize(width, height)
            // 将临时节点渲染到 RenderTexture 中
            renderCamera.targetTexture = dstRT
            //渲染
            director.root.frameMove(0)
            this.scheduleOnce(()=>{
                renderCamera.targetTexture = null
                KawaseBlur.hideRenderNodes()
                this.spriteFramePool.enqueue(sp)
                resolve()
            },0)

        })

    }

    private static createRenderNodes() {
        if (isValid(this.renderCamera)) return
        const sf = new Node().addComponent(Sprite)
        const layer = 1 << Layers.nameToLayer(NodeLayer.RenderTexture)
        sf.node.layer = layer
        const uiCamera = CameraController.Inst.UiCamera
        const cameraNode = new Node("RenderCamera")
        cameraNode.addComponent(UITransform)
        sf.node.setParent(cameraNode)
        //z轴至少-1
        sf.node.setPosition(v3(0, 0, -1))
        //摄像机
        cameraNode.layer = layer
        cameraNode.setParent(UIMgr.UIRoot)
        cameraNode.setPosition(Vec3.ZERO)
        const camera = cameraNode.addComponent(Camera);
        camera.clearColor = uiCamera.clearColor
        camera.clearFlags = uiCamera.clearFlags
        camera.projection = uiCamera.projection
        camera.orthoHeight = uiCamera.orthoHeight
        camera.priority = 0
        camera.far = uiCamera.far
        camera.visibility = layer

        this.renderCamera = camera
        this.renderSprite = sf
    }

    private static showRenderNodes(): { renderCamera: Camera, renderSprite: Sprite } {
        this.renderCounter++
        return { renderCamera: this.renderCamera, renderSprite: this.renderSprite }
    }

    private static hideRenderNodes() {
        this.renderCounter = Math.max(0, --this.renderCounter)
        this.renderCamera.node.active = this.renderCounter > 0
    }

}
