
import { _decorator, Component, Node, Label, resources, SpriteFrame, Sprite, instantiate, Prefab, director, UITransform, Layers, find } from 'cc';
import { ItemCtrl, OperatorType } from './ItemCtrl';
const { ccclass, property } = _decorator;

@ccclass('GameCtrl')
export class GameCtrl extends Component {

    @property(Node)
    itemRoot: Node = null;

    @property(Label)
    hpLabel: Label = null;

    @property(Label)
    gameOverLabel: Label = null;

    private hp: number = 0;
    private isGameOver: boolean = false;
    private itemBgSpriteFrame: SpriteFrame = null;
    private spawnTimer: number = 0;
    private spawnInterval: number = 1.5;

    start() {
        console.log("GameCtrl start");
        
        // 动态查找保底
        if (!this.hpLabel) {
            const labelNode = find("Canvas/UIRoot/HpLabel");
            if (labelNode) {
                this.hpLabel = labelNode.getComponent(Label);
                console.log("HpLabel found dynamically");
            }
        }
        if (!this.gameOverLabel) {
            const labelNode = find("Canvas/UIRoot/GameOverLabel");
            if (labelNode) {
                this.gameOverLabel = labelNode.getComponent(Label);
            }
        }
        if (!this.itemRoot) {
            this.itemRoot = find("Canvas/GameRoot/ItemRoot");
        }

        if (this.hpLabel) {
            console.log("HpLabel found");
            this.hpLabel.string = "INIT HP";
        } else {
            console.error("HpLabel NOT found");
        }
        this.initGame();
        this.loadResources();
    }

    initGame() {
        this.hp = Math.floor(Math.random() * 31) + 20; // 20-50
        this.updateHpUI();
        this.isGameOver = false;
        if (this.gameOverLabel) this.gameOverLabel.node.active = false;
    }

    loadResources() {
        // 动态加载 item_bg
        resources.load("Image/item_bg/spriteFrame", SpriteFrame, (err, spriteFrame) => {
            if (err) {
                console.error("Failed to load item_bg:", err);
                // 尝试加载 texture 并转为 spriteFrame (兼容性)
                resources.load("Image/item_bg", (err2, asset) => {
                     if (err2) console.error("Failed to load item_bg as texture:", err2);
                });
                // 保底
                resources.load("default_sprite_splash", SpriteFrame, (err3, sf) => {
                    if (!err3) this.itemBgSpriteFrame = sf;
                });
                return;
            }
            this.itemBgSpriteFrame = spriteFrame;
        });
        
        // 强制给 Player 和 Background 设置保底图片
        resources.load("default_sprite_splash", SpriteFrame, (err, sf) => {
            if (!err) {
                // Player
                const player = find("Canvas/GameRoot/Player");
                if (player) {
                    const sprite = player.getComponent(Sprite);
                    if (sprite && !sprite.spriteFrame) { // 只有当没有图片时才设置，或者强制设置
                        sprite.spriteFrame = sf;
                        console.log("Set default sprite for Player");
                    }
                }
                // Background
                const bg = find("Canvas/Background");
                if (bg) {
                    const sprite = bg.getComponent(Sprite);
                    if (sprite && !sprite.spriteFrame) {
                         sprite.spriteFrame = sf;
                         console.log("Set default sprite for Background");
                    }
                }
            }
        });
    }

    update(deltaTime: number) {
        if (this.isGameOver) return;

        this.spawnTimer += deltaTime;
        if (this.spawnTimer >= this.spawnInterval) {
            this.spawnTimer = 0;
            this.spawnItem();
        }
    }

    spawnItem() {
        // 创建节点
        const node = new Node("Item");
        node.layer = Layers.Enum.UI_2D; // 确保可见
        
        // 添加 Sprite
        const sprite = node.addComponent(Sprite);
        if (this.itemBgSpriteFrame) {
            sprite.spriteFrame = this.itemBgSpriteFrame;
        } else {
            // 如果没有加载到，尝试加载默认的
            resources.load("default_sprite_splash", SpriteFrame, (err, sf) => {
                if (!err && sprite.isValid) {
                    sprite.spriteFrame = sf;
                }
            });
        }
        sprite.color.set(255, 255, 0, 255); // 黄色背景
        
        // 添加 UITransform
        const uiTransform = node.getComponent(UITransform) || node.addComponent(UITransform);
        uiTransform.setContentSize(100, 100);

        // 添加 Label 子节点
        const labelNode = new Node("Label");
        labelNode.layer = Layers.Enum.UI_2D;
        const label = labelNode.addComponent(Label);
        label.fontSize = 40;
        label.color.set(0, 0, 0, 255); // 黑色字体
        labelNode.parent = node;

        // 添加脚本
        const itemCtrl = node.addComponent(ItemCtrl);
        itemCtrl.label = label; // 直接赋值，确保 init 时可用
        
        // 随机属性
        const op = Math.floor(Math.random() * 4); // 0-3
        const val = Math.floor(Math.random() * 9) + 1; // 1-9
        itemCtrl.init(op, val);

        // 随机位置 (左: -150, 右: 150)
        const isLeft = Math.random() < 0.5;
        node.setPosition(isLeft ? -150 : 150, 600, 0); // 从上方生成

        node.parent = this.itemRoot;
    }

    onItemHit(op: OperatorType, val: number) {
        if (this.isGameOver) return;

        console.log(`Item Hit: op=${op}, val=${val}, currentHp=${this.hp}`);

        switch (op) {
            case OperatorType.ADD: this.hp += val; break;
            case OperatorType.SUB: this.hp -= val; break;
            case OperatorType.MUL: this.hp *= val; break;
            case OperatorType.DIV: this.hp /= val; break;
        }

        this.hp = Math.floor(this.hp);
        console.log(`New HP: ${this.hp}`);
        this.updateHpUI();

        if (this.hp <= 0) {
            console.log("Game Over Triggered");
            this.gameOver();
        }
    }

    updateHpUI() {
        if (this.hpLabel) {
            this.hpLabel.string = `HP: ${this.hp}`;
        }
    }

    gameOver() {
        this.isGameOver = true;
        if (this.gameOverLabel) {
            this.gameOverLabel.node.active = true;
            this.gameOverLabel.string = "GAME OVER";
        }
        // 停止所有 Item 移动? 可以在 ItemCtrl 里判断 GameCtrl.isGameOver
        // 或者简单地暂停场景
        // director.pause(); // 暂停会导致无法重玩，这里仅逻辑停止
    }
}
