import { _decorator, Component, Node, Vec3, tween, Graphics, CCBoolean, CCInteger, Color } from 'cc';
const { ccclass, property, executeInEditMode } = _decorator;

// 声明全局常量
declare const CC_EDITOR: boolean;

@ccclass('BezierTest')
@executeInEditMode(true)
export class BezierTest extends Component {

    @property(Node)
    bulletNode: Node = null!;

    @property(Node)
    startPoint: Node = null!;

    @property(Node)
    controlPoint: Node = null!;

    @property(Node)
    controlPoint2: Node = null!;

    @property(Node)
    endPoint: Node = null!;

    @property(Graphics)
    graphics: Graphics = null!;

    @property(CCBoolean)
    showInEditor: boolean = true;

    @property(CCInteger)
    curveSegments: number = 50;

    @property(CCBoolean)
    showControlLines: boolean = true;

    private _lastStartPos: Vec3 = new Vec3();
    private _lastControlPos: Vec3 = new Vec3();
    private _lastControl2Pos: Vec3 = new Vec3();
    private _lastEndPos: Vec3 = new Vec3();

    start() {
        if (!CC_EDITOR) {
            // 运行时自动开始演示
            this.testBezierMovement();
        }
    }

    update(deltaTime: number) {
        if (CC_EDITOR && this.showInEditor) {
            this.updateEditorPreview();
        }
    }

    /**
     * 编辑器模式下更新预览
     */
    private updateEditorPreview(): void {
        if (!this.startPoint || !this.endPoint || !this.controlPoint || !this.graphics) return;

        const startPos = this.startPoint.getPosition();
        const endPos = this.endPoint.getPosition();
        const controlPos = this.controlPoint.getPosition();

        // 检查是否使用三次贝塞尔曲线
        if (this.controlPoint2) {
            const controlPos2 = this.controlPoint2.getPosition();
            // 检查位置是否发生变化（三次曲线）
            if (!this.positionsChangedCubic(startPos, controlPos, controlPos2, endPos)) {
                return;
            }
            // 更新上次位置
            this._lastStartPos.set(startPos);
            this._lastControlPos.set(controlPos);
            this._lastControl2Pos.set(controlPos2);
            this._lastEndPos.set(endPos);
            // 绘制三次贝塞尔曲线
            this.drawCubicBezierCurve(startPos, controlPos, controlPos2, endPos);
        } else {
            // 检查位置是否发生变化（二次曲线）
            if (!this.positionsChanged(startPos, controlPos, endPos)) {
                return;
            }
            // 更新上次位置
            this._lastStartPos.set(startPos);
            this._lastControlPos.set(controlPos);
            this._lastEndPos.set(endPos);
            // 绘制二次贝塞尔曲线
            this.drawBezierCurve(startPos, controlPos, endPos);
        }
    }

    /**
     * 检查位置是否发生变化（三次曲线）
     */
    private positionsChangedCubic(startPos: Vec3, controlPos: Vec3, control2Pos: Vec3, endPos: Vec3): boolean {
        return !startPos.equals(this._lastStartPos) || 
               !controlPos.equals(this._lastControlPos) || 
               !control2Pos.equals(this._lastControl2Pos) ||
               !endPos.equals(this._lastEndPos);
    }

    /**
     * 绘制三次贝塞尔曲线（编辑器预览版本）
     */
    private drawCubicBezierCurve(startPos: Vec3, controlPos: Vec3, control2Pos: Vec3, endPos: Vec3): void {
        if (!this.graphics) return;

        this.graphics.clear();

        // 绘制控制线
        if (this.showControlLines) {
            this.graphics.lineWidth = 1;
            this.graphics.strokeColor = Color.BLUE;
            
            // 起点到控制点1
            this.graphics.moveTo(startPos.x, startPos.y);
            this.graphics.lineTo(controlPos.x, controlPos.y);
            this.graphics.stroke();
            
            // 控制点1到控制点2
            this.graphics.moveTo(controlPos.x, controlPos.y);
            this.graphics.lineTo(control2Pos.x, control2Pos.y);
            this.graphics.stroke();
            
            // 控制点2到终点
            this.graphics.moveTo(control2Pos.x, control2Pos.y);
            this.graphics.lineTo(endPos.x, endPos.y);
            this.graphics.stroke();
        }

        // 绘制三次贝塞尔曲线
        this.graphics.lineWidth = 3;
        this.graphics.strokeColor = Color.GREEN;
        
        this.graphics.moveTo(startPos.x, startPos.y);
        
        // 使用多段直线近似三次贝塞尔曲线
        for (let i = 1; i <= this.curveSegments; i++) {
            const t = i / this.curveSegments;
            const point = this.Bezier_Cubic(t, startPos, controlPos, control2Pos, endPos);
            this.graphics.lineTo(point.x, point.y);
        }
        
        this.graphics.stroke();
    }

    /**
     * 检查位置是否发生变化（二次曲线）
     */
    private positionsChanged(startPos: Vec3, controlPos: Vec3, endPos: Vec3): boolean {
        return !startPos.equals(this._lastStartPos) || 
               !controlPos.equals(this._lastControlPos) || 
               !endPos.equals(this._lastEndPos);
    }
    /**
     * 绘制贝塞尔曲线（编辑器预览版本）
     */
    private drawBezierCurve(startPos: Vec3, controlPos: Vec3, endPos: Vec3): void {
        if (!this.graphics) return;

        this.graphics.clear();

        // 绘制控制线
        if (this.showControlLines) {
            this.graphics.lineWidth = 1;
            this.graphics.strokeColor = Color.BLUE;
            
            // 起点到控制点
            this.graphics.moveTo(startPos.x, startPos.y);
            this.graphics.lineTo(controlPos.x, controlPos.y);
            this.graphics.stroke();
            
            // 控制点到终点
            this.graphics.moveTo(controlPos.x, controlPos.y);
            this.graphics.lineTo(endPos.x, endPos.y);
            this.graphics.stroke();
        }

        // 绘制贝塞尔曲线
        this.graphics.lineWidth = 3;
        this.graphics.strokeColor = Color.GREEN;
        
        this.graphics.moveTo(startPos.x, startPos.y);
        
        // 使用多段直线近似贝塞尔曲线
        for (let i = 1; i <= this.curveSegments; i++) {
            const t = i / this.curveSegments;
            const point = this.Bezier_Quadratic(t, startPos, controlPos, endPos);
            this.graphics.lineTo(point.x, point.y);
        }
        
        this.graphics.stroke();
    }

    /**
     * 二阶贝塞尔曲线计算
     * B(t) = (1-t)²P₀ + 2t(1-t)P₁ + t²P₂
     */
    private Bezier_Quadratic(t: number, p0: Vec3, p1: Vec3, p2: Vec3): Vec3 {
        // 确保t在[0,1]范围内
        t = Math.min(Math.max(t, 0), 1); 
        // 原式：B(t) = (1-t)²P₀ + 2t(1-t)P₁ + t²P₂
        // 使用中间变量避免重复计算，令u=1-t：B(t) = uu * P₀ + _2tu * P₁ + tt * P₂
        const u = 1 - t;
        const uu = u * u;
        const tt = t * t;
        const _2tu = 2 * t * u;
        // 计算x, y, z坐标
        const x = uu * p0.x + _2tu * p1.x + tt * p2.x;
        const y = uu * p0.y + _2tu * p1.y + tt * p2.y;
        const z = uu * p0.z + _2tu * p1.z + tt * p2.z;

        return new Vec3(x, y, z);
    }

    /**
     * 三阶贝塞尔曲线计算
     * B(t) = (1-t)³P₀ + 3t(1-t)²P₁ + 3t²(1-t)P₂ + t³P₃
     */
    private Bezier_Cubic(t: number, p0: Vec3, p1: Vec3, p2: Vec3, p3: Vec3): Vec3 {
        // 确保t在[0,1]范围内
        t = Math.min(Math.max(t, 0), 1); 
        // 原式：B(t) = (1-t)³P₀ + 3t(1-t)²P₁ + 3t²(1-t)P₂ + t³P₃
        // 使用中间变量避免重复计算，令u=1-t：B(t) = uuu * P₀ + _3tuu * P₁ + _3ttu * P₂ + ttt * P₃
        const u = 1 - t;
        const uu = u * u;
        const uuu = uu * u;
        const tt = t * t;
        const ttt = tt * t;

        const _3tuu = 3 * t * uu;
        const _3ttu = 3 * tt * u;

        const x = uuu * p0.x + _3tuu * p1.x + _3ttu * p2.x + ttt * p3.x;
        const y = uuu * p0.y + _3tuu * p1.y + _3ttu * p2.y + ttt * p3.y;
        const z = uuu * p0.z + _3tuu * p1.z + _3ttu * p2.z + ttt * p3.z;
    
        return new Vec3(x, y, z);
    }

    /**
     * 计算二次贝塞尔曲线在 t 时刻的切线方向
     * 用于确定子弹的朝向
     */
    private calculateQuadraticBezierTangent(t: number, p0: Vec3, p1: Vec3, p2: Vec3): Vec3 {
        // 二次贝塞尔曲线
        // 原式：B(t) = (1-t)²P₀ + 2t(1-t)P₁ + t²P₂
        // 求导：B'(t) = 2(1-t)(P₁-P₀) + 2t(P₂-P₁)
        const u = 1 - t;
        const dx = 2 * u * (p1.x - p0.x) + 2 * t * (p2.x - p1.x);
        const dy = 2 * u * (p1.y - p0.y) + 2 * t * (p2.y - p1.y);
        
        return new Vec3(dx, dy, 0).normalize();
    }

    /**
     * 计算三次贝塞尔曲线在 t 时刻的切线方向
     */
    private calculateCubicBezierTangent(t: number, p0: Vec3, p1: Vec3, p2: Vec3, p3: Vec3): Vec3 {
        // 三次贝塞尔曲线求导：B'(t) = 3(1-t)²(P₁-P₀) + 6t(1-t)(P₂-P₁) + 3t²(P₃-P₂)
        const u = 1 - t;
        const uu = u * u;
        const tt = t * t;
        
        const dx = 3 * uu * (p1.x - p0.x) + 6 * t * u * (p2.x - p1.x) + 3 * tt * (p3.x - p2.x);
        const dy = 3 * uu * (p1.y - p0.y) + 6 * t * u * (p2.y - p1.y) + 3 * tt * (p3.y - p2.y);
        
        return new Vec3(dx, dy, 0).normalize();
    }

    /**
     * 沿二次贝塞尔曲线移动节点
     */
    private startBezierMovement(targetNode: Node, startPos: Vec3, control1: Vec3, endPos: Vec3): void {
        // 使用虚拟对象进行tween
        const tweenObj = { progress: 0 };

        tween(tweenObj)
        .to(1, { progress: 1 }, {
            onUpdate: (target: any, ratio: number) => {
                const t = target.progress; 
                // 计算当前位置
                const position = this.Bezier_Quadratic(t, startPos, control1, endPos);
                targetNode.setPosition(position);

                // 当前切线
                const tangent = this.calculateQuadraticBezierTangent(t, startPos, control1, endPos);
                // 计算角度（弧度）
                const angle = Math.atan2(tangent.y, tangent.x);
                // 转换为度数并设置旋转
                const degrees = angle * 180 / Math.PI;
                targetNode.setRotationFromEuler(0, 0, degrees);
            },
            onComplete: () => {
                console.log('贝塞尔曲线运动完成');
            }
        })
        .start();
    }

    /**
     * 沿三次贝塞尔曲线移动节点
     */
    private startCubicBezierMovement(targetNode: Node, startPos: Vec3, control1: Vec3, control2: Vec3, endPos: Vec3): void {
        const tweenObj = { progress: 0 };

        tween(tweenObj)
        .to(1.5, { progress: 1 }, {
            onUpdate: (target: any, ratio: number) => {
                const t = target.progress; 
                // 计算当前位置
                const position = this.Bezier_Cubic(t, startPos, control1, control2, endPos);
                targetNode.setPosition(position);

                // 当前切线
                const tangent = this.calculateCubicBezierTangent(t, startPos, control1, control2, endPos);
                // 计算角度（弧度）
                const angle = Math.atan2(tangent.y, tangent.x);
                // 转换为度数并设置旋转
                const degrees = angle * 180 / Math.PI;
                targetNode.setRotationFromEuler(0, 0, degrees);
            },
            onComplete: () => {
                console.log('三次贝塞尔曲线运动完成');
            }
        })
        .start();
    }

    /**
     * 生成随机控制点
     * 在起始、结束两点间中垂线上随机距离
     */
    private generateRandomControlPoint(startPos: Vec3, endPos: Vec3): Vec3 {
        // 计算中点
        const midPoint = startPos.clone().lerp(endPos, 0.5);
        // 计算方向向量（从起点到终点）
        const direction = endPos.clone().subtract(startPos).normalize();
        // 计算垂直向量（垂直于连接线）
        const perpendicular = new Vec3(-direction.y, direction.x, 0).normalize();
        // 在垂直方向上随机偏移
        const randomHeight = Math.random() * 400 - 200;
        // 控制点位于中点正上方（或正下方）的垂直线上
        const control1 = midPoint.clone().add(perpendicular.multiplyScalar(randomHeight));
        
        return control1;
    }

    /**
     * 绘制贝塞尔曲线路径（运行时版本）
     */
    private drawBezierPath(p0: Vec3, p1: Vec3, p2: Vec3, p3?: Vec3) {
        if (!this.graphics) return;
        
        const graphics = this.graphics; 
        graphics.clear();
        graphics.lineWidth = 3;
        graphics.strokeColor = Color.RED;

        graphics.moveTo(p0.x, p0.y);
        if (p3) {
            // 三次贝塞尔曲线
            graphics.bezierCurveTo(p1.x, p1.y, p2.x, p2.y, p3.x, p3.y);
        } else {
            // 二次贝塞尔曲线
            graphics.quadraticCurveTo(p1.x, p1.y, p2.x, p2.y);
        }        
        graphics.stroke();
    }

    /**
     * 测试贝塞尔曲线运动（自动检测二次或三次）
     */
    private testBezierMovement(): void {
        if (!this.bulletNode || !this.startPoint || !this.endPoint || !this.controlPoint) {
            console.warn('请在编辑器中设置必要的节点引用（startPoint, controlPoint, endPoint, bulletNode）');
            return;
        }

        const startPos = this.startPoint.getPosition();
        const endPos = this.endPoint.getPosition();
        const controlPos = this.controlPoint.getPosition();

        // 检查是否有第二个控制点，决定使用二次还是三次贝塞尔曲线
        if (this.controlPoint2) {
            const control2Pos = this.controlPoint2.getPosition();
            // 使用三次贝塞尔曲线
            this.drawBezierPath(startPos, controlPos, control2Pos, endPos);
            this.startCubicBezierMovement(this.bulletNode, startPos, controlPos, control2Pos, endPos);
        } else {
            // 使用二次贝塞尔曲线
            this.drawBezierPath(startPos, controlPos, endPos);
            this.startBezierMovement(this.bulletNode, startPos, controlPos, endPos);
        }
    }

    /**
     * 重新开始运动（可在编辑器中调用）
     */
    public restart(): void {
        this.testBezierMovement();
    }

    /**
     * 强制刷新编辑器预览
     */
    public refreshPreview(): void {
        if (CC_EDITOR) {
            this._lastStartPos.set(Vec3.ZERO);
            this._lastControlPos.set(Vec3.ZERO);
            this._lastControl2Pos.set(Vec3.ZERO);
            this._lastEndPos.set(Vec3.ZERO);
        }
    }
}