const { ccclass, property } = cc._decorator;

@ccclass
export default class Helloworld extends cc.Component {


    @property({ tooltip: "采样间距（像素）" })
    minSegment: number = 2;
    // ====== 缩放 ======
    @property({ tooltip: "滚轮缩放倍率（>1放大）" })
    zoomStep: number = 1.1;

    @property
    minScale: number = 0.2;

    @property
    maxScale: number = 5;

    @property({ tooltip: "是否以鼠标为缩放中心" })
    zoomAtCursor: boolean = true;
    @property(cc.Camera)
    camera: cc.Camera = null;//烘焙相机
    private _rt: cc.RenderTexture = null;   // 历史纹理
    // private _cam: cc.Camera = null;         // 烘焙相机（把 Work 烘进 _rt）
    private _drawing = false;
    private _lastPt: cc.Vec2 = null;
    @property(cc.Graphics)
    work: cc.Graphics = null;               // 工作层（当前笔/橡皮绘制）
    @property(cc.Sprite)
    historySprite: cc.Sprite = null;        // 显示历史纹理（可不拖，代码会自动创建）
    @property(cc.Node)
    bg: cc.Node = null;

    // clearFlags 常量（不清颜色 => 累计绘制）
    private _CLR_COLOR = (cc.Camera.ClearFlags as any).COLOR || 1;
    private _CLR_DEPTH = (cc.Camera.ClearFlags as any).DEPTH || 2;
    private _CLR_STENCIL = (cc.Camera.ClearFlags as any).STENCIL || 4;
    private get _NO_COLOR(): number { return this._CLR_DEPTH | this._CLR_STENCIL; }
    private get _WITH_COLOR(): number { return this._CLR_COLOR | this._CLR_DEPTH | this._CLR_STENCIL; }

    start() {
        this.initHistoryRT();
        this.addEvent();
    }


    protected addEvent(): void {
        // 事件（绘制 & 滚轮）
        this.node.on(cc.Node.EventType.TOUCH_START, this.onTouchStart, this);
        this.node.on(cc.Node.EventType.TOUCH_MOVE, this.onTouchMove, this);
        this.node.on(cc.Node.EventType.TOUCH_END, this.onTouchEnd, this);
        this.node.on(cc.Node.EventType.TOUCH_CANCEL, this.onTouchEnd, this);
        this.node.on(cc.Node.EventType.MOUSE_WHEEL, this.onMouseWheel, this);
    }

    // ===================== 滚轮缩放（作用在 bg） =====================
    private onMouseWheel(e: cc.Event.EventMouse) {

        const scrollY = e.getScrollY();
        if (!scrollY) return;

        let factor = scrollY > 0 ? this.zoomStep : 1 / this.zoomStep;
        const old = this.bg.scale;
        let next = cc.misc.clampf(old * factor, this.minScale, this.maxScale);
        factor = next / old;
        if (factor === 1) return;

        if (!this.zoomAtCursor) { this.bg.scale = next; return; }

        const screen = e.getLocation();
        const localBefore = this.bg.convertToNodeSpaceAR(screen);
        this.bg.scale = next;
        const worldAfter = this.bg.convertToWorldSpaceAR(localBefore);
        const deltaWorld = screen.sub(worldAfter);

        const parent = this.bg.parent;
        const p0 = parent.convertToNodeSpaceAR(worldAfter);
        const p1 = parent.convertToNodeSpaceAR(worldAfter.add(deltaWorld));
        const deltaInParent = p1.sub(p0);
        this.bg.position = this.bg.position.add(deltaInParent);
    }


    private _path: cc.Vec2[] = [];
    protected onTouchStart(e: cc.Event.EventTouch) {
        const p = this.uiToLocal(e);
        // 检查触摸点是否在画布内，如果超出则不绘制
        this._drawing = true;
        this._lastPt = p.clone();

        this._path.length = 0;
        this._path.push(p);

        // 起笔补圆避免断头（可保留）
        this.work.clear();           // ✅ 预览层每次重绘前先清
        this.work.circle(p.x, p.y, this.work.lineWidth * 0.5);
        this.work.fill();
    }

    private uiToLocal(event: cc.Event.EventTouch): cc.Vec2 {
        const p = event.getLocation();
        return this.bg.convertToNodeSpaceAR(p); // Work 与 HistorySprite 都在 bg 下，坐标统一
    }


    protected onTouchMove(e: cc.Event.EventTouch) {
        if (!this._drawing) return;
        const p = this.uiToLocal(e);
        if (this._lastPt && p.sub(this._lastPt).mag() < this.minSegment) return;
        this._path.push(p);
        this._lastPt = p;

        // ✅ 重绘一条路径（每帧 1 次 DC）
        this.work.clear();
        const first = this._path[0];
        this.work.moveTo(first.x, first.y);
        for (let i = 1; i < this._path.length; i++) {
            const pt = this._path[i];
            this.work.lineTo(pt.x, pt.y);
        }
        this.work.stroke();          // ✅ 只保留当前这一条预览线
    }

    onTouchEnd() {
        if (!this._drawing) return;
        this._drawing = false;
        // ✅ 一次性烘焙到 RT（history）
        this.commitWork();
        // ✅ 清掉预览
        this.work.clear();
        this._path.length = 0;
        this._lastPt = null;
    }


    /** 确保烘焙相机和 bg 对齐 */
    private fitBakeCamToBG() {
        if (!this.camera) return;
        this.camera.alignWithScreen = false;
        this.camera.node.position = new cc.Vec3(this.bg.position.x, this.bg.position.y, 1);
        this.camera.orthoSize = (this.bg.height * 0.5) * this.bg.scaleY;
    }


    // ===================== 烘焙：把 Work 累计进历史纹理（仅渲染 work 组） =====================
    private commitWork() {
        if (!this.camera || !this._rt || !this.work) return;
        this.fitBakeCamToBG();
        this.getPixelData();
        this.work.clear();
    }


    // 获取像素数据
    private _historyRT: cc.RenderTexture = null;
    getPixelData() {
        const width = this._rt.width;
        const height = this._rt.height;
        const pixelData = new Uint8Array(width * height * 4); // RGBA 数据
        // 使用 renderTexture 获取像素数据
        this.camera.targetTexture = this._rt;
        this.camera.clearFlags = this._NO_COLOR;
        this.camera.backgroundColor = cc.color(0, 0, 0, 0);
        this.camera.alignWithScreen = false;
        this.camera.render();

        let result = this.filpYImage(this._rt.readPixels(), this._rt.width, this._rt.height);

        if (!this._historyRT) {
            this._historyRT = new cc.RenderTexture();
        }

        this._historyRT.initWithData(result, cc.Texture2D.PixelFormat.RGBA8888, width, height);
        const sf = new cc.SpriteFrame();
        (sf as any).setTexture ? (sf as any).setTexture(this._historyRT) : (sf as any)._setTexture(this._historyRT);
        this.historySprite.spriteFrame = sf;
        // pixelData 就是你绘制的内容的像素数据，可以进行后续处理
        // console.log(result);
    }

    // 像素y坐标翻转
    filpYImage(data, width, height): Uint8Array {
        // create the data array
        let picData = new Uint8Array(width * height * 4);
        let rowBytes = width * 4;
        for (let row = 0; row < height; row++) {
            let srow = height - 1 - row;
            let start = srow * width * 4;
            let reStart = row * width * 4;
            // save the piexls data
            for (let i = 0; i < rowBytes; i++) {
                picData[reStart + i] = data[start + i];
            }
        }
        return picData;
    }


    // ===================== 初始化：历史RT & Sprite =====================
    private initHistoryRT() {
        const w = Math.max(1, Math.round(this.bg.width));
        const h = Math.max(1, Math.round(this.bg.height));
        this._rt = new cc.RenderTexture();
        // 2.4.x
        this._rt.initWithSize(w, h);
    }

}
