import { _decorator, CCFloat, Component, Enum, Material, math, sp, Sprite, Texture2D, UITransform, v2, Vec2, Vec4 } from "cc";
import { EDITOR } from "cc/env";

const { ccclass, property, executeInEditMode } = _decorator;

enum CropType {
    /**矩形 */
    RECT = 0,
    /**圆形 */
    CIRCLE = 1,
    /**图片 */
    IMAGE_ALPHA
}

const ECropType = Enum({
    ["矩形"]: CropType.RECT,
    ["圆形/椭圆"]: CropType.CIRCLE,
    ["图片"]: CropType.IMAGE_ALPHA,
})

@ccclass
@executeInEditMode
export class CropEffect extends Component {
    @property({
        type: ECropType,
        tooltip: '裁剪类型: 矩形/圆形/图片Alpha'
    })
    type: CropType = CropType.RECT;

    @property({
        type: Material,
        tooltip: '使用的材质',
        visible() { return this._material !== null; }
    })
    get material() { return this._material; }
    set material(value) {
        this._material = value;
        this.updateMaterial();
    }
    private _material: Material = null;

    @property({
        type: Texture2D,
        tooltip: '遮罩纹理 (用于图片Alpha裁剪)',
        visible() { return this.type === CropType.IMAGE_ALPHA; }
    })
    maskTexture: Texture2D = null;

    @property({
        type: CCFloat,
        tooltip: 'Alpha阈值 (用于图片Alpha裁剪)',
        range: [0, 1, 0.1],
        visible() { return this.type === CropType.IMAGE_ALPHA; }
    })
    alphaThreshold: number = 0.5;

    @property({
        type: CCFloat,
        tooltip: '边缘的模糊程度',
        range: [0, 1, 0.01]
    })
    blur: number = 0.5;

    @property({
        // type: CCFloat,
        tooltip: '中心点',
        range: [0, 1, 0.1]
    })
    center: Vec2 = new Vec2(0.5, 0.5);

    @property({
        tooltip: '显示宽高',
    })
    size: math.Size = math.size(0, 0);

    // @property({
    //     tooltip: '旋转角度',
    //     range: [0, 1, 0.1],
    //     visible() { return this.type === CropType.RECT || this.type === CropType.CIRCLE; }
    // })
    // angle: number = 0;

    protected onLoad() {
        // return;
        if (!this.material) {
            const sprite = this.getComponent(Sprite);
            if (sprite) {
                this.material = sprite.getSharedMaterial(0);
            }
        }else{
            this.updateMaterial();
        }
        // this.updateMaterial();
    }
    
    protected onEnable(): void {
        // this.updateMaterial();
    }

    update() {
        if (EDITOR) {
            this.updateMaterial();
        }
    }

    private updateMaterial() {
        if (!this.material) return;
        // 设置裁剪类型
        this.material.setProperty('cropType', this.type);
        this.material.setProperty("blur", this.blur);/*  */
        this.material.setProperty("center", this.center);
        let UITransNode: UITransform = this.node.getComponent(UITransform);
        // 设置参数
        switch (this.type) {
            case CropType.RECT:
                this.material.setProperty('size', v2(this.size.width / UITransNode.width,this.size.height / UITransNode.height));
                break;
            case CropType.CIRCLE:
                this.material.setProperty('radiusX', this.size.width / UITransNode.width / 2);
                this.material.setProperty('radiusY', this.size.height / UITransNode.height / 2);
                break;
            case CropType.IMAGE_ALPHA:
                this.material.setProperty('alphaThreshold', this.alphaThreshold);
                this.material.setProperty('texture_mask', this.maskTexture);
                break;
        }
    }
}