import GlobalPool from "../../Script/Common/GlobalPool";
import yyComponent from "../../Script/Common/yyComponent";
import { EventType } from "../../Script/GameSpecial/GameEventType";
import DrawingBoard from "../../Script/Utils/DrawingBoard";
import List, { ListNode } from "../../Script/Utils/List";
import PolygonSplitProgram from "../../Script/Utils/PolygonSplitProgram";
import TileSprite from "./TileSprite";

const { ccclass, property } = cc._decorator;
/**可炸毁的物体 */
@ccclass
export default class Tile extends yyComponent {

    start() {
        this.init();
    }

    protected inited: boolean = false;
    public init(data?) {
        if (!this.inited) {
            this.inited = true;
            this.initAssembler();
            this.initCollider();
            this.initBorderColor();
            this.initDrawBoard();
            this.initRigid();
            this.onEvents();
        }

        if (undefined !== data) {
            this.setData(data);
        }
    }
    protected onEvents() {
        console.error("需定义关卡事件：在指定位置爆炸");
        this.on(EventType.LevelEvent.baoZha, this.onBaoZha, this);
    }
    protected setData(data: { p: cc.Vec2, angle: number, drawBoardData?: ArrayBuffer, pointList?: PointList }) {
        if (undefined !== data.p) {
            this.setPosition(data.p);
        }
        if (undefined !== data.angle) {
            this.node.angle = data.angle;
            this.node.getComponent(cc.RigidBody).syncRotation(false);
        }

        if (undefined !== data.drawBoardData) {
            this.drawBoard.setData(data.drawBoardData);
            this.synDrawToMat();
        }
        if (undefined !== data.pointList) {
            this.pointList = data.pointList;
        }
        this.synCollider();
        this.setRenderPoints();
    }
    public setPosition(p) {
        this.node.setPosition(p);
        this.node.getComponent(cc.RigidBody).syncPosition(false);
    }
    public reset() {
        this.resetCollider();
        this.resetDrawBoard();
        this.resetRigid();
    }

    public setDrawBoardData(data) {
        this.drawBoard.setData(data);
        this.synDrawToMat();
    }

    /**************************************************************对外功能**************************************************************/
    //#region 炸毁边缘痕迹效果设置
    /**设置炸毁边缘的痕迹的宽度，单位：像素 */
    public setBordWidth(w: number) {
        if (this._bordWidthPixel == w) return;
        if (w < 0) w = 0;
        this._bordWidthPixel = w;
        this.synBordSize();
    }
    /**设置炸毁边缘发亮时叠加的颜色 */
    public setBorderLightColor(c: cc.Color) {
        this.borderLightColor = c;
    }
    /**设置炸毁边缘变暗时叠加的颜色 */
    public setBorderDarkColor(c: cc.Color) {
        this.borderDarkColor = c;
    }
    //#endregion

    /**************************************************************管理数据**************************************************************/
    //#region 刚体
    protected initRigid() {
        let rid = this.node.getComponent(cc.RigidBody);
        rid.linearVelocity = cc.v2();
        rid.angularVelocity = 0;
    }
    protected resetRigid() {
        let rid = this.node.getComponent(cc.RigidBody);
        rid.linearVelocity = cc.v2();
        rid.angularVelocity = 0;
    }
    //#endregion

    //#region 碰撞体
    /**多边形碰撞组件 */
    protected collider: cc.PhysicsPolygonCollider = null;
    /**原始碰撞范围 */
    protected originalPoints: cc.Vec2[];
    /**当前多边形碰撞体的顶点数据，按逆时针顺序存储 */
    protected pointList: PointList;
    protected initCollider() {
        this.collider = this.node.getComponent(cc.PhysicsPolygonCollider);
        this.pointList = new List<cc.Vec2>();
        this.createColliderByImg();
        this.collider.points = this.originalPoints;
        this.collider.apply();
    }
    protected resetCollider() {
        this.collider.points = this.originalPoints;
        this.collider.apply();
        this.pointList.clear();
        let points = this.originalPoints;
        for (let i = 0, c = points.length; i < c; ++i) {
            this.pointList.push(points[i]);
        }
    }
    /**通过节点显示的图片创建多边形 */
    protected createColliderByImg() {
        let size = this.img.spriteFrame.getOriginalSize();
        if (!this.originalPoints) {
            this.originalPoints = [];
            this.originalPoints.push(cc.v2(0, 0));
            this.originalPoints.push(cc.v2(size.width, 0));
            this.originalPoints.push(cc.v2(size.width, size.height));
            this.originalPoints.push(cc.v2(0, size.height));
        } else {
            this.originalPoints[1].x = size.width;
            this.originalPoints[2].x = size.width;
            this.originalPoints[2].y = size.height;
            this.originalPoints[3].y = size.height;
        }

        this.pointList.clear();
        let points = this.originalPoints;
        for (let i = 0, c = points.length; i < c; ++i) {
            this.pointList.push(points[i]);
        }
    }

    /**
     * 将多边形剪掉一个圆
     * @param center 圆心位置，节点局部坐标
     * @param radius 圆的半径
     */
    protected cutCircle(center: cc.Vec2, radius: number) {
        //创建圆的顶点
        let circle = this.createCirclePoints(center, radius);
        //裁剪多边形
        let cross = this.createPointsByCross(this.pointList, this.collider.points, circle, center, radius);

        if (cross) {
            this.synCollider();
            this.setRenderPoints();
        }
    }
    /**
     * 创建圆的碰撞顶点数据
     * @param center 圆的坐标
     * @param radius 圆的半径
     */
    protected createCirclePoints(center: cc.Vec2, radius: number): PointList {
        let circlePoints: PointList = new List<cc.Vec2>();
        let d = radius / 3;
        if (d > 15) {
            d = 15;
        } else if (d < 10) {
            d = 10;
        }
        let step = d / radius;//每个顶点跨越的弧度
        for (let r = 0; r < 6.28; r += step) {
            let p = cc.v2();
            p.x = Math.cos(r) * radius + center.x;
            p.y = Math.sin(r) * radius + center.y;
            p["radian"] = r;
            circlePoints.unshift(p);
        }
        return circlePoints;
    }


    //裁剪掉在圆形内部的顶点，生成新的顶点数据
    protected createPointsByCross(polygon: PointList, points: cc.Vec2[], circle: PointList, center: cc.Vec2, radius: number) {
        //从外部进入圆周的交点
        let enterCross: {
            crossNode: Point,
            circleNode: Point,
            polygonNode: Point,
        }[] = [];
        //从圆周内部穿出的交点
        let outCross: {
            crossNode: Point,
            circleNode: Point,
            polygonNode: Point,
        }[] = [];
        let node = polygon.first;
        let dis = radius * radius;
        //第一个交点是否处于圆周内
        let firstNodeResult: boolean = this.checkPointInCircle(node.value, center, dis);
        //上一个节点是否处于圆周内
        let preResult: boolean = firstNodeResult;
        //计算所有多边形顶点中处于圆周内部的顶点所在线段与圆周的交点
        node = node.next;
        while (node) {
            let curResult = this.checkPointInCircle(node.value, center, dis);
            if (curResult !== preResult) {
                this.getCrossData(polygon, node, node.prev, circle, center, radius, enterCross, outCross, preResult);
            } else if (!preResult) {
                //直线与圆相交，两个端点都在圆外面的情况
                let arr = this.getLineCrossCirclePoint(node.value, node.prev.value, center, radius);
                if (arr.length == 2) {
                    let p = arr[0];
                    if (!p || !this.checkPointBetweenLine(p, node.value, node.prev.value)) {
                        p = arr[1];
                    }
                    if (!p || !this.checkPointBetweenLine(p, node.value, node.prev.value)) {
                        p = null;
                    }
                    if (!!p) {
                        let x = p.x - node.value.x;
                        let y = p.y - node.value.y;
                        let d1 = x * x + y * y;
                        let p2 = arr[1];
                        x = p2.x - node.value.x;
                        y = p2.y - node.value.y;
                        let d2 = x * x + y * y;
                        if (d1 < d2) {
                            p = arr[1];
                            p2 = arr[0];
                        }
                        //进入交点：
                        this.getRadian(p, center);
                        this.addCrossNode(polygon, node, circle, p, enterCross);
                        //穿出交点：
                        this.getRadian(p2, center);
                        this.addCrossNode(polygon, node, circle, p2, outCross);
                    }
                } else if (arr.length == 1) {

                }
            }
            preResult = curResult;
            node = node.next;
        }
        if (preResult != firstNodeResult) {
            this.getCrossData(polygon, polygon.first, polygon.last, circle, center, radius, enterCross, outCross, preResult);
        } else if (!!firstNodeResult && !!preResult && outCross.length == 0) {
            //全部顶点都在圆周内部
            GlobalPool.put(this.node);
            return;
        }

        //排错：
        if (enterCross.length != outCross.length) {
            return false;
        }
        //多边形所有顶点都不在圆周内部的情况
        if (enterCross.length == 0) {
            return this.crossCircleInPolygon(polygon, points, circle, center, radius);
        }
        circle.last.next = circle.first;
        circle.first.prev = circle.last;
        polygon.last.next = polygon.first;
        polygon.first.prev = polygon.last;
        this.linkCrossNode(polygon, circle, enterCross, outCross);
        this.processCrossData(enterCross, outCross);
        return true;
    }
    /**点是否在圆内 */
    protected checkPointInCircle(p: cc.Vec2, center: cc.Vec2, radiusSqr: number) {
        let x = p.x - center.x;
        let y = p.y - center.y;
        return x * x + y * y < radiusSqr;
    }
    /**获取多边形与圆的交点信息 */
    protected getCrossData(polygon: PointList, polygonCurNode: Point, polygonPreNode: Point, circle: PointList, center: cc.Vec2, radius: number, enterCross: any[], outCross: any[], preResult: boolean) {
        //计算直线与圆的交点坐标
        let arr = this.getLineCrossCirclePoint(polygonCurNode.value, polygonPreNode.value, center, radius);
        if (arr.length == 0) {
            return;
        }
        let p = arr[0];
        if (!p || !this.checkPointBetweenLine(p, polygonCurNode.value, polygonPreNode.value)) {
            p = arr[1];
        }
        if (!p || !this.checkPointBetweenLine(p, polygonCurNode.value, polygonPreNode.value)) {
            return;
        }
        this.getRadian(p, center);
        if (preResult) {
            this.addCrossNode(polygon, polygonCurNode, circle, p, outCross);
        } else {
            this.addCrossNode(polygon, polygonCurNode, circle, p, enterCross);
        }
    }
    /**
     * 在多边形与圆周上添加交点，并记录交点信息
     * @param polygon 多边形
     * @param polygonCurNode 多边形链表上当前检测到的与圆周相交的点
     * @param circle    圆周
     * @param crossPos  多边形与圆周的交点坐标，需包含该点到圆心与X轴夹角的弧度
     * @param crossRecord 记录交点信息的数组
     */
    protected addCrossNode(polygon: PointList, polygonCurNode: Point, circle: PointList, crossPos: cc.Vec2, crossRecord: any[]) {
        //交叉点
        let crossNode = new ListNode<cc.Vec2>(crossPos);
        //圆周链表下一个点
        let circleNode = this.getNextCirclePoint(crossNode, circle);
        //圆周链表新增的点
        let circleCrossNode = new ListNode<cc.Vec2>(crossPos.clone());
        circleCrossNode.value["radian"] = crossNode.value["radian"];
        circle.insertNodeBefore(circleNode, circleCrossNode);
        //多边形新增的点
        let polygonCrossNode = new ListNode<cc.Vec2>(crossPos.clone());
        polygon.insertNodeBefore(polygonCurNode, polygonCrossNode);
        //添加交点记录
        crossRecord.push({
            crossNode: crossNode,
            circleNode: circleCrossNode,
            polygonNode: polygonCrossNode,
        });
    }
    /**检测点是否在线段的中间 */
    protected checkPointBetweenLine(p: cc.Vec2, p1: cc.Vec2, p2: cc.Vec2) {
        return !((p.x < p1.x && p.x < p2.x)
            || (p.x > p1.x && p.x > p2.x)
            || (p.y < p1.y && p.y < p2.y)
            || (p.y > p1.y && p.y > p2.y));
    }
    /**直线与圆的交点 */
    protected getLineCrossCirclePoint(p1: cc.Vec2, p2: cc.Vec2, center: cc.Vec2, radius: number) {
        if (p1.x == p2.x) {
            if (Math.abs(p1.x - center.x) > radius) {
                return [];
            }
            let x = p1.x - center.x;
            let y = Math.sqrt(radius * radius - x * x);
            if (y > 0) {
                return [
                    cc.v2(p1.x, center.y - y),
                    cc.v2(p1.x, center.y + y),
                ];
            } else {
                return [cc.v2(p1.x, center.y)];
            }
        }
        if (p1.y == p2.y) {
            if (Math.abs(p1.y - center.y) > radius) {
                return [];
            }
            let y = p1.y - center.y;
            let x = Math.sqrt(radius * radius - y * y);
            if (x > 0) {
                return [
                    cc.v2(center.x - x, p1.y),
                    cc.v2(center.x + x, p1.y),
                ];
            } else {
                return [cc.v2(center.x, p1.y)];
            }
        }
        let x1 = p1.x - center.x;
        let y1 = p1.y - center.y;
        if (x1 * x1 + y1 * y1 == radius * radius) {
            return [cc.v2(p1.x, p1.y)];
        }
        let x2 = p2.x - center.x;
        let y2 = p2.y - center.y;
        if (x2 * x2 + y2 * y2 == radius * radius) {
            return [cc.v2(p2.x, p2.y)];
        }

        let k = (y1 - y2) / (x1 - x2);
        let o = y1 - k * x1;
        let a = k * k + 1;
        let b = 2 * k * o;
        let c = o * o - radius * radius;
        let sqrt = b * b - 4 * a * c;
        if (sqrt < 0) {
            return [];
        }
        sqrt = Math.sqrt(sqrt);
        a *= 2;
        b /= -a;
        sqrt /= a;
        let result1 = b + sqrt;
        let result2 = b - sqrt;
        if (sqrt == 0) {
            return [cc.v2(result1 + center.x, k * result1 + o + center.y)];
        } else {
            return [cc.v2(result1 + center.x, k * result1 + o + center.y), cc.v2(result2 + center.x, k * result2 + o + center.y)];
        }
    }
    /**获取交点在圆周上对应的下一个节点 */
    protected getNextCirclePoint(node: Point, circle: PointList) {
        let r = node.value["radian"];
        let point = circle.first;
        if (r > point.value["radian"]) {
            return point;
        }
        while (point) {
            if (r <= point.value["radian"]) {
                let n = point.next;
                if (!n) {
                    n = circle.first;
                }
                if (r > n.value["radian"]) {
                    return n;
                }
            }
            point = point.next;
        }
        console.error("未找到圆周上的下一个点");
        debugger;
        return point;
    }
    /**
     * 计算从center到p的连线与X轴的夹角弧度
     * @param p
     * @param center
     */
    protected getRadian(p: cc.Vec2, center: cc.Vec2): number {
        let a = this._computeRadian(p, center);
        p["radian"] = a;
        return a;
    }
    protected _computeRadian(p: cc.Vec2, center: cc.Vec2): number {
        let x = p.x - center.x;
        if (x === 0) {
            return p.y > center.y ? 1.57 : 4.71;
        }
        let y = p.y - center.y;
        if (y === 0) {
            return x > 0 ? 0 : 3.14;
        }
        let a = Math.atan(Math.abs(y / x));
        if (x > 0) {
            if (y > 0) {
                return a;
            } else {
                return 6.28 - a;
            }
        } else {
            if (y > 0) {
                return 3.14 - a;
            } else {
                return 3.14 + a;
            }
        }
    }
    /**计算两条直线的交点 */
    protected getCrossPos(a1, a2, b1, b2) {
        let ax = a1.x - a2.x;
        let ay = a1.y - a2.y;
        let bx = b1.x - b2.x;
        let by = b1.y - b2.y;

        if (ax == 0) {
            let x = a1.x;
            let y = b2.y + by * (x - b2.x) / bx;
            return cc.v2(x, y);
        }
        if (bx == 0) {
            let x = b1.x;
            let y = a2.y + ay * (x - a2.x) / ax;
            return cc.v2(x, y);
        }

        let ak = ay / ax;
        let ab = a1.y - ak * a1.x;

        let bk = by / bx;
        let bb = b1.y - bk * b1.x;

        let x = (bb - ab) / (ak - bk);
        let y = ak * x + ab;
        return cc.v2(x, y);
    }

    //裁剪掉圆周上的在多边形中的顶点
    protected crossCircleInPolygon(polygon: PointList, points: cc.Vec2[], circle: PointList, center: cc.Vec2, radius: number) {
        //从外部进入多边形的交点
        let enterCross: {
            crossNode: Point,
            circleNode: Point,
            polygonNode: Point,
        }[] = [];
        //从多边形内部穿出的交点
        let outCross: {
            crossNode: Point,
            circleNode: Point,
            polygonNode: Point,
        }[] = [];
        let node = circle.first;
        let dis = radius * radius;
        //第一个交点是否处于多边形内
        let firstNodeResult: boolean = cc.Intersection.pointInPolygon(node.value, points);
        //上一个节点是否处于多边形内
        let preResult: boolean = firstNodeResult;
        //计算所有圆周顶点中处于多边形内部的顶点所在线段与圆周的交点
        node = node.next;
        while (node) {
            let curResult = cc.Intersection.pointInPolygon(node.value, points);
            if (curResult !== preResult) {
                this.getCircleCrossData(circle, center, radius, node, node.prev, polygon, enterCross, outCross, preResult);
            }
            preResult = curResult;
            node = node.next;
        }
        if (preResult != firstNodeResult) {
            this.getCircleCrossData(circle, center, radius, circle.first, circle.last, polygon, enterCross, outCross, preResult);
        }

        //排错：
        if (enterCross.length != outCross.length) {
            return false;
        }

        if (enterCross.length == 0) {
            return false;
        }
        circle.last.next = circle.first;
        circle.first.prev = circle.last;
        polygon.last.next = polygon.first;
        polygon.first.prev = polygon.last;
        this.linkCrossNode(polygon, circle, enterCross, outCross);
        this.processCrossData(enterCross, outCross);
        return true;
    }
    /**圆与多边形的交点信息 */
    protected getCircleCrossData(circle: PointList, center: cc.Vec2, radius: number, p1: Point, p2: Point, polygon: PointList, enterCross: any[], outCross: any[], preResult: boolean) {
        let n1 = polygon.last;
        let n2 = polygon.first;
        while (n2) {
            let cross = cc.Intersection.lineLine(p1.value, p2.value, n1.value, n2.value);
            if (cross) {
                let p = this.getCrossPos(p1.value, p2.value, n1.value, n2.value);
                this.getRadian(p, center);
                if (preResult) {
                    this.addCrossNode(polygon, n2, circle, p, outCross);
                } else {
                    this.addCrossNode(polygon, n2, circle, p, enterCross);
                }
                return;
            }
            n1 = n2;
            n2 = n2.next;
        }
    }


    /**连接交点与多边形顶点 */
    protected linkCrossNode(polygon: PointList, circle: PointList, enterCross: any[], outCross: any[]) {
        let count = enterCross.length;
        for (let i = 0; i < count; ++i) {
            //进入交点
            let data = enterCross[i];
            let pNext = data.polygonNode.next;
            let cNext = data.circleNode.next;
            pNext.prev = data.circleNode;
            data.circleNode.next = pNext;
            cNext.prev = data.polygonNode;
            data.polygonNode.next = cNext;
            //退出交点
            data = outCross[i];
            let pPrev = data.polygonNode.prev;
            let cPrev = data.circleNode.prev;
            pPrev.next = data.circleNode;
            data.circleNode.prev = pPrev;
            cPrev.next = data.polygonNode;
            data.polygonNode.prev = cPrev;
        }
    }

    /**根据交叉点创建新的多边形 */
    protected processCrossData(enterCross: any[], outCross: any[]) {
        let allList: List<cc.Vec2>[] = [];
        for (let i = 0; i < outCross.length; ++i) {
            let firstNode = outCross[i].polygonNode;
            let b = false;
            for (let j = 0; j < allList.length; ++j) {
                let li = allList[j];
                let n = li.first;
                while (n) {
                    if (n === firstNode) {
                        b = true;
                        break;
                    }
                    n = n.next;
                }
                if (b) {
                    break;
                }
            }
            if (b) {
                continue;
            }
            let lastNode = firstNode.prev;
            lastNode.next = null;
            firstNode.prev = null;
            let l = new List<cc.Vec2>();
            l.insertNode(firstNode, 0);
            allList.push(l);
        }
        let len = 0;
        let index = 0;
        for (let i = 0; i < allList.length; ++i) {
            if (len < allList[i].length) {
                len = allList[i].length;
                index = i;
            }
        }
        this.removeRepeatPoint(allList[index]);
        this.pointList = allList[index];
        allList.splice(index, 1);
        for (let i = 0; i < allList.length; ++i) {
            this.removeRepeatPoint(allList[i]);
            this.createNewCollider(allList[i]);
        }
    }
    /**根据分割的数据创建一个新的节点 */
    protected createNewCollider(pointList: PointList) {
        //贴图
        let img = this.img.spriteFrame;
        //绘制的遮罩
        let drawBoardData = this.drawBoard.copyData();
        let p = this.node.getPosition();
        let angle = this.node.angle;
        let data = {
            p: p,
            angle: angle,
            img: img,
            drawBoardData: drawBoardData,
            pointList: pointList,
        };
        let node = GlobalPool.get(this.node.name, data);
        this.node.parent.addChild(node);
    }

    protected removeRepeatPoint(pointList: PointList) {
        let node = pointList.first;
        while (node && node.next) {
            let p1 = node.value;
            let p2 = node.next.value;
            node = node.next;
            if (p1.x == p2.x && p1.y == p2.y) {
                pointList.removeNode(node.prev);
            }
        }
    }

    /**将多边形顶点数据应用到多边形碰撞体 */
    protected synCollider() {
        let points = [];
        let node = this.pointList.first;
        points.push(node.value);
        node = node.next;
        let size = this.img.spriteFrame.getOriginalSize();
        let index = 0;
        while (node) {
            if (!!node.next
                && node.value.x != 0 && node.value.x != size.width
                && node.value.y != 0 && node.value.y != size.height) {
                let r = this.checkPointOnLine(points[index], node.value, node.next.value);
                if (!r) {
                    r = this.checkPointNear(points[index], node.value);
                }
                if (!r) {
                    points.push(node.value);
                    index++;
                } else {
                    //移除顶点
                    // this.pointList.removeNode(node);
                    node.prev.next = node.next;
                    node.next.prev = node.prev;
                    this.pointList.addLength(-1);
                }
            } else {
                points.push(node.value);
                index++;
            }
            node = node.next;
        }

        if (points.length !== this.pointList.length) {
            console.log("剔除顶点数量：", this.pointList.length - points.length);
        }
        this.collider.points = points;
        this.collider.apply();
    }

    /**检测两个顶点是否距离很近 */
    protected checkPointNear(p1, p2) {
        let x = p1.x - p2.x;
        let y = p1.y - p2.y;
        return x * x + y * y < 81;
    }

    /**检测三个顶点是否在同一直线上 */
    protected checkPointOnLine(p1: cc.Vec2, p2: cc.Vec2, p3: cc.Vec2) {
        let o = 1;
        let x1 = p1.x - p2.x;
        if (x1 > -o || x1 < o) {
            return false;
        }
        let x2 = p2.x - p3.x;
        if (x2 > -o || x2 < o) {
            return false;
        }
        let y1 = p1.y - p2.y;
        // let k1 = y1 / x1;
        let y2 = p2.y - p3.y;
        // let k2 = y2 / x2;
        let a1 = Math.atan(y1 / x1);
        let a2 = Math.atan(y2 / x2);
        return Math.abs(a1 - a2) < 0.17453;
        // return Math.abs(k1 - k2) < o;
    }

    //#endregion

    /**************************************************************渲染效果**************************************************************/
    //#region 炸毁渲染
    protected img: TileSprite = null;
    protected initAssembler() {
        this.img = this.node.getComponent(TileSprite);
        this.img.initAssembler();
    }

    /**将绘图板数据转换给材质球的贴图资源 */
    protected tex: cc.RenderTexture;
    /**贴图的材质球 */
    protected mat: cc.Material = null;
    /**绘图板 */
    protected drawBoard: DrawingBoard = null;
    /**默认绘图颜色分量 */
    protected drawColor: number;
    /**被炸之前的绘图板上的像素个数 */
    protected prePixelCount: number = 0;

    protected initDrawBoard() {
        //画板
        let size = this.img.spriteFrame.getOriginalSize();
        this.drawBoard = new DrawingBoard(size.width, size.height);
        this.drawColor = 255;
        this.drawBoard.setColor(this.drawColor, this.drawColor, this.drawColor, this.drawColor);
        //遮罩贴图
        this.tex = new cc.RenderTexture();
        this.tex.initWithSize(size.width, size.height);
        //材质
        this.mat = this.img.getMaterial(0);
        this.mat.setProperty("mask", this.tex);
        this.synBordSize();
        //状态
        this.prePixelCount = 0;
    }
    protected resetDrawBoard() {
        this.drawBoard.reset();
        this.prePixelCount = 0;
        this.synDrawToMat();
    }

    /**将炸毁边缘的UV尺寸应用到材质球 */
    protected synBordSize() {
        if (!this.img) {
            this.img = this.node.getComponent(TileSprite);
        }
        if (!this.mat) {
            this.mat = this.img.getMaterial(0);
        }
        let size = this.img.spriteFrame.getOriginalSize();
        let x = this.bordWidth / size.width;
        let y = this.bordWidth / size.height;
        let bordSize = this.mat.getProperty("bordSize", 0);
        bordSize[0] = x;
        bordSize[1] = y;
        bordSize[2] = 1;
        bordSize[3] = 1;
        this.mat.setProperty("bordSize", bordSize);
    }

    /**设置渲染组件需要的顶点数据 */
    protected setRenderPoints() {
        let points = this.pointList.convertToArray();
        let frag = PolygonSplitProgram.process(points);
        this.img.setPoints(points, frag);
    }
    /**
     * 绘制一个圆，绘制区域将不显示图案；区域边缘将显示炸毁效果
     * @param center 圆心位置，节点局部坐标
     * @param radius 圆的半径
     */
    protected drawCircle(center: cc.Vec2, radius: number) {
        this.drawBoard.circle(center.x, this.drawBoard.height - center.y, radius);
        let c = this.drawBoard.getColorCount(this.drawColor, this.drawColor, this.drawColor, this.drawColor);
        if (c != this.prePixelCount) {
            this.prePixelCount = c;
            this.synDrawToMat();
            this.cutCircle(center, radius);
        }
    }
    /**将绘制的图案应用到材质球 */
    protected synDrawToMat() {
        let data = this.drawBoard.getData();
        this.tex.initWithData(data, cc.Texture2D.PixelFormat.RGBA8888, this.drawBoard.width, this.drawBoard.height);
        this.mat.setProperty("mask", this.tex, 0, true);
    }
    //#endregion

    //#region 炸毁效果
    @property
    protected _borderLightColor: cc.Color = cc.Color.WHITE;
    @property({
        tooltip: "炸毁边缘变亮时叠加的颜色",
    })
    protected get borderLightColor() { return this._borderLightColor; }
    protected set borderLightColor(v) {
        this._borderLightColor = v;
        if (!!this.img) {
            let mat = this.img.getMaterial(0);
            let d = [this._borderLightColor.getR() / 255, this._borderLightColor.getG() / 255, this._borderLightColor.getB() / 255, this._borderLightColor.getA() / 255];
            mat.setProperty("borderLightColor", d);
        }
    }
    @property
    protected _borderDarkColor: cc.Color = cc.Color.WHITE;
    @property({
        tooltip: "炸毁边缘变暗时叠加的颜色",
    })
    protected get borderDarkColor() { return this._borderDarkColor; }
    protected set borderDarkColor(v) {
        this._borderDarkColor = v;
        if (!!this.img) {
            let mat = this.img.getMaterial(0);
            let d = [this._borderDarkColor.getR() / 255, this._borderDarkColor.getG() / 255, this._borderDarkColor.getB() / 255, this._borderDarkColor.getA() / 255];
            mat.setProperty("borderDarkColor", d);
        }
    }
    /**炸毁边缘的宽度，单位：像素 */
    @property
    protected _bordWidthPixel: number = 10;
    @property({
        tooltip: "炸毁边缘痕迹的宽度，单位：像素"
    })
    protected get bordWidth() { return this._bordWidthPixel; }
    protected set bordWidth(v) {
        this.setBordWidth(v);
    }
    protected initBorderColor() {
        let mat = this.mat;
        if (!mat) {
            mat = this.img.getMaterial(0);
        }
        let d = [this._borderLightColor.getR() / 255, this._borderLightColor.getG() / 255, this._borderLightColor.getB() / 255, this._borderLightColor.getA() / 255];
        mat.setProperty("borderLightColor", d);
        d = [this._borderDarkColor.getR() / 255, this._borderDarkColor.getG() / 255, this._borderDarkColor.getB() / 255, this._borderDarkColor.getA() / 255];
        mat.setProperty("borderDarkColor", d);
    }
    //#endregion

    /**************************************************************事件回调**************************************************************/
    /**
     * 响应爆炸事件
     * @param [data.p]         爆炸位置，世界坐标系
     * @param [data.radius]    爆炸半径
     */
    protected onBaoZha(data: { p: cc.Vec2, radius: number }) {
        let pos = this.node.convertToNodeSpaceAR(data.p);
        this.drawCircle(pos, data.radius);
    }
}

/**顶点链表 */
type PointList = List<cc.Vec2>;
/**顶点节点 */
type Point = ListNode<cc.Vec2>;

