const gfx = cc.gfx;
var vfmt = new gfx.VertexFormat([
    { name: gfx.ATTR_POSITION, type: gfx.ATTR_TYPE_FLOAT32, num: 2 },
    { name: gfx.ATTR_UV0, type: gfx.ATTR_TYPE_FLOAT32, num: 2 },
    { name: gfx.ATTR_COLOR, type: gfx.ATTR_TYPE_UINT8, num: 4, normalize: true },
]);

const { ccclass, property } = cc._decorator;
/**可炸毁的图块渲染组件 */
@ccclass
export default class TileSprite extends cc.RenderComponent {

    @property(cc.SpriteFrame)
    protected _sf: cc.SpriteFrame = null;
    @property(cc.SpriteFrame)
    public get spriteFrame() { return this._sf; }
    public set spriteFrame(v) {
        this._sf = v;
        let mat = this.getMaterial(0);
        mat.setProperty("texture", this._sf.getTexture());
        let size = this._sf.getOriginalSize();
        this.node.width = size.width;
        this.node.height = size.height;
        this._assembler = null;
        TileAssembler.init(this);
        this._assembler.imgSize.x = size.width;
        this._assembler.imgSize.y = size.height;

        let p = [
            cc.v2(0, 0),
            cc.v2(size.width, 0),
            cc.v2(size.width, size.height),
            cc.v2(0, size.height),
        ];
        let frag = [0, 1, 2, 0, 2, 3];
        this.setPoints(p, frag);
    }

    __preload() {
        this.initAssembler();
    }

    _resetAssembler() {
        if (!this._assembler) {
            TileAssembler.init(this);
            let size = this.spriteFrame.getOriginalSize();
            this._assembler.imgSize.x = size.width;
            this._assembler.imgSize.y = size.height;
        }
        this.setVertsDirty();
    }


    protected _mat: cc.Material = null;
    protected get mat() {
        if (!this._mat) {
            this._mat = this.getMaterial(0);
        }
        return this._mat;
    }

    public initAssembler(data?: { points: cc.Vec2[], frag: number[] }) {
        let mat = this.getMaterial(0);
        mat.setProperty("texture", this.spriteFrame.getTexture());
        let size = this.spriteFrame.getOriginalSize();
        this.node.width = size.width;
        this.node.height = size.height;

        if (undefined !== data) {
            this.setPoints(data.points, data.frag);
        } else if (!this._assembler) {
            let size = this.spriteFrame.getOriginalSize();
            let p = [
                cc.v2(0, 0),
                cc.v2(size.width, 0),
                cc.v2(size.width, size.height),
                cc.v2(0, size.height),
            ];
            let frag = [0, 1, 2, 0, 2, 3];
            this.setPoints(p, frag);
        }
    }

    public setPoints(points: cc.Vec2[], frag: number[]) {
        this._resetAssembler();
        this._assembler.setPoints(points, frag);
    }

    protected synWorldMat() {
        let matrix = this.node._worldMatrix;
        this.mat.setProperty("worldMat", matrix);
    }
}

let vec3_temps = [];
for (let i = 0; i < 4; i++) {
    vec3_temps.push(cc.v3());
}
export class TileAssembler extends cc.Assembler {
    protected _renderData;
    protected floatsPerVert = 5;

    protected verticesCount = 4;
    protected indicesCount = 6;

    protected uvOffset = 2;
    protected colorOffset = 4;
    protected _local;

    public imgSize: cc.Vec2 = cc.v2(1, 1);

    constructor() {
        super();
        this._renderData = new cc.RenderData();
        this._renderData.init(this);

        this.initData();
    }

    get verticesFloats() {
        return this.verticesCount * this.floatsPerVert;
    }

    initData() {
        let data = this._renderData;
        data.createData(0, this.verticesFloats, this.indicesCount);
        this.initQuadIndices(data.iDatas[0]);
    }
    initQuadIndices(indices) {
        let count = indices.length / 6;
        for (let i = 0, idx = 0; i < count; i++) {
            let vertextID = i * 2;
            indices[idx++] = vertextID;
            indices[idx++] = vertextID + 1;
            indices[idx++] = vertextID + 2;
            indices[idx++] = vertextID + 1;
            indices[idx++] = vertextID + 3;
            indices[idx++] = vertextID + 2;
        }
    }

    public setPoints(data: { x: number, y: number }[], frag: number[]) {
        let points = data;
        this.verticesCount = points.length;
        this.indicesCount = frag.length;
        this._renderData.createData(0, this.verticesFloats, this.indicesCount);
        this._renderData.iDatas[0] = frag;

        let vData = this._renderData.vDatas[0];

        let floatsPerVert = this.floatsPerVert;
        let index = 0;
        let rateX = 1 / this.imgSize.x;
        let rateY = 1 / this.imgSize.y;
        for (let i = 0, c = points.length; i < c; i += 1) {
            vData[index] = points[i].x;
            vData[index + 1] = points[i].y;
            vData[index + 2] = points[i].x * rateX;
            vData[index + 3] = 1 - points[i].y * rateY;
            index += floatsPerVert;
        }
        let val = cc.Color.WHITE._val;
        let colors = this._renderData.uintVDatas[0];
        for (let i = 0; i < this.indicesCount; ++i) {
            colors[i * floatsPerVert + this.colorOffset] = val;
        }
    }

    getBuffer(renderer) {
        return cc.renderer._handle._meshBuffer;
    }
    getVfmt() {
        return vfmt;
    }

    fillBuffers(comp, renderer) {
        if (renderer.worldMatDirty) {
            comp.synWorldMat();
        }

        let renderData = this._renderData;
        let vData = renderData.vDatas[0];
        let iData = renderData.iDatas[0];

        let buffer = this.getBuffer(renderer);
        let offsetInfo = buffer.request(this.verticesCount, this.indicesCount);

        // buffer data may be realloc, need get reference after request.

        // fill vertices
        let vertexOffset = offsetInfo.byteOffset >> 2,
            vbuf = buffer._vData;

        if (vData.length + vertexOffset > vbuf.length) {
            vbuf.set(vData.subarray(0, vbuf.length - vertexOffset), vertexOffset);
        } else {
            vbuf.set(vData, vertexOffset);
        }

        // fill indices
        let ibuf = buffer._iData,
            indiceOffset = offsetInfo.indiceOffset,
            vertexId = offsetInfo.vertexOffset;
        for (let i = 0, l = iData.length; i < l; i++) {
            ibuf[indiceOffset++] = vertexId + iData[i];
        }
    }

    packToDynamicAtlas(comp, frame) {
        return;
    }
}

cc.Assembler.register(TileSprite, TileAssembler);
