import List, { ListNode } from "./List";
type Vec2 = {
    x: number,
    y: number
}
export default class PolygonSplitProgram {
    private static pList: List<PointStatus> = null;
    /**
     * 分割多边形
     * @param points 多边形顶点的坐标数组，按逆时针排列
     * @returns 分割后的三角形的顶点在points数组中的索引，每三个连续的顶点表示一个三角形，按逆时针排列
     */
    public static process(points: Vec2[]): number[] {
        if (points.length < 3) return [];
        //记录顶点状态
        this.pList = new List<PointStatus>();
        for (let i = 0, c = points.length; i < c; ++i) {
            this.pList.push(new PointStatus(i, points[i]));
        }
        let node = this.pList.first;
        //统计凹顶点数量
        let concaveCount = 0;
        while (!!node) {
            this.updatePointStatus(node);
            if (!node.value.isConvex) {
                concaveCount++;
            }
            node = node.next;
        }
        //凸多边形处理：
        if (concaveCount == 0) {
            let tri = this.processConvexPlygon(points);
            this.reset();
            return tri;
        }
        //只有一个凹顶点的多边形处理：
        if (concaveCount == 1) {
            //找到凹顶点在数组中的索引
            let index = 0;
            let node = this.pList.first;
            while (node.value.isConvex) {
                index++;
                node = node.next;
            }
            let tri = this.processSingleConcavePlygon(points, index);
            this.reset();
            return tri;
        }
        //有多个凹顶点的多边形处理
        // let tri = this.processMultilpeConcavePlygon(this.pList);
        let tri: number[] = [];
        node = this.pList.first;
        while (this.pList.length >= 3) {
            //从当前顶点往链表后面查找可分隔的顶点
            while (!!node && !node.value.isSeparable) {
                node = node.next;
            }
            //若直到链表尾部都不存在可分隔顶点，则从链表头部重新开始查找
            if (!node) {
                node = this.pList.first;
                while (!!node && !node.value.isSeparable) {
                    node = node.next;
                }
                if (!node) {
                    console.warn("多边形分割尚未未完成，但找不到可分割顶点:", this.pList);
                    break;
                }
            }
            //可分隔顶点与其相邻顶点组成一个三角形
            let prev = node.prev ? node.prev : this.pList.last;
            let next = node.next ? node.next : this.pList.first;
            tri.push(prev.value.pIndex, node.value.pIndex, next.value.pIndex);
            //将当前顶点移除，继续分隔剩余的顶点
            this.pList.removeNode(node);
            node.destroy();
            //更新当前顶点相邻的两个顶点的状态，其他顶点不受影响
            this.updatePointStatus(prev);
            this.updatePointStatus(next);
            node = next;
        }
        this.reset();
        return tri;
    }
    /**计算顶点的状态 */
    private static updatePointStatus(node: ListNode<PointStatus>) {
        let prev = node.prev ? node.prev : this.pList.last;
        let next = node.next ? node.next : this.pList.first;
        //顶点原有记录为凹顶点时，可能会因相邻顶点被移除而变为凸顶点
        if (!node.value.isConvex) {
            if (this.isConvex(node.value.point, prev.value.point, next.value.point)) {
                //变为凸顶点后，其与相邻顶点组成的三角形，若包含了其他凹顶点，则仍不能分割
                node.value.isConvex = true;
            } else {
                //仍为凹顶点时，不可分割
                node.value.isSeparable = false;
                return;
            }
        }
        //检测顶点与相邻点组成的三角形，是否包含了其他凹顶点
        let tri = [prev.value.point, node.value.point, next.value.point];
        let p = this.pList.first;
        while (!!p) {
            if (!p.value.isConvex && p != node && p != prev && p != next) {
                if (this.inTriangle(p.value.point, tri)) {
                    node.value.isSeparable = false;
                    return;
                }
            }
            p = p.next;
        }
        //顶点为凸顶点且与相邻顶点组成的三角形未包含其他凹顶点时，顶点可分割
        node.value.isSeparable = true;
    }
    /**分割凸多边形 */
    private static processConvexPlygon(points: Vec2[]): number[] {
        let tri: number[] = [];
        for (let i = 1, c = points.length - 1; i < c; ++i) {
            tri.push(0, i, i + 1);
        }
        return tri;
    }
    /**分割只有一个凹顶点的多边形 */
    private static processSingleConcavePlygon(points: Vec2[], concavePointIndex: number): number[] {
        if (concavePointIndex == 0) {
            return this.processConvexPlygon(points);
        }
        let tri: number[] = [];
        let index = concavePointIndex;
        let lastIndex = points.length - 1;
        for (let i = concavePointIndex + 1; i < lastIndex; ++i) {
            tri.push(index, i, i + 1);
        }
        if (index !== lastIndex) {
            tri.push(index, lastIndex, 0);
        }
        let preIndex = index - 1;
        for (let i = 0; i < preIndex; ++i) {
            tri.push(index, i, i + 1);
        }
        return tri;
    }
    /**分割存在多个凹顶点的多边形 */
    private static processMultilpeConcavePlygon(pList: List<PointStatus>): number[] {
        let tri: number[] = [];
        let node = pList.first;
        while (pList.length >= 3) {
            //从当前顶点往链表后面查找可分隔的顶点
            while (!!node && !node.value.isSeparable) {
                node = node.next;
            }
            //若直到链表尾部都不存在可分隔顶点，则从链表头部重新开始查找
            if (!node) {
                node = pList.first;
                while (!!node && !node.value.isSeparable) {
                    node = node.next;
                }
                if (!node) {
                    console.warn("多边形分割尚未未完成，但找不到可分割顶点:", pList);
                    break;
                }
            }
            //可分隔顶点与其相邻顶点组成一个三角形
            let prev = node.prev ? node.prev : pList.last;
            let next = node.next ? node.next : pList.first;
            tri.push(prev.value.pIndex, node.value.pIndex, next.value.pIndex);
            //将当前顶点移除，继续分隔剩余的顶点
            pList.removeNode(node);
            node.destroy();
            //更新当前顶点相邻的两个顶点的状态，其他顶点不受影响
            this.updatePointStatus(prev);
            this.updatePointStatus(next);
            node = next;
        }
        return tri;
    }
    /**顶点p是否凸点 */
    private static isConvex(p: Vec2, p1: Vec2, p2: Vec2): boolean {
        return this.cross(this.sub(p, p1), this.sub(p2, p)) >= 0;
    }
    /**顶点是否在多边形内部 */
    private static inTriangle(p: Vec2, tri: Vec2[]): boolean {
        let ab = this.sub(tri[1], tri[0]);
        let ac = this.sub(tri[2], tri[0]);
        let bc = this.sub(tri[2], tri[1]);
        let ad = this.sub(p, tri[0]);
        let bd = this.sub(p, tri[1]);
        let b = this.cross(ab, ac) >= 0;
        return (b !== (this.cross(ab, ad) < 0)) &&
            (b !== this.cross(ac, ad) >= 0) &&
            (this.cross(bc, ab) > 0) !== (this.cross(bc, bd) >= 0);
    }
    private static reset() {
        if (!!this.pList) {
            this.pList.reset();
            this.pList = null;
        }
    }
    private static cross(p1: Vec2, p2: Vec2): number {
        return p1.x * p2.y - p1.y * p2.x;
    }
    private static sub(p1: Vec2, p2: Vec2): Vec2 {
        return {
            x: p1.x - p2.x,
            y: p1.y - p2.y
        };
    }
}

class PointStatus {
    /**顶点在数组中的索引 */
    public pIndex: number;
    /**顶点坐标 */
    public point: Vec2 = null;
    /**顶点是否为凸顶点 */
    public isConvex: boolean = false;
    /**顶点是否可分隔 */
    public isSeparable: boolean = false;
    public constructor(index: number, v: Vec2) {
        this.pIndex = index;
        this.point = v;
    }
}