const TAG = "TouchControllerV2";

const MOVE_THRESHOLD = cc.sys.isNative && cc.sys.os === cc.sys.OS_ANDROID ? 1.5 : 1.5;

const TouchStatus = {
    INTERCEPTED: 1,
    ABANDONED: 2
};

cc.Class({
    
    extends: cc.Component,

    properties: {
        TargetNode : cc.Node,
        RotationSpeed : 5.
    },

    registerTouchEvt( evt, sender ){
        this.dragEvent = evt;
        this.eventSender = sender;
        this.previusTouchPos = cc.v2( 0., 0. );
    },

    initSize(){
        if( this.node && cc.isValid( this.node ) ){
            this.node.width = cc.winSize.width + 10.;
            this.node.height = cc.winSize.height + 10.;
        }
    },

    start () {
        this.factor = 1;
        this.isStartRotation = false;
        this.deltaTime = 0.;
        this.startTouchTime = 0.;

        // 支持横行滑动模式
        this.isHoriMode = false;
        // 支持纵向滑动模式
        this.isVerticalMode = false;
        //

        this.rotAngle = 12.;
        if( cc.winSize.width >= 1920.){
            this.rotAngle *= cc.winSize.width / 1920.;
        }
        this.rotSpeed = 2.4;

        if( this.TargetNode && cc.isValid(this.TargetNode)){
            this.TargetNode.is3DNode = true;
        }

        this.node.on( cc.Node.EventType.TOUCH_START,  this.onTouchBegan, this );
        this.node.on( cc.Node.EventType.TOUCH_MOVE,   this.onTouchMoved, this );
        this.node.on( cc.Node.EventType.TOUCH_END,    this.onTouchEnded, this );

        if( this.ResetButton && this.ResetButton.node && cc.isValid(this.ResetButton.node) ){
            let touchCtrl = this;
            this.ResetButton.node.on(cc.Node.EventType.TOUCH_START, function (event) {
                
                if( touchCtrl 
                        && touchCtrl.TargetNode 
                            && cc.isValid( touchCtrl.TargetNode)){
                    touchCtrl.resetTarget( touchCtrl.TargetNode);
                }
            });
        }
        this.cacheTouches = [];
    },

    resetTarget( targetNode ){
        if( this.isStartRotation){
            this.isStartRotation = false;
        }
        this.scheduleOnce(()=>{
            if( targetNode && cc.isValid( targetNode)){
                let oriModelRot = cc.v3( 0.,0.,0.);//targetNode.oriModelRot ? targetNode.oriModelRot : cc.v3( 0.,0.,0.);
                cc.tween( targetNode).to( 0.4,{
                    eulerAngles : oriModelRot
                }).start();
            }
        }, 0.1);
    },

    onDestroy(){
        this.node.off( cc.Node.EventType.TOUCH_START,  this.onTouchBegan, this );
        this.node.off( cc.Node.EventType.TOUCH_MOVE,   this.onTouchMoved, this );
        this.node.off( cc.Node.EventType.TOUCH_END,    this.onTouchEnded, this );
    },

    LockTouch(){
        if( this.uiCtrl ){
            this.uiCtrl.enableResetBtn( false );
        }
        this.touchStatus = TouchStatus.ABANDONED;
    },

    RecoverTouch(){
        if( this.uiCtrl ){
            this.uiCtrl.enableResetBtn( true );
        }
        this.touchStatus = TouchStatus.INTERCEPTED;
    },

    isTouchLock(){
        return this.touchStatus == TouchStatus.ABANDONED;
    },

    //
    enableHoriMode(isEnable=true){
        this.isHoriMode = isEnable;
    },

    //
    enbaleVertiMode(isEnable=true){
        this.isVerticalMode = isEnable;
    },

    onTouchBegan( event){

        if( !!event.touch && event.touch.getID() > 0 ) return;
        if( TouchStatus.ABANDONED == this.touchStatus ) return;
        
        this.hasMoved = false;
        event.stopPropagation();

        this.touchStatus = TouchStatus.INTERCEPTED;

        this.startTouchTime = Date.now();
        //cclog.i( TAG, ` ## Cocos Output ## cls: TouchController func:onTouchBegan info: 
                            //start Touch Time:${ this.startTouchTime} `);
        
        // 处于惯性中
        if( this.isStartRotation){
            this.deltaTime = 0;
            this.isStartRotation = false;
        }
        //

        this.previusTouchPos = event.getLocation();
        if( this.dragEvent != null){
            this.dragEvent( 0, this.eventSender );
        }
    },

    onTouchMoved( event ){

        if( !!event.touch && event.touch.getID() > 0) return;
        if( TouchStatus.ABANDONED == this.touchStatus ) return;
        
        if( !this.hasMoved ){

            let delta = event.getDelta().mul(1.);

            if(Math.abs(delta.x) < MOVE_THRESHOLD && Math.abs(delta.y) < MOVE_THRESHOLD)return;

            if( this.TargetNode && cc.isValid( this.TargetNode)){
                
                // 单纯水平滑动
                if( this.isHoriMode ){
                    delta.y = 0.;
                }
                // 竖直水平滑动
                if( this.isVerticalMode ){
                    delta.x = 0.;
                }
                //

                // 常规旋转
                this.targetQuat = this.calculateRotAngle( this.TargetNode, delta.x, delta.y );
                if( this.targetQuat)
                    this.TargetNode.setRotation( this.targetQuat.x, this.targetQuat.y, this.targetQuat.z, this.targetQuat.w);

                if( !this.cacheTouches || this.cacheTouches && this.cacheTouches.length > 5){
                    this.cacheTouches = [];
                }

                let timeStamp = Date.now();
                this.cacheTouches.push({
                    deltaTime : (timeStamp - this.startTouchTime) / 1000,
                    deltaMove : event.getDelta()
                });

                //cc.log( ` ## Cocos Output ## cls:TouchController func:onTouchMoved info: rot= x-${this.TargetNode.eulerAngles.x}
                             //y-${this.TargetNode.eulerAngles.y} z-${this.TargetNode.eulerAngles.z} ` );
            }
        }

        event.stopPropagation();

        if( this.dragEvent != null){

            let prevPos = this.previusTouchPos;
            let currentPos = event.getLocation();
            let moveDelta  = cc.v2(0);
            let moveRot = this.TargetNode ? this.TargetNode.eulerAngles : cc.v3( 0 );
            
            if( prevPos && currentPos 
                   && prevPos.mag() > 0.
                        && cc.v2( currentPos.x - prevPos.x, currentPos.y - prevPos.y ).mag() > 0 ){
                
                if( this.isVerticalMode ){
                    currentPos.x = prevPos.x = 0.;
                }
                if( this.isHoriMode ){
                    currentPos.y = prevPos.y = 0.;
                }
                moveDelta = cc.v2( currentPos.x - prevPos.x, currentPos.y - prevPos.y );

               //moveDelta = cc.v2( currentPos.x - prevPos.x, currentPos.y - prevPos.y ).mag();
            }

            this.dragEvent( 1, this.eventSender, moveDelta , moveRot );
        }
    },

    onTouchEnded( event){

        if( !this.TargetNode){
            return;
        }

        if(!!event.touch && event.touch.getID() > 0) return;
        if(TouchStatus.ABANDONED == this.touchStatus) return;

        event.stopPropagation();

        this.hasMoved = false;
        this.previusTouchPos = cc.v2( 0.,0.);

        if( this.dragEvent != null){
            this.dragEvent( 2, this.eventSender );
        }

        let currentQuat = new cc.Quat();
        currentQuat = this.TargetNode.getRotation( currentQuat);
        cc.Quat.normalize( currentQuat, currentQuat);
        this.currentQuat = currentQuat;

        let totalDeltaMove = cc.v2( 0., 0.);
        let totalDeltaTime = 0.;
        
        //
        if( this.cacheTouches && !this.isHoriMode && !this.isVerticalMode ){

            this.cacheTouches.forEach( element => {
                totalDeltaMove = totalDeltaMove.add(element.deltaMove);
                totalDeltaTime += element.deltaTime;
            });
            
            if( totalDeltaTime == 0){
                totalDeltaTime = 1;
            }
            let averageTouchMove = totalDeltaMove.div( totalDeltaTime);
            // 惯性旋转
            this.targetQuat = this.calculateRotAngle( this.TargetNode, averageTouchMove.x, averageTouchMove.y, 5., 360.);
            if( this.targetQuat){

                this.deltaTime = 0.;
                this.isStartRotation = true;
                
                this.deltaQuat = new cc.Quat();
                cc.Quat.identity( this.deltaQuat);
            }
        }

        //cclog.i(` ## Cocos Output ## cls: TouchController func:onTouchEnded info: 
                           //total deltaX:${ totalDeltaMove.x } total deltaY:${ totalDeltaMove.y } total deltaTime:${ totalDeltaTime} `);
        this.cacheTouches = [];
    },

    calculateRotAngle( node, deltaX, deltaY, minAngle = 0., maxAngle = 0.){

        let targetQuat = new cc.Quat();
        if( deltaX == undefined || deltaY == undefined
                || ( deltaX == 0. && deltaY == 0. )
                    || !node || ( node && !cc.isValid(node)) ){
            return undefined;
        }

        targetQuat = node.getRotation( targetQuat);
        cc.Quat.normalize( targetQuat, targetQuat);

        let axis = cc.v3( 0.,0.,0.);
        let moveX  = Math.abs( deltaY) * this.RotationSpeed * cc.director.getDeltaTime();

        let moveY  = Math.abs( deltaX) * this.RotationSpeed * cc.director.getDeltaTime();

        let angle = 0.;
        angle = cc.v2( moveX, moveY ).mag();
        
        let quat = new cc.Quat();
        
        moveX = deltaY > 0. ?  -moveX : moveX;
        moveY = deltaX >= 0. ? moveY : -moveY;

        axis = cc.v3( moveX, moveY, 0.);
        axis.normalizeSelf();

        if( maxAngle != 0 && Math.abs( angle) >= maxAngle){
            angle = maxAngle * ( angle >= 0. ? 1 : -1 );
        }
        if( minAngle != 0 && Math.abs( angle) <= minAngle ){
            return targetQuat;
        }

        targetQuat = cc.Quat.rotateAround( quat, targetQuat, axis, angle * Math.PI / 180.);
    
        cc.Quat.normalize( targetQuat, targetQuat);

        return targetQuat;
    },

    update(dt){

        if( this.isStartRotation 
                && this.currentQuat 
                    && this.targetQuat ){

            if( this.deltaTime > 1.0){
                
                this.TargetNode.setRotation( this.targetQuat.x, this.targetQuat.y, this.targetQuat.z, this.targetQuat.w);

                this.deltaTime = 0;
                this.isStartRotation = false;

                return;
            }

            this.deltaQuat = cc.Quat.lerp( this.deltaQuat, this.currentQuat, this.targetQuat, this.deltaTime);
            cc.Quat.normalize( this.deltaQuat, this.deltaQuat);

            this.TargetNode.setRotation( this.deltaQuat.x, this.deltaQuat.y, this.deltaQuat.z, this.deltaQuat.w);
            this.deltaTime += dt;
        }
    }
    
});
