
const gfx = cc['gfx'];

let vfmtMultiTexture = new gfx.VertexFormat([
    // 节点的世界坐标，占2个float32
    { name: gfx.ATTR_POSITION, type: gfx.ATTR_TYPE_FLOAT32, num: 2 },
    // 节点的纹理uv坐标，占2个float32
    // 如果节点使用了独立的纹理（未合图），这里的uv值通常是0或1
    // 合图后的纹理，这里的uv对应其在图集里的相对位置，取值范围在[0,1)内
    { name: gfx.ATTR_UV0, type: gfx.ATTR_TYPE_FLOAT32, num: 2 }, 
    // 节点颜色值，cc.Sprite组件上可以设置。占4个uint8 = 1个float32
    { name: gfx.ATTR_COLOR, type: gfx.ATTR_TYPE_UINT8, num: 4, normalize: true },
    // 纹理索引，占1个float32
    { name: "a_texture_idx", type: gfx.ATTR_TYPE_FLOAT32, num: 1 },
]);

// @ts-ignore
export class MultiAssembler extends cc.Assembler {

    // 渲染数据
    protected _renderData = null;
    // 本地数据
    protected _local = [];

    // 顶点数据长度,包含坐标，uv，颜色，纹理索引（x, y, u, v, color, texIdx）
    protected floatsPerVert: number = 6;
    // 顶点数量
    protected verticesCount: number = 4;
    // 索引数量
    protected indicesCount: number = 6;
    // uv偏移量
    protected uvOffset: number = 2;
    // 颜色偏移量
    protected colorOffset: number = 4;


    constructor() {
        super();

        // renderData.vDatas用来存储pos、uv、color数据
        // renderData.iDatas用来存储顶点索引数据
        // @ts-ignore
        this._renderData = new cc.RenderData();
        this._renderData.init(this);

        this.initData();
        this.initLocal();
    }

    public updateTextureIdx(sprite) {
        let verts = this._renderData.vDatas[0];

        /**
         * 顶点0 (左下): [x0, y0, u0, v0, color0, texIdx0 ]  // 索引 0-5
         * 顶点1 (右下): [x1, y1, u1, v1, color1, texIdx1 ]  // 索引 6-11  
         * 顶点2 (左上): [x2, y2, u2, v2, color2, texIdx2 ]  // 索引 12-17
         * 顶点3 (右上): [x3, y3, u3, v3, color3, texIdx3 ]  // 索引 18-23
         */
        verts[5] = sprite.textureIdx;
        verts[11] = sprite.textureIdx;
        verts[17] = sprite.textureIdx;
        verts[23] = sprite.textureIdx;
    }
    
    init (renderComp) {
        // @ts-ignore
        this._renderComp = renderComp;
    }

    getVfmt () {
        return vfmtMultiTexture;
    }
    // 顶点数据长度
    get verticesFloats () {
        // 当前节点的所有顶点数据总大小
        return this.verticesCount * this.floatsPerVert;
    }

    initData () {
        let data = this._renderData;
        // 创建一个足够长的空间用来存储顶点数据 & 顶点索引数据
      	// 这个方法内部会初始化顶点索引数据
        // data.createQuadData(0, this.verticesFloats, this.indicesCount);
        // createFlexData支持创建指定格式的renderData
        data.createFlexData(0, this.verticesFloats, this.indicesCount, this.getVfmt());

        // createFlexData不会填充顶点索引信息，手动补充一下
        let indices = data.iDatas[0];
        let count = indices.length / 6;
        for (let i = 0, idx = 0; i < count; i++) {
            let vertextID = i * 4;
            indices[idx++] = vertextID;
            indices[idx++] = vertextID+1;
            indices[idx++] = vertextID+2;
            indices[idx++] = vertextID+1;
            indices[idx++] = vertextID+3;
            indices[idx++] = vertextID+2;
        }
    }
    initLocal () {
        this._local = [];
        this._local.length = 4;
    }

    updateColor (comp, color) {
        let uintVerts = this._renderData.uintVDatas[0];
        if (!uintVerts) return;
        color = color != null ? color : comp.node.color._val;
        let floatsPerVert = this.floatsPerVert;
        let colorOffset = this.colorOffset;
        for (let i = colorOffset, l = uintVerts.length; i < l; i += floatsPerVert) {
            uintVerts[i] = color;
        }
    }

    getBuffer (v) {
        // @ts-ignore
        return cc.renderer._handle.getBuffer("mesh", this.getVfmt());
    }

    updateWorldVerts (comp) {
        let local = this._local;
        let verts = this._renderData.vDatas[0];

        let matrix = comp.node._worldMatrix;
        let matrixm = matrix.m,
            a = matrixm[0], b = matrixm[1], c = matrixm[4], d = matrixm[5],
            tx = matrixm[12], ty = matrixm[13];

        let vl = local[0], vr = local[2],
            vb = local[1], vt = local[3];
        
        let floatsPerVert = this.floatsPerVert;
        let vertexOffset = 0;
        let justTranslate = a === 1 && b === 0 && c === 0 && d === 1;

        if (justTranslate) {
            // left bottom
            verts[vertexOffset] = vl + tx;
            verts[vertexOffset + 1] = vb + ty;
            vertexOffset += floatsPerVert;
            // right bottom
            verts[vertexOffset] = vr + tx;
            verts[vertexOffset + 1] = vb + ty;
            vertexOffset += floatsPerVert;
            // left top
            verts[vertexOffset] = vl + tx;
            verts[vertexOffset + 1] = vt + ty;
            vertexOffset += floatsPerVert;
            // right top
            verts[vertexOffset] = vr + tx;
            verts[vertexOffset + 1] = vt + ty;
        } else {
            let al = a * vl, ar = a * vr,
            bl = b * vl, br = b * vr,
            cb = c * vb, ct = c * vt,
            db = d * vb, dt = d * vt;

            // left bottom
            verts[vertexOffset] = al + cb + tx;
            verts[vertexOffset + 1] = bl + db + ty;
            vertexOffset += floatsPerVert;
            // right bottom
            verts[vertexOffset] = ar + cb + tx;
            verts[vertexOffset + 1] = br + db + ty;
            vertexOffset += floatsPerVert;
            // left top
            verts[vertexOffset] = al + ct + tx;
            verts[vertexOffset + 1] = bl + dt + ty;
            vertexOffset += floatsPerVert;
            // right top
            verts[vertexOffset] = ar + ct + tx;
            verts[vertexOffset + 1] = br + dt + ty;
        }
    }

    fillBuffers (comp, renderer) {
        // 如果节点的世界坐标发生变化，重新从当前节点的世界坐标计算一次顶点数据
        if (renderer.worldMatDirty) {
            this.updateWorldVerts(comp);
        }
        // 获取准备好的顶点数据
      	// vData包含pos、uv、color数据
      	// iData包含三角剖分后的顶点索引数据
        let renderData = this._renderData;
        let vData = renderData.vDatas[0];
        let iData = renderData.iDatas[0];

        // 获取顶点缓存
        let buffer = this.getBuffer(renderer);
        // 获取当前节点的顶点数据对应最终buffer的偏移量
      	// 可以简单理解为当前节点和其他同格式节点的数据，都将按顺序追加到这个大buffer里
        let offsetInfo = buffer.request(this.verticesCount, this.indicesCount);

        // buffer data may be realloc, need get reference after request.

        // fill vertices
        let vertexOffset = offsetInfo.byteOffset >> 2,
            vbuf = buffer._vData;

        // 将准备好的vData拷贝到VetexBuffer里。
        // 这里会判断如果buffer装不下了，vData会被截断一部分。
      	// 通常不会出现装不下这种情况，因为buffer.request中会分配足够大的空间；如果出现，则当前组件只能被渲染一部分
        if (vData.length + vertexOffset > vbuf.length) {
            vbuf.set(vData.subarray(0, vbuf.length - vertexOffset), vertexOffset);
        } else {
            vbuf.set(vData, vertexOffset);
        }

        // 将准备好的iData拷贝到IndiceBuffer里
        let ibuf = buffer._iData,
            indiceOffset = offsetInfo.indiceOffset,
            vertexId = offsetInfo.vertexOffset;
        /** Since the update of indicesCout may occur before the update of iData.length, indicesCout does not always equal iData.length. 
          * It may be greater than or less than iData.length.
          */
        for (let i = 0, l = Math.min(this.indicesCount, iData.length); i < l; i++) {
            ibuf[indiceOffset++] = vertexId + iData[i];
        }
    }

    packToDynamicAtlas (comp, frame) {
        if (CC_TEST) return;
        
        if (!frame._original && cc.dynamicAtlasManager && frame._texture.packable && frame._texture.loaded) {
            let packedFrame = cc.dynamicAtlasManager.insertSpriteFrame(frame);
            if (packedFrame) {
                frame._setDynamicAtlasFrame(packedFrame);
            }
        }
        let material = comp._materials[0];
        if (!material) return;
        
        if (material.getProperty('texture0') !== frame._texture._texture) {
            // texture was packed to dynamic atlas, should update uvs
            comp._vertsDirty = true;
            comp._updateMaterial();
        }
    }
    
    updateRenderData (comp) {
    }
    
    
}