import { _decorator, Component, Node, Sprite, instantiate, Vec3, EventMouse, director, input, Input, Event } from 'cc';
const { ccclass, property } = _decorator;

interface PathNode {
    position: Vec3;
    parent: PathNode | null;
    g: number; // 从起点到当前节点的实际代价
    h: number; // 从当前节点到终点的估计代价
    f: number; // 总代价 (f = g + h)
}

@ccclass('ChessboardGenerator')
export class ChessboardGenerator extends Component {
    @property(Sprite)
    floorTileSprite: Sprite = null;

    @property(Sprite)
    wallTileSprite: Sprite = null;

    @property(Sprite)
    characterSprite: Sprite = null;

    @property(Sprite)
    goalSprite: Sprite = null;

    @property
    rows: number = 15;

    @property
    cols: number = 15;

    @property
    tileSpacing: number = 32;

    chessboardLayout: number[][] = [
        [1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
        [1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1],
        [1, 0, 1, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 1],
        [1, 0, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 1, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1],
        [1, 0, 1, 1, 0, 1, 1, 1, 0, 1, 0, 1, 1, 0, 1],
        [1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1],
        [1, 0, 1, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 1],
        [1, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 1],
        [1, 0, 1, 1, 0, 1, 1, 0, 1, 1, 0, 0, 0, 0, 1],
        [1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1],
        [1, 0, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 0, 1, 1],
        [1, 0, 1, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1],
        [1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1],
        [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1]
    ];

    private characterNode: Node = null;
    private goalNode: Node = null;
    private path: Vec3[] = [];
    private currentPathIndex: number = 0;

    start() {
        console.log("Chessboard Layout:", this.chessboardLayout);
        this.generateChessboard();
        this.spawnCharacter();
        this.spawnGoal();
    }

    onLoad(): void {
        input.on(Input.EventType.MOUSE_DOWN, this.onMouseDown, this);
    }

    onDestroy() {
        input.off(Input.EventType.MOUSE_DOWN, this.onMouseDown, this);
    }

    generateChessboard() {
        for (let row = 0; row < this.rows; row++) {
            for (let col = 0; col < this.cols; col++) {
                const tileType = this.chessboardLayout[row][col];
                const tile = this.instantiateTile(tileType);
                if (tile) {
                    const position = this.calculateTilePosition(row, col);
                    tile.node.setPosition(position);
                    this.node.addChild(tile.node);
                }
            }
        }
    }

    instantiateTile(tileType: number): Sprite | null {
        let tileNode: Node;
        if (tileType === 0) {
            tileNode = instantiate(this.floorTileSprite.node);
        } else if (tileType === 1) {
            tileNode = instantiate(this.wallTileSprite.node);
        } else {
            return null;
        }
        return tileNode.getComponent(Sprite);
    }

    calculateTilePosition(row: number, col: number): Vec3 {
        const x = (col - (this.cols - 1) / 2) * this.tileSpacing;
        const y = (-(row - (this.rows - 1) / 2)) * this.tileSpacing;
        return new Vec3(x, y, 0);
    }

    spawnCharacter() {
        if (this.characterSprite) {
            this.characterNode = instantiate(this.characterSprite.node);
            const position = this.calculateTilePosition(1, 1);
            this.characterNode.setPosition(position);
            this.node.addChild(this.characterNode);
        } else {
            console.error("Character sprite is not assigned.");
        }
    }

    spawnGoal() {
        if (this.goalSprite) {
            this.goalNode = instantiate(this.goalSprite.node);
            const position = this.calculateTilePosition(13, 7);
            this.goalNode.setPosition(position);
            this.node.addChild(this.goalNode);
        } else {
            console.error("Goal sprite is not assigned.");
        }
    }

    onMouseDown(event: EventMouse) {
        console.log("Mouse clicked");
        this.calculatePath();
        if (this.path.length > 0) {
            this.currentPathIndex = 0; // 重置路径索引
            this.moveCharacter(); // 移动到路径上的下一个节点
        }
    }

    calculatePath() {
        if (!this.characterNode || !this.goalNode) {
            console.error("Character or Goal node is not assigned.");
            return;
        }

        const start = this.getGridPosition(this.characterNode.position);
        const goal = this.getGridPosition(this.goalNode.position);

        console.log("Start:", start);
        console.log("Goal:", goal);

        const openSet: PathNode[] = [];
        const closedSet: PathNode[] = [];

        const startNode: PathNode = {
            position: this.characterNode.position,
            parent: null,
            g: 0,
            h: this.heuristic(start, goal),
            f: 0
        };

        openSet.push(startNode);

        while (openSet.length > 0) {
            let currentNode = openSet[0];
            let currentIndex = 0;

            for (let i = 1; i < openSet.length; i++) {
                if (openSet[i].f < currentNode.f) {
                    currentNode = openSet[i];
                    currentIndex = i;
                }
            }

            openSet.splice(currentIndex, 1);
            closedSet.push(currentNode);

            if (this.getGridPosition(currentNode.position).equals(goal)) {
                this.path = this.reconstructPath(currentNode);
                this.currentPathIndex = 0;
                console.log("Path found:", this.path);
                return;
            }

            const neighbors = this.getNeighbors(currentNode.position);

            for (const neighbor of neighbors) {
                const neighborNode: PathNode = {
                    position: neighbor,
                    parent: currentNode,
                    g: currentNode.g + 1,
                    h: this.heuristic(this.getGridPosition(neighbor), goal),
                    f: 0
                };

                neighborNode.f = neighborNode.g + neighborNode.h;

                if (this.isInSet(closedSet, neighborNode)) {
                    continue;
                }

                if (!this.isInSet(openSet, neighborNode) || neighborNode.g < this.getNode(openSet, neighborNode).g) {
                    openSet.push(neighborNode);
                }
            }
        }

        console.error("No path found.");
    }

    heuristic(a: Vec3, b: Vec3): number {
        const dx = Math.abs(a.x - b.x);
        const dy = Math.abs(a.y - b.y);
        return dx + dy;
    }

    getGridPosition(position: Vec3): Vec3 {
        const col = Math.round(position.x / this.tileSpacing) + (this.cols - 1) / 2;
        const row = -(Math.round(position.y / this.tileSpacing) - (this.rows - 1) / 2);
        return new Vec3(col, row, 0);
    }

    getTilePosition(gridPosition: Vec3): Vec3 {
        const x = (gridPosition.x - (this.cols - 1) / 2) * this.tileSpacing;
        const y = (-(gridPosition.y - (this.rows - 1) / 2)) * this.tileSpacing;
        return new Vec3(x, y, 0);
    }

    getNeighbors(position: Vec3): Vec3[] {
        const gridPosition = this.getGridPosition(position);
        const neighbors: Vec3[] = [];

        const directions = [
            new Vec3(1, 0, 0),  // right
            new Vec3(-1, 0, 0), // left
            new Vec3(0, 1, 0),  // up
            new Vec3(0, -1, 0)  // down
        ];

        for (const direction of directions) {
            const neighborGridPosition = gridPosition.clone().add(direction);
            if (this.isValidPosition(neighborGridPosition)) {
                neighbors.push(this.getTilePosition(neighborGridPosition));
            }
        }

        return neighbors;
    }

    isValidPosition(position: Vec3): boolean {
        const row = Math.round(position.y);
        const col = Math.round(position.x);
        return row >= 0 && row < this.rows && col >= 0 && col < this.cols && this.chessboardLayout[row][col] === 0;
    }

    isInSet(set: PathNode[], node: PathNode): boolean {
        for (const item of set) {
            if (item.position.equals(node.position)) {
                return true;
            }
        }
        return false;
    }

    getNode(set: PathNode[], node: PathNode): PathNode {
        for (const item of set) {
            if (item.position.equals(node.position)) {
                return item;
            }
        }
        return null;
    }

    reconstructPath(node: PathNode): Vec3[] {
        const path: Vec3[] = [];
        let currentNode = node;
        while (currentNode) {
            path.push(currentNode.position);
            currentNode = currentNode.parent;
        }
        path.reverse();
        return path;
    }

    moveCharacter() {
        if (this.path.length > 0 && this.currentPathIndex < this.path.length) {
            const targetPosition = this.path[this.currentPathIndex];
            this.characterNode.setPosition(targetPosition);
            this.currentPathIndex++;
            console.log("Moved to:", targetPosition);
        } else {
            console.log("Reached the goal or no path available.");
        }
    }

    update(dt: number): void {
        // 不再在每一帧调用moveCharacter
    }
}