import { Component, _decorator, sp } from "cc";

const { ccclass, property, menu, requireComponent } = _decorator;

export let config = {
    1: {
        'head': {
            bone: 'head',
            slot: 'head',
            attachments: ['head_1']
        },
        'body': {
            bone: 'body',
            slot: 'body',
            attachments: ['body_1']
        },
        'hand': {
            bone: 'hand',
            slot: 'hand',
            attachments: ['hand_1']
        }

    }
}


export type SkinPartEntry = {
    id?: string;
    bone?: string;
    slot?: string;
    attachments?: string[];
};

class SkinPartCache {
    bones: sp.spine.BoneData[] = [];
    attachments: { [key: string]: sp.spine.Attachment }[] = []; // slotID为数组下表
}

// 应用：将spine切换为纸娃娃系统，可局部换装
@ccclass
@menu('GFun/Spine/SpineSkinPartPlayer')
@requireComponent(sp.Skeleton)
export default class SpineSkinPartPlayer extends Component {

    @property({ displayName: '自定义皮肤名称' })
    customSkinName: string = 'custom';

    private _skeleton: sp.Skeleton;

    private _skin: sp.spine.Skin = null; // 当前使用的皮肤
    private _skinPartCache: { [partName: string]: SkinPartCache } = {}; // 当前正在使用的部件信息

    private _isDirty: boolean = false;
    private _config: any = null;

    onEnable() {
        this._skeleton = this.node.getComponent(sp.Skeleton);
    }
    onDisable() {
        if (this._skin) {
            this.removeSkin(this._skin);
            this._skin = null;
        }
    }

    update(dt: number) {
        if (this._isDirty) {
            this._isDirty = false;

            this.resetSkin();
        }
    }

    private addSkin(): sp.spine.Skin {
        if (!this._skeleton) {
            return null;
        }
        // 
        let skin;
        let data: sp.spine.SkeletonData = this._skeleton.skeletonData.getRuntimeData();
        if (data) {
            skin = data.findSkin(this.customSkinName);
        }
        return skin;
    }
    private removeSkin(skin: sp.spine.Skin) {
        if (!skin) {
            return;
        }
        // TODO: 
    }

    setSkinConfig(config: {}) {
        this._config = config;
    }

    setSkinPart(partID: string, skin: string) {
        if (!this._skeleton || !this._config) {
            return;
        }
        this.updateSkinPart(partID, skin);
        this._isDirty = true;
    }

    private updateSkinPart(partID: string, skinID: string) {

        // 根据部件配置进行内容赋值
        let partEntries: SkinPartEntry[] = this._config.getEntry(partID);
        if (!partEntries) {
            return;
        }

        // 构建新的部件表单
        let skinPartCache = this._skinPartCache[partID];
        if (!skinPartCache) {
            skinPartCache = this._skinPartCache[partID] = new SkinPartCache;
        }
        else {
            skinPartCache.bones.length = 0;
            skinPartCache.attachments.length = 0;
        }

        let data: sp.spine.SkeletonData = this._skeleton.skeletonData.getRuntimeData();
        if (data) {
            //
            let bones = {};
            let slots = {};
            for (let i = 0; i < partEntries.length; ++i) {
                let partEntry = partEntries[i];
                if (partEntry) {
                    if (partEntry.bone) {
                        bones[partEntry.bone] = partEntry.bone;
                    }
                    if (partEntry.slot) {
                        slots[partEntry.slot] = partEntry.attachments;
                    }
                }
                let skin = data.findSkin(skinID);
                if (skin) {
                    for (let j = 0; j < skin.bones.length; ++i) {
                        let bone = skin.bones[j];
                        if (bone.name) {
                            skinPartCache.bones.push(bone);
                        }
                    }
                    for (let index in slots) {
                        let slotID = data.findSlotIndex(index);
                        let skinAttachments = {};
                        let attachments = slots[index];
                        for (let j = 0; j < attachments.length; ++j) {
                            let attachment = attachments[j];
                            let skinAttachment = skin.getAttachment(slotID, attachment);
                            // if (skinAttachment) {
                            skinAttachments[attachment] = skinAttachment;
                            // }
                        }
                        if (skinAttachments) {
                            if (slotID >= skinPartCache.attachments.length) {
                                skinPartCache.attachments.length = slotID + 1;
                            }
                            skinPartCache.attachments[slotID] = skinAttachments;
                        }
                    }
                }
            }
        }
    }

    private resetSkin() {

        // 构建新的皮肤配置
        let skin = this.addSkin();
        for (let partID in this._skinPartCache) {
            let avatarSkinPart = this._skinPartCache[partID];
            if (avatarSkinPart) {
                for (let i = 0; i < avatarSkinPart.bones.length; ++i) {
                    skin.bones.push(avatarSkinPart.bones[i]);
                }
                for (let slotID = 0; slotID < avatarSkinPart.attachments.length; ++slotID) {
                    let avatarSkinPartAttachments = avatarSkinPart.attachments[slotID];
                    if (avatarSkinPartAttachments) {
                        for (let index in avatarSkinPartAttachments) {
                            let attachment = avatarSkinPartAttachments[index];
                            if (attachment) {
                                skin.setAttachment(slotID, index, attachment);
                            }
                            else {
                                skin.removeAttachment(slotID, index);
                            }
                        }
                    }
                }
            }
        }

        // 将新的皮肤设置进去
        this._skeleton.setSkin(skin.name);

        // 之前的皮肤就失效了
        this.removeSkin(this._skin);
        this._skin = skin;
    }
}