
import { _decorator, Node, Touch, view, math, Vec2, RenderTexture, UITransform, Vec3, Camera, SpriteFrame } from 'cc';

export class RenderTextureHelper {
    public static offset: Readonly<Vec2> = new Vec2(-900, -1600);

    /** 取得點擊位置對應的3D射線發射位置
     * @param touch 點擊事件的參數
     * @param cameraUI 2D顯示的相機
     * @param renderTexture 3D顯示的相機
     * @param rect 3D顯示畫面Sprite的UITransform
     * @returns 相機射線位置 */
    public static GetRaycastPosition(touch: Touch, cameraUI: Camera, camera3D: Camera, rect: UITransform): Vec3 {
        // 取得點擊位置
        const touchPos: Vec2 = touch.getLocation();

        // 取得3D繪圖的renderTexture
        const renderTexture: RenderTexture = camera3D.targetTexture;

        // 計算畫面左下角至畫面中心的位置差
        const canvasSize: math.Size = view.getCanvasSize();
        const deltaVec: Vec2 = new Vec2(canvasSize.width * 0.5, canvasSize.height * 0.5);

        // 3D畫面與renderTexture預設尺寸的比率
        // 比如畫面大小為2520X2520，renderTexture為2048X2048，則比率為(0.8127, 0.8127)
        const ratio: Vec2 = new Vec2(renderTexture.width / rect.width, renderTexture.height / rect.height);
        const offset: Vec2 = this.GetAdjustedOffset(renderTexture.width, renderTexture.height);

        // 將點擊位置轉成相機畫面位置
        const cameraPos: Vec3 = Vec3.ZERO.clone();

        // 調整後的數據
        const adjustPos: Vec2 = Vec2.ZERO.clone();
        adjustPos.x = ratio.x * touchPos.x + (1 - ratio.x) * deltaVec.x;
        adjustPos.y = ratio.y * touchPos.y + (1 - ratio.y) * deltaVec.y;
        cameraUI.screenToWorld(new Vec3(adjustPos.x, adjustPos.y), cameraPos);
        return new Vec3(cameraPos.x + offset.x, cameraPos.y + offset.y);
    }

    /** 取得點擊位置在UI上的位置
     * @param touch 點擊事件的參數
     * @param cameraUI 2D顯示的相機
     * @param spriteFrame 點擊對象的貼圖資料 
     * @returns 以畫面中心為起點的2D位置資訊 */
    public static GetTouchPosition(touch: Touch, cameraUI: Camera, spriteFrame: SpriteFrame): Vec2 {
        // 取得點擊位置
        const touchPos: Vec2 = touch.getLocation();

        // 將點擊位置轉至螢幕位置
        const screenPos: Vec3 = Vec3.ZERO.clone();
        const offset: Vec2 = this.GetAdjustedOffset(spriteFrame.width, spriteFrame.height);
        const deltaVec: Vec2 = new Vec2(spriteFrame.width * 0.5, spriteFrame.height * 0.5);

        cameraUI.screenToWorld(new Vec3(touchPos.x, touchPos.y), screenPos);
        return new Vec2(screenPos.x + offset.x - deltaVec.x, screenPos.y + offset.y - deltaVec.y);
    }

    /** 取得3D轉至2DUI的位置
     * @param camera3D 3D相機
     * @param target3D 3D物件
     * @param view3D 3D顯示的畫布
     * @returns 以畫面中心為原點的位置資訊 */
    public static GetUIPosition(camera3D: Camera, target3D: Node, view3D: UITransform): Vec2 {
        const renderTexture: RenderTexture = camera3D.targetTexture;

        // 3D匯圖像素至畫面中心的位置差
        const deltaVec: Vec2 = new Vec2(renderTexture.width * 0.5, renderTexture.height * 0.5);
        
        // 取得初步3D轉至相機螢幕的位置
        const localPos: Vec3 = Vec3.ZERO.clone();
        const worldPos: Vec3 = target3D.worldPosition.clone();
        camera3D.worldToScreen(worldPos, localPos);
        
        // 3D畫面與renderTexture預設尺寸的比率
        // 比如畫面大小為2520X2520，renderTexture為2048X2048，則比率為(1.23, 1.23)
        const ratio: Vec2 = new Vec2(view3D.width / renderTexture.width, view3D.height / renderTexture.height);

        // 將螢幕位置轉換座標得到已畫面中心為原點的位置
        const x = ratio.x * (localPos.x - deltaVec.x);
        const y = ratio.y * (localPos.y - deltaVec.y);

        return new Vec2(x, y);
    }

    /** 計算回推至正確的相機位置的校正向量 */
    private static GetAdjustedOffset(width: number, height: number): Vec2 {
        const visibleSize: math.Size = view.getVisibleSize();
        const difference: Vec2 = new Vec2(width - visibleSize.width, height - visibleSize.height);
        return new Vec2(this.offset.x + difference.x * 0.5, this.offset.y + difference.y * 0.5);

    }
}
