import { _decorator, Component, Node, Sprite, Camera, SystemEvent, Touch, EventTouch, geometry, Vec3, PhysicsSystem, Enum, physics, UIOpacity, Vec2, UITransform } from 'cc';
import { RenderTextureHelper } from './RenderTextureHelper';
const { ccclass, property, requireComponent } = _decorator;

enum ERaycastType {
    ALL,
    CLOSEST,
}
Enum(ERaycastType);

@ccclass('RenderTextureRaycast')
@requireComponent(Sprite)
@requireComponent(UIOpacity)
export class RenderTextureRaycast extends Component {
    @property({ displayName: 'UI相機', type: Camera })
    public cameraUI: Camera = null;
    @property({ displayName: '3D相機', type: Camera })
    public camera3D: Camera = null;
    @property({ displayName: '碰撞3D物件的結果', type: ERaycastType })
    private raycastType: ERaycastType = ERaycastType.CLOSEST;

    /** 點擊事件 */
    public hitAction: HitAction = null;

    private _uiOpacity: UIOpacity = null;
    public get uiOpacity(): UIOpacity {
        if (this._uiOpacity === null) {
            this._uiOpacity = this.getComponent(UIOpacity) ?? this.node.addComponent(UIOpacity);
        }
        return this._uiOpacity;
    }

    private _rect: UITransform = null;
    public get rect(): UITransform {
        if (this._rect === null) {
            this._rect = this.getComponent(UITransform) ?? this.node.addComponent(UITransform);
        }
        return this._rect;
    }

    private _sprite: Sprite = null;
    public get sprite(): Sprite {
        if (this._sprite === null) {
            this._sprite = this.getComponent(Sprite) ?? this.node.addComponent(Sprite);
        }
        return this._sprite;
    }

    onEnable() {
        this.node.on(Node.EventType.TOUCH_START, this.OnTouchStart, this);
    }

    onDisable() {
        this.node.off(Node.EventType.TOUCH_START, this.OnTouchStart, this);
    }

    private OnTouchStart(touch: Touch, event: EventTouch) {
        if (false === this.enabled) return;
        if (this.cameraUI === null) return;

        const ray = new geometry.Ray();
        const rayPos: Vec3 = RenderTextureHelper.GetRaycastPosition(touch, this.cameraUI, this.camera3D, this.rect);console.error(`(${rayPos.x}, ${rayPos.y})`)

        // 發送射線
        this.camera3D.screenPointToRay(rayPos.x, rayPos.y, ray);
        if (PhysicsSystem.instance.raycastClosest(ray)) {
            const results: physics.PhysicsRayResult[] = [];

            // 依照設定取得最近目標物還是所以射線穿過的3D物件
            if (this.raycastType === ERaycastType.CLOSEST) {
                results.push(PhysicsSystem.instance.raycastClosestResult);
            }
            else {
                results.concat(PhysicsSystem.instance.raycastResults);
            }

            // 執行點擊後的動作
            for (const res of results) {
                if (res && this.hitAction) {
                    const clickPos: Vec2 = RenderTextureHelper.GetTouchPosition(touch, this.cameraUI, this.sprite.spriteFrame);            
                    this.hitAction(res.collider.node, clickPos);
                }
            }
        }
    }
}

type HitAction = (node: Node, pos: Vec2) => void;
