import { tickManager } from '../ticker/TickManager';
import { IPromiseCancellable, PromiseExecutorFunction, ICreatePromiseOptions } from './PromiseDefines';
import { PromiseBase } from './PromiseBase';

/**
 * 独立的可取消 Promise 类
 */
export class PromiseCancellable<T> extends PromiseBase implements IPromiseCancellable<T> {
    private _promise: Promise<T>;
    private _resolve!: (value: T | PromiseLike<T>) => void;
    private _reject!: (reason?: any) => void;
    private _isSettled = false;
    private _isCancelled = false;
    private _timeoutId?: number;
    private _cancelResolve?: () => void;

    public readonly id: string;
    public readonly category: string;

    constructor(
        executor: PromiseExecutorFunction<T>,
        private readonly options: ICreatePromiseOptions<T>,
        private readonly onCleanup: (id: string) => void,
    ) {
        super();
        this.id = options.id!;
        this.category = options.category!;

        this._promise = new Promise<T>((resolve, reject) => {
            this._resolve = resolve;
            this._reject = reject;
            this.setupPromise(executor);
        });
    }

    get promise(): Promise<T> {
        return this._promise;
    }

    get isCancelled(): boolean {
        return this._isCancelled;
    }

    /**
     * 设置 Promise 的内部逻辑
     */
    private setupPromise(executor: PromiseExecutorFunction<T>): void {
        // 设置超时
        this.setupTimeout();

        // 执行原始的 Promise 逻辑
        executor(
            (value) => {
                if (!this._isCancelled) {
                    this.settle(false, value);
                }
            },
            (reason) => {
                if (!this._isCancelled) {
                    this.settle(true, reason);
                }
            },
        );
    }

    /**
     * 设置超时逻辑
     */
    private setupTimeout(): void {
        const { timeout, timeoutAction, timeoutValue, timeoutReason, text } = this.options;

        if (!timeout || timeout <= 0) {
            return;
        }

        this._timeoutId = tickManager.once(
            () => {
                if (!this._isCancelled) {
                    if (timeoutAction === 'resolve') {
                        this.settle(false, timeoutValue);
                    } else {
                        this.settle(true, timeoutReason ?? new Error(`Promise ${this.id} 超时, text:${text}`));
                    }
                }
            },
            this,
            timeout,
        );
    }

    /**
     * 清理超时定时器
     */
    private clearTimeout(): void {
        if (this._timeoutId !== undefined) {
            tickManager.clear(this._timeoutId);
            this._timeoutId = undefined;
        }
    }

    /**
     * 结算 Promise（完成或拒绝）
     */
    private async settle(isRejected: boolean, valueOrReason?: any, isCancellation = false): Promise<void> {
        if (this._isSettled) return;
        this._isSettled = true;

        this.clearTimeout();

        // 如果是取消操作，执行 onCancel 回调
        if (isCancellation) {
            await this.safeExecuteCallback(this.options.onCancel, valueOrReason);
        }

        // 结算 Promise
        isRejected ? this._reject(valueOrReason) : this._resolve(valueOrReason);

        // 通知管理器清理
        this.onCleanup(this.id);

        // 解决取消操作的 Promise
        this._cancelResolve?.();
    }

    /**
     * 取消 Promise
     */
    public cancel = (reason?: any): Promise<void> => {
        if (this._isCancelled) {
            return Promise.resolve();
        }

        this._isCancelled = true;
        return new Promise<void>((resolveCancel) => {
            this._cancelResolve = resolveCancel;
            this.settle(true, reason ?? new Error(`Promise ${this.id} 被主动取消了, text:${this.options.text}`), true);
        });
    };
}
