/**
 * 表示一个可取消的 Promise 及其取消函数。
 */
export interface IPromiseCancellable<T> {
    promise: Promise<T>;
    cancel: (reason?: any) => Promise<void>;
    isCancelled: boolean;
    id: string;
    category: string;
}

/**
 * Promise 执行器函数类型
 */
export type PromiseExecutorFunction<T> = (resolve: (value: T | PromiseLike<T>) => void, reject: (reason?: any) => void) => void;

/**
 * Promise 超时行为配置
 */
export type TimeoutAction = 'resolve' | 'reject';

/**
 * 取消回调函数类型
 */
export type CancelCallback = (reason?: any) => void | Promise<void>;

/**
 * Promise 配置选项接口。
 */
export interface ICreatePromiseOptions<T> {
    /** Promise 的唯一标识符 */
    id?: string;
    /** Promise 的分类标识 */
    category?: string;
    /** 超时时间（毫秒） */
    timeout?: number;
    /** 超时时的处理方式 */
    timeoutAction?: TimeoutAction;
    /** 超时时的解析值（仅在 timeoutAction 为 'resolve' 时使用） */
    timeoutValue?: T;
    /** 超时时的拒绝原因（仅在 timeoutAction 为 'reject' 时使用） */
    timeoutReason?: any;
    /** 用于调试的文本描述 */
    text?: string;
    /** 当 Promise 被取消时触发的回调函数 */
    onCancel?: CancelCallback;
}

/**
 * 更精确的期望值类型定义
 */
export type ExpectedValue<T extends object, K extends keyof T> = T[K] | (() => boolean);

/**
 * 等待条件变化的配置选项接口。
 */
export interface ICreateWaitActionOptions<T extends object> extends ICreatePromiseOptions<void> {
    /** 检查属性的频率（毫秒）。默认值为 tickManager.frameTime */
    checkInterval?: number;
    /**
     * 期望的目标值或检查函数
     * - 如果是值，则等待 target[key] 变为此值
     * - 如果是函数，则等待函数返回 true
     * - 如果未提供，则等待 target[key] 发生任何变化
     */
    expectedValue?: ExpectedValue<T, keyof T>;
}

/**
 * Promise 管理器的统计信息
 */
export interface IPromiseManagerStats {
    /** 活跃的 Promise 总数 */
    total: number;
    /** 按分类统计 */
    byCategory: Record<string, number>;
    /** 最新的 Promise ID */
    latestId?: string;
}

/**
 * 批量操作的结果
 */
export interface IBatchOperationResult {
    /** 成功处理的数量 */
    success: number;
    /** 失败的数量 */
    failed: number;
    /** 失败的 Promise ID 列表 */
    failedIds: string[];
}
