import { ManagerBase } from '../../managers/ManagerBase';
import { StringUtils } from '../../utils/StringUtils';
import { WaitForChangePromise } from './PromiseWaitForChange';
import { PromiseCancellable } from './PromiseCancellable';
import { PromiseExecutorFunction, ICreatePromiseOptions, ICreateWaitActionOptions } from './PromiseDefines';

/**
 * @class PromiseManager
 * @description 统一管理和创建可取消的 Promise。
 */
export class PromiseManager extends ManagerBase {
    private _activePromises = new Map<string, PromiseCancellable<any>>();

    /**
     * 清理 Promise 的回调
     */
    private onPromiseCleanup = (id: string): void => {
        this._activePromises.delete(id);
    };

    /**
     * 创建一个可取消的 Promise。
     */
    public create<T>(executor: PromiseExecutorFunction<T>, options: ICreatePromiseOptions<T> = {}): PromiseCancellable<T> {
        const category = options.category ?? 'CancellablePromise';
        const promiseId = options.id ?? `${category}_${StringUtils.uuid(false)}`;
        const finalOptions: ICreatePromiseOptions<T> = { ...options, id: promiseId, category };
        const cancellablePromise = new PromiseCancellable(executor, finalOptions, this.onPromiseCleanup);
        this._activePromises.set(promiseId, cancellablePromise);

        return cancellablePromise;
    }

    /**
     * 根据 ID 取消一个特定的 Promise。
     */
    public async cancel(id: string, reason?: any): Promise<void> {
        const cancellable = this._activePromises.get(id);
        if (cancellable && !cancellable.isCancelled) {
            return cancellable.cancel(reason);
        }
        return Promise.resolve();
    }

    /**
     * 批量取消 Promise
     */
    private async cancelBatch(promises: PromiseCancellable<any>[], reason?: any): Promise<void> {
        const cancelPromises = promises.filter((p) => !p.isCancelled).map((p) => p.cancel(reason));

        await Promise.all(cancelPromises);

        // 清理任何残留项
        if (this._activePromises.size > 0) {
            const remaining = Array.from(this._activePromises.values()).filter((p) => p.isCancelled);
            remaining.forEach((p) => this._activePromises.delete(p.id));
        }
    }

    /**
     * 取消所有当前活跃的 Promise。
     */
    public async cancelAll(reason?: any): Promise<void> {
        const promises = Array.from(this._activePromises.values());
        await this.cancelBatch(promises, reason);
    }

    /**
     * 取消指定分类的所有活跃 Promise。
     */
    public async cancelCategory(category: string, reason?: any): Promise<void> {
        const promises = Array.from(this._activePromises.values()).filter((p) => p.category === category);
        await this.cancelBatch(promises, reason);
    }

    /**
     * 获取所有活跃的 Promise ID。
     */
    public getActivePromiseIds(): string[] {
        return Array.from(this._activePromises.keys());
    }

    /**
     * 获取指定 ID 的可取消 Promise。
     */
    public getCancellablePromise(id: string): PromiseCancellable<any> | undefined {
        return this._activePromises.get(id);
    }

    /**
     * 创建一个等待指定对象属性变化或达到预期值的 Promise。
     */
    public waitForChange<T extends object>(target: T, key: keyof T, options: ICreateWaitActionOptions<T> = {}): Promise<void> {
        const category = options.category ?? 'CATEGORY_WAIT_ACTION';
        const promiseId = options.id ?? `${category}_${StringUtils.uuid(false)}`;
        const finalOptions: ICreateWaitActionOptions<T> = { ...options, id: promiseId, category };
        const waitPromise = new WaitForChangePromise(target, key, finalOptions, this.onPromiseCleanup);
        this._activePromises.set(promiseId, waitPromise);
        return waitPromise.promise;
    }
}

export const promiseManager: PromiseManager = new PromiseManager();
