import { tickManager } from '../ticker/TickManager';
import { PromiseCancellable } from './PromiseCancellable';
import { ICreateWaitActionOptions } from './PromiseDefines';

/**
 * 等待条件变化的特殊 Promise 类
 */
export class WaitForChangePromise<T extends object> extends PromiseCancellable<void> {
    private _tickId?: number;

    constructor(target: T, key: keyof T, options: ICreateWaitActionOptions<T>, onCleanup: (id: string) => void) {
        const checkInterval = options.checkInterval ?? tickManager.frameTime;
        const hasExpectedValue = 'expectedValue' in options;
        const isFunction = typeof options.expectedValue === 'function';
        const initialValue = target[key];

        super(
            (resolve) => {
                // 立即检查条件
                if (this.checkCondition(target, key, hasExpectedValue, isFunction, initialValue, options)) {
                    return resolve();
                }

                // 启动定时检查
                this._tickId = tickManager.forever(
                    () => {
                        if (this.checkCondition(target, key, hasExpectedValue, isFunction, initialValue, options)) {
                            this.clearTicker();
                            resolve();
                        }
                    },
                    this,
                    checkInterval,
                );
            },
            {
                id: options.id,
                category: options.category,
                timeout: options.timeout,
                timeoutAction: options.timeoutAction,
                timeoutReason: options.timeoutReason,
                text: options.text,
                onCancel: async (reason?: any) => {
                    this.clearTicker();
                    // 调用原始 onCancel
                    if (options.onCancel) {
                        await this.safeExecuteCallback(options.onCancel, reason);
                    }
                },
            },
            onCleanup,
        );
    }

    /**
     * 安全检查条件
     */
    private checkCondition(
        target: T,
        key: keyof T,
        hasExpectedValue: boolean,
        isFunction: boolean,
        initialValue: any,
        options: ICreateWaitActionOptions<T>,
    ): boolean {
        return this.safeCheck(() => {
            if (hasExpectedValue) {
                return isFunction ? (options.expectedValue as () => boolean)() : target[key] === options.expectedValue;
            }
            return target[key] !== initialValue;
        }, `waitForChange 条件检查异常`);
    }

    /**
     * 清理定时器
     */
    private clearTicker(): void {
        if (this._tickId !== undefined) {
            tickManager.clear(this._tickId);
            this._tickId = undefined;
        }
    }
}
