

const { readFileSync } = require('fs');
const { join } = require('path');
const { readJSONSync, writeJSONSync } = require('fs-extra');
const { SceneDataManager } = require('./scene-data/scene-data-manager');




// 面板的内容
exports.template = readFileSync(join(__dirname, 'default.html'), 'utf8');
// 面板上的样式
exports.style = `
.mainPanel {
    display: flex;
    flex-direction: column;
}
.header {
    height: 100;
    width: 100%;
    display: flex;
    flex-direction: row;
}
.SceneFilePanel {
    width: 100%;
    display: flex;
    flex-direction: row;
    height: 500px;
}
.hierarchy {
    width: 33.3%;
}
.inspector {
    width: 33.3%;
    display: flex;
    flex-direction: column;
}
.dataView {
    width: 33.3%;
}
.sceneTree { 
    height: 100%;
    width: 100%;
}
.nodeDataTree { 
    height: 100%;
    width: 100%;
}
.jsonView {
    height: 100%;
    width: 100%;
}

`;
// 快捷选择器
exports.$ = {
    elem: '.header',
    sceneTree: '.sceneTree',
    nodeDataTree: '.nodeDataTree',
    jsonView: '.jsonView',
    saveButton: '.saveButton',
    showAllCheckbox: '.showAllCheckbox',
    nameFilterInput: '.nameFilterInput'
};

// 监听面板事件
exports.listeners = {
    // 面板显示的时候触发的钩子
    show() {},
    // 面板隐藏的时候触发的钩子
    hide() {},
};

let curEditJsonFile = '';
let curEditSceneUUID = '';
// 是否把组织到树形结构里的数据的原始数据也显示出来
let showAllData = true;

let sceneDataMgr = null;
const CCNodeType = 'cc.Node';

let curSelectedSceneItemIndex = -1;

const normalColor = '#a9adb9';
const prefabAssetColor = '#5cbf8c';

exports.methods = {
    initSceneTree() {
        this.$.sceneTree.setTemplate('text', `<span class="arrayIndex"></span><span class="name"></span>`);
        this.$.sceneTree.setTemplateInit('text', ($text) => {
            $text.$name = $text.querySelector('.name');
            $text.$arrayIndex = $text.querySelector('.arrayIndex');
        });
        this.$.sceneTree.setRender('text', ($text, data) => {
            let obj = data.detail.data;
            $text.$name.textContent = obj.name;
            $text.$name.style.color = normalColor;

            if (obj.indexOfArray >= 0) {
                $text.$arrayIndex.innerHTML = `[${obj.indexOfArray}]`;
            } else {
                $text.$arrayIndex.innerHTML = '';
            }


            if (obj.isPrefab) {
                // 标记为prefab的节点
                $text.$name.style.color = prefabAssetColor;
            }

        });
    
        this.$.sceneTree.setTemplate('right', '<ui-icon value="close"></ui-icon>');
        this.$.sceneTree.setTemplateInit('right', ($right) => {
            $right.$remove = $right.querySelector('ui-icon');
            $right.$remove.addEventListener('click', (event) => {
                event.stopPropagation();
                this.onSceneItemDelete($right.data);
            });
        });

    
        this.$.sceneTree.addEventListener('keydown', (event) => {
            const $dom = this.$.sceneTree;
            
            if (event.code === 'ArrowUp') {
                const item = $dom.selectItems[$dom.selectItems.length - 1];
                const index = Math.max(item.index - 1, 0);
                if (event.shiftKey) {
                    $dom.select($dom.list[index]);
                } else {
                    $dom.clear();
                    $dom.select($dom.list[index]);
                }
                $dom.render();
            } else if (event.code === 'ArrowDown') {
                const item = $dom.selectItems[$dom.selectItems.length - 1];
                const index = Math.min(item.index + 1, $dom.list.length - 1);
                if (event.shiftKey) {
                    $dom.select($dom.list[index]);
                } else {
                    $dom.clear();
                    $dom.select($dom.list[index]);
                }
                $dom.render();
            }
        });
    
        this.$.sceneTree.setTemplateInit('item', ($div) => {
            const $dom = this.$.sceneTree;
            $div.addEventListener('click', (event) => {
                if (event.ctrlKey || event.metaKey) {
                    $dom.select($div.data);
                } else {
                    $dom.clear();
                    $dom.select($div.data);
                }
                this.onSceneItemSelected($div.data);
                $dom.render();
            });
        });
        // this.$.sceneTree.setRender('item', ($div, data) => {
        //     if (data.detail.disabled) {
        //         $div.setAttribute('disabled', '');
        //     } else {
        //         $div.removeAttribute('disabled');
        //     }
        // });
    
        this.$.sceneTree.setItemRender;
    
        this.$.sceneTree.css = `
            .item[disabled] {
                opacity: 0.4;
            }
    
            .text > .arrayIndex {
                color: yellow;
            }
    
            .right > ui-icon {
                cursor: pointer;
                color: red;
            }
                `;
    },

    initNodeDataTree() {
        this.$.nodeDataTree.setTemplate('text', `<span class="arrayIndex"></span><span class="name"></span>`);
        this.$.nodeDataTree.setTemplateInit('text', ($text) => {
            $text.$name = $text.querySelector('.name');
            $text.$arrayIndex = $text.querySelector('.arrayIndex');
        });
        this.$.nodeDataTree.setRender('text', ($text, data) => {
            const obj = data.detail.data;
            $text.$name.textContent = `${obj.name}`;
            if (obj.indexOfArray != undefined && obj.indexOfArray >= 0) {
                $text.$arrayIndex.innerHTML = `[${obj.indexOfArray}]`;
            } else {
                $text.$arrayIndex.innerHTML = '';
            }

            $text.$name.style.color = normalColor;
            if (obj.isPrefab) {
                // 标记为prefab的节点
                $text.$name.style.color = prefabAssetColor;
            }
        });
    
        this.$.nodeDataTree.setTemplateInit('item', ($div) => {
            const $dom = this.$.nodeDataTree;
            $div.addEventListener('click', (event) => {
                if (event.ctrlKey || event.metaKey) {
                    $dom.select($div.data);
                } else {
                    $dom.clear();
                    $dom.select($div.data);
                }
                this.onNodeDataItemSelected($div.data);
                $dom.render();
            });
        });

            
        this.$.nodeDataTree.setTemplate('right', '<ui-icon value="close"></ui-icon>');
        this.$.nodeDataTree.setTemplateInit('right', ($right) => {
            $right.$remove = $right.querySelector('ui-icon');
            $right.$remove.addEventListener('click', (event) => {
                event.stopPropagation();
                this.onSceneItemDelete($right.data);
            });
        });

        this.$.nodeDataTree.setItemRender;
    
        this.$.nodeDataTree.css = `
            .text > .arrayIndex {
                color: yellow;
            }
    
            .right > ui-icon {
                cursor: pointer;
                color: red;
            }
                `;
    },

    onSceneItemSelected(itemData) {
        
        const obj = itemData.detail.data;
        this.$.jsonView.textContent = JSON.stringify(obj.jsonData, null, 4);

        let inspectorTree = [];

        const inspectorTreeItem = obj.getInspectorTreeItemData();
        if (inspectorTreeItem) {
            if (Array.isArray(inspectorTreeItem)) {
                inspectorTree = inspectorTree.concat(inspectorTreeItem);
            } else {
                inspectorTree.push(inspectorTreeItem);
            }
            
        }
        

        this.$.nodeDataTree.tree = inspectorTree;
    },

    onSceneItemDelete(itemData) {
        curSelectedSceneItemIndex = itemData.index;
        const obj = itemData.detail.data;
        sceneDataMgr.removeSceneItem(obj);
        this.refreshSceneTree();
    },

    onNodeDataItemSelected(itemData) {
        let obj = itemData.detail.data;
        this.$.jsonView.textContent = JSON.stringify(obj.jsonData, null, 2);
    },

    refreshSceneTree(filterName) {
        let treeData = sceneDataMgr.getSceneTree(showAllData, filterName);
        this.$.sceneTree.tree = treeData;

        const $dom = this.$.sceneTree;
        if (curSelectedSceneItemIndex >= 0 && curSelectedSceneItemIndex < ($dom.list.length)) {
            $dom.clear();
            $dom.select($dom.list[curSelectedSceneItemIndex]);
            $dom.render();
        }
    }

}

// 面板启动后触发的钩子函数
exports.ready = function() {
    //this.$.elem.innerHTML = 'Scene File Editor';

    this.initSceneTree();
    this.initNodeDataTree();

    Editor.Message.addBroadcastListener('selection:select', async (type, uuid) => {
        if (type !== 'asset') {
            return;
        }

        let assetInfo = await Editor.Message.request('asset-db', 'query-asset-info', uuid);

        const importer = assetInfo.importer;
        if (importer != 'scene' && importer != 'prefab') {
            return;
        }

        curEditJsonFile = assetInfo.file;
        curEditSceneUUID = assetInfo.uuid;
        const json = readJSONSync(curEditJsonFile);

        sceneDataMgr = new SceneDataManager();
        await sceneDataMgr.parseJson(json);

        let treeData = sceneDataMgr.getSceneTree(showAllData);
        this.$.sceneTree.tree = treeData;
        this.$.nodeDataTree.tree = null;
    });
    
    this.$.showAllCheckbox.addEventListener('click', (event) => {
        showAllData = event.target.value;
        this.refreshSceneTree(this.$.nameFilterInput.value);
    });

    this.$.saveButton.addEventListener('click', async (event) => {
        writeJSONSync(curEditJsonFile, sceneDataMgr.sceneJsonData, {spaces: 2});

        await Editor.Message.request('asset-db', 'refresh-asset', curEditSceneUUID);
    });

    this.$.nameFilterInput.addEventListener('change', (event)=> {
        this.refreshSceneTree(event.target.value);
    });
};


// 面板准备关闭的时候会触发的函数，return false 的话，会终止关闭面板
exports.beforeClose = function() {};

// 面板关闭后的钩子函数
exports.close = function() {};