/*
 * @Author: gongqinying (eycodes@gmail.com)
 * @Date: 2025-08-18 14:13:45
 * @LastEditTime: 2025-08-28 12:56:01
 * @LastEditors: gongqinying (eycodes@gmail.com)
 * @FilePath: \dev_client\assets\Script\data\ZipBundleLoader.ts
 * @Description: 
 */

declare const JSZip: any;
const { ccclass } = cc._decorator;

@ccclass
export default class ZipBundleLoader {
    public static async load(bundleName: string) {
        const defaultJsonHandler = cc.assetManager.downloader._downloaders['.json'];
        const defaultImageHandler = cc.assetManager.downloader._downloaders['.png'];
        const defaultPlistHandler = cc.assetManager.downloader._downloaders['.plist'];
        const defaultAudioHandler = cc.assetManager.downloader._downloaders['.mp3'] ||
            cc.assetManager.downloader._downloaders['.ogg'] ||
            cc.assetManager.downloader._downloaders['.wav'];

        // @ts-ignore
        const bundleVers = cc.assetManager.downloader.bundleVers;
        console.log("bundleVers====>", bundleVers);
        if (!bundleVers) return;
        
        const md5key: string = bundleVers[bundleName];
        const zipUrl = `assets/${bundleName}/${bundleName}.${md5key}.zip`; // 主包根目录下
        console.log(`加载本地 zip: ${zipUrl}`);

        // 1. 下载 zip
        let jszip = new JSZip();
        const res = await fetch(zipUrl);
        if (!res.ok) throw new Error(`加载 zip 失败: ${res.statusText}`);
        const zipData = await res.arrayBuffer();
        const zip = await jszip.loadAsync(zipData);

        // 2. 存储文件数据到一个 map
        const fileMap: Record<string, Uint8Array> = {};
        for (const key of Object.keys(zip.files)) {
            const file = zip.files[key];
            if (!file.dir) {
                fileMap[`assets/${bundleName}/${key}`] = await file.async('uint8array');
            }
        }

        // 3. 注册自定义下载器，拦截该分包路径
        cc.assetManager.downloader.register('.json', (url, options, onComplete) => {
            if (url.includes(`assets/${bundleName}`)) {
                const fileData = fileMap[url];
                if (!fileData) {
                    return onComplete(new Error('File not found in memory: ' + url), null);
                }
                // 假设是文本型
                const text = new TextDecoder().decode(fileData);
                delete fileMap[url];
                onComplete(null!, JSON.parse(text));
                return;
            }

            // 交给原始处理器
            defaultJsonHandler(url, options, onComplete);

        });

        cc.assetManager.downloader.register('.png', (url, options, onComplete) => {
            if (url.includes(`assets/${bundleName}`)) {
                const fileData = fileMap[url];
                if (!fileData) {
                    return onComplete(new Error('File not found in memory: ' + url), null);
                }
                const blob = new Blob([fileData], { type: "image/png" });
                const img = new Image();
                img.onload = () => {
                    URL.revokeObjectURL(img.src);
                    delete fileMap[url];
                    onComplete(null!, img);
                };
                img.onerror = () => {
                    URL.revokeObjectURL(img.src);
                    delete fileMap[url];
                    onComplete(new Error('Failed to load image: ' + url), null);
                };
                img.src = URL.createObjectURL(blob);
                return;
            }

            // 交给原始处理器
            defaultImageHandler(url, options, onComplete);
        });

        // ---------------- PLIST ----------------
        cc.assetManager.downloader.register('.plist', (url, options, onComplete) => {
            if (url.includes(`assets/${bundleName}/`)) {
                const relative = url.split(`assets/${bundleName}/`)[1];
                const fileData = fileMap[relative];
                if (!fileData) return onComplete(new Error('File not found: ' + relative), null);
                const text = new TextDecoder().decode(fileData);
                delete fileMap[relative];
                return onComplete(null!, text); // Cocos 内部会解析
            }
            defaultPlistHandler(url, options, onComplete);
        });

        // ---------------- Audio ----------------
        const audioExts = ['.mp3', '.ogg', '.wav'];
        audioExts.forEach(ext => {
            cc.assetManager.downloader.register(ext, (url, options, onComplete) => {
                if (url.includes(`assets/${bundleName}/`)) {
                    const relative = url.split(`assets/${bundleName}/`)[1];
                    const fileData = fileMap[relative];
                    if (!fileData) return onComplete(new Error('File not found: ' + relative), null);

                    let mimeType = '';
                    if (ext === '.mp3') mimeType = 'audio/mpeg';
                    else if (ext === '.ogg') mimeType = 'audio/ogg';
                    else if (ext === '.wav') mimeType = 'audio/wav';

                    const blob = new Blob([fileData], { type: mimeType });
                    delete fileMap[relative];
                    onComplete(null!, blob); // Cocos 会接管 blob 播放
                    return;
                }
                defaultAudioHandler(url, options, onComplete);
            });
        });

        // 4. 加载分包（会走我们注册的 handler）
        cc.assetManager.loadBundle(bundleName, (err, bundle) => {
            if (err) {
                console.error(err);
            } else {
                console.log(`${bundleName} loaded from memory`);
            }
        });
    }
}