/*
 * Created by yuejz on 2020/12/24.
 * Copyright 2015－2020 Sensors Data Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "../include/ObjectNode.h"
#include <sstream>

using namespace sensorsdata;

void ObjectNode::setNumber(const string &property_name, double value) {
    properties_map_[property_name] = ValueNode(value);
}

void ObjectNode::setNumber(const string &property_name, int32_t value) {
    properties_map_[property_name] = ValueNode(static_cast<int64_t>(value));
}

void ObjectNode::setNumber(const string &property_name, int64_t value) {
    properties_map_[property_name] = ValueNode(value);
}

void ObjectNode::setString(const string &property_name, const string &value) {
    properties_map_[property_name] = ValueNode(value);
}

void ObjectNode::setString(const string &property_name, const char *value) {
    setString(property_name, string(value));
}

void ObjectNode::setBool(const string &property_name, bool value) {
    properties_map_[property_name] = ValueNode(value);
}

void ObjectNode::setList(const string &property_name, const std::vector<string> &value) {
    properties_map_[property_name] = ValueNode(value);
}

void ObjectNode::setDateTime(const string &property_name, const time_t seconds, int milliseconds) {
    properties_map_[property_name] = ValueNode(seconds, milliseconds);
}

void ObjectNode::setDateTime(const string &property_name, const string &value) {
    properties_map_[property_name] = ValueNode(value);
}

void ObjectNode::clear() {
    properties_map_.clear();
}

void ObjectNode::dumpNode(const ObjectNode &node, string *buffer) {
    *buffer += '{';
    bool first = true;

    for (std::map<string, ValueNode>::const_iterator iterator = node.properties_map_.begin(); iterator != node.properties_map_.end(); ++iterator) {
        if (first) {
            first = false;
        } else {
            *buffer += ',';
        }
        *buffer += '"' + iterator->first + "\":";
        ValueNode::toStr(iterator->second, buffer);
    }
    *buffer += '}';
}

void ObjectNode::ValueNode::dumpString(const string &value, string *buffer) {
    *buffer += '"';
    for (std::string::size_type i = 0; i < value.length(); ++i) {
        char c = value[i];
        switch (c) {
            case '"':
                *buffer += "\\\"";
                break;
            case '\\':
                *buffer += "\\\\";
                break;
            case '\b':
                *buffer += "\\b";
                break;
            case '\f':
                *buffer += "\\f";
                break;
            case '\n':
                *buffer += "\\n";
                break;
            case '\r':
                *buffer += "\\r";
                break;
            case '\t':
                *buffer += "\\t";
                break;
            default:
                *buffer += c;
                break;
        }
    }
    *buffer += '"';
}

void ObjectNode::ValueNode::dumpList(const std::vector<string> &value, string *buffer) {
    *buffer += '[';
    bool first = true;
    for (std::vector<string>::const_iterator iterator = value.begin(); iterator != value.end(); ++iterator) {
        if (first) {
            first = false;
        } else {
            *buffer += ',';
        }
        dumpString(*iterator, buffer);
    }
    *buffer += ']';
}

#if defined(__linux__)
#define SA_SDK_LOCALTIME(seconds, now) localtime_r((seconds), (now))
#elif defined(__APPLE__)
#define SA_SDK_LOCALTIME(seconds, now) localtime_r((seconds), (now))
#elif defined(_WIN32)
#define SA_SDK_LOCALTIME(seconds, now) localtime_s((now), (seconds))
#define snprintf sprintf_s
#endif

void ObjectNode::ValueNode::dumpDateTime(const time_t &seconds, int milliseconds, string *buffer) {
    struct tm tm = {};
    SA_SDK_LOCALTIME(&seconds, &tm);
    char buff[64];
    snprintf(buff, sizeof(buff), "\"%04d-%02d-%02d %02d:%02d:%02d.%03d\"",
             tm.tm_year + 1900,
             tm.tm_mon + 1,
             tm.tm_mday,
             tm.tm_hour,
             tm.tm_min,
             tm.tm_sec,
             milliseconds);
    *buffer += buff;
}

string ObjectNode::toJson(const ObjectNode &node) {
    string buffer;
    dumpNode(node, &buffer);
    return buffer;
}

ObjectNode::ValueNode::ValueNode(double value) : node_type_(NUMBER) {
    value_.number_value = value;
}

ObjectNode::ValueNode::ValueNode(int64_t value) : node_type_(INT) {
    value_.int_value = value;
}

ObjectNode::ValueNode::ValueNode(const string &value) : node_type_(STRING), string_data_(value) {}

ObjectNode::ValueNode::ValueNode(bool value) : node_type_(BOOL) {
    value_.bool_value = value;
}

ObjectNode::ValueNode::ValueNode(const std::vector<string> &value) : node_type_(LIST), list_data_(value) {}

ObjectNode::ValueNode::ValueNode(time_t seconds, int milliseconds) : node_type_(DATETIME) {
    value_.date_time_value.seconds = seconds;
    value_.date_time_value.milliseconds = milliseconds;
}

void ObjectNode::ValueNode::toStr(const ObjectNode::ValueNode &node, string *buffer) {
    switch (node.node_type_) {
        case NUMBER:
            dumpNumber(node.value_.number_value, buffer);
            break;
        case INT:
            dumpNumber(node.value_.int_value, buffer);
            break;
        case STRING:
            dumpString(node.string_data_, buffer);
            break;
        case LIST:
            dumpList(node.list_data_, buffer);
            break;
        case BOOL:
            *buffer += (node.value_.bool_value ? "true" : "false");
            break;
        case DATETIME:
            dumpDateTime(node.value_.date_time_value.seconds,
                         node.value_.date_time_value.milliseconds, buffer);
            break;
        default:
            break;
    }
}

void ObjectNode::ValueNode::dumpNumber(double value, string *buffer) {
    std::ostringstream buf;
    buf << value;
    *buffer += buf.str();
}

void ObjectNode::ValueNode::dumpNumber(int64_t value, string *buffer) {
    std::ostringstream buf;
    buf << value;
    *buffer += buf.str();
}

void ObjectNode::mergeFrom(const ObjectNode &another_node) {
    for (std::map<string, ValueNode>::const_iterator
                 iterator = another_node.properties_map_.begin();
         iterator != another_node.properties_map_.end(); ++iterator) {
        properties_map_[iterator->first] = iterator->second;
    }
}
