import ArrayUtil from "./ArrayUtil";


export default class MathUtil {

    static randomRangeInt(min: number, max: number): number {
        let rand = Math.random();
        return min + Math.floor(rand * (max - min));
    }


    static randomRangeFloat(min: number, max: number): number {
        return min + (Math.random() * (max - min));
    }


    /**
     * 获取数组中随机一个单元
     * @param arr 数组数据源
     */
    static randomArray(arr: Array<any>): any {
        const index: number = this.randomRangeInt(0, arr.length) | 0;
        return arr[index];
    }


    //取出随机数, maxNum为 取出随机数的个数
    static RandomIntBoth(min: number, max: number, num: number) {
        min = Math.floor(min)
        max = Math.floor(max)
        if (min >= max || max - min < num || num == 0) {
            clog.error("min > max ||  max - min < num || num == 0")
            return null;
        }
        let ary: number[] = []
        for (let i = min; i < max; i++) {
            ary.push(i)
        }
        ArrayUtil.shuffle(ary)
        let len = Math.min(ary.length, num)
        let list: number[] = []
        for (let i = 0; i < len; i++) {
            list.push(ary.shift())
        }
        return list;
    }

    /**
     * 弧度制转换为角度值
     * @param {number} radian
     * @returns {number}
     */
    static getAngle(radian: number): number {
        return 180 * radian / Math.PI;
    }

    /**
     * 角度值转换为弧度制
     * @param {number} angle
     */
    static getRadian(angle: number): number {
        return angle / 180 * Math.PI;
    }

    /**
     * 获取两点间弧度
     * @param {cc.Vec2} p1
     * @param {cc.Vec2} p2
     * @returns {number}
     */
    static getRadianTwoPoint(p1: cc.Vec3, p2: cc.Vec3): number {
        const xdis: number = p2.x - p1.x;
        const ydis: number = p2.y - p1.y;
        return Math.atan2(ydis, xdis);
    }

    /**
     * 获取两点间旋转角度（顺时针）
     * @param {cc.Vec2} p1
     * @param {cc.Vec2} p2
     * @returns {number}
     */
    static getAngleTwoPoint(p1: { x: number, y: number }, p2: { x: number, y: number }): number {
        const vy: number = p2.y - p1.y;
        const vx: number = p2.x - p1.x;
        let ang: number;

        if (vy == 0) {
            if (vx < 0) {
                return 180;
            }
            return 0;
        }

        if (vx == 0) { //正切是vy/vx所以vx==0排除
            if (vy > 0) {
                ang = 90;
            }
            else if (vy < 0) {
                ang = 270;
            }
            return ang;
        }

        ang = this.getAngle(Math.atan(Math.abs(vy) / Math.abs(vx)));
        if (vx > 0) {
            if (vy < 0) {
                ang = 360 - ang;
            }
        }
        else {
            if (vy > 0) {
                ang = 180 - ang;
            }
            else {
                ang = 180 + ang;
            }
        }
        return ang;
    }

    /**
     * 获取两点间距离
     * @param {cc.Vec2} p1
     * @param {cc.Vec2} p2
     * @returns {number}
     */
    static getDistance(p1: { x: number, y: number }, p2: { x: number, y: number }): number {
        if (!p1 || !p2) {
            return 10000
        }
        const disX: number = p2.x - p1.x;
        const disY: number = p2.y - p1.y;
        const disQ: number = Math.pow(disX, 2) + Math.pow(disY, 2);
        return Math.sqrt(disQ);
    }
    /**
     * 获取两点间距离的平方
     * @param {cc.Vec3} p1
     * @param {cc.Vec3} p2
     * @returns {number}
     */
    static getDistanceSq(p1: cc.Vec3, p2: cc.Vec3): number {
        if (!p1 || !p2) {
            return 0
        }
        const disX: number = p2.x - p1.x;
        const disY: number = p2.y - p1.y;
        const disQ: number = Math.pow(disX, 2) + Math.pow(disY, 2);
        return disQ;
    }

    /**
     * 精确到小数点后多少位（舍尾）
     * @param {number} 精确值
     * @param {number} 精确位数
     * @return {number}
     * */
    static exactCount(exactValue: number, count: number = 0): number {
        const num: number = Math.pow(10, count);
        const value: number = (exactValue * num) | 0;
        return value / num;
    }

    /**
     * [0-1]区间获取二次贝塞尔曲线点切线角度
     * @param {cc.Vec2} p0起点
     * @param {cc.Vec2} p1控制点
     * @param {cc.Vec2} p2终点
     * @param {number} t [0-1]区间
     * @return {number}
     * */
    static getBezierCutAngle(p0: cc.Vec2, p1: cc.Vec2, p2: cc.Vec2, t: number): number {
        const _x: number = 2 * (p0.x * (t - 1) + p1.x * (1 - 2 * t) + p2.x * t);
        const _y: number = 2 * (p0.y * (t - 1) + p1.y * (1 - 2 * t) + p2.y * t);
        const angle: number = this.getAngle(Math.atan2(_y, _x));
        return angle;
    }

    /**
     * [0-1]区间获取二次贝塞尔曲线上某点坐标
     * @param {cc.Vec2} p0 起点
     * @param {cc.Vec2} p1 控制点
     * @param {cc.Vec2} p2 终点
     * @param {number} t [0-1]区间
     * @param {cc.Vec2} 缓存的点对象，如不存在则生成新的点对象
     * @return {cc.Vec2}
     * */
    static getBezierPoint(p0: cc.Vec2, p1: cc.Vec2, p2: cc.Vec2, t: number, point: cc.Vec2 = null): cc.Vec2 {
        if (!point) {
            point = new cc.Vec2();
        }
        point.x = (1 - t) * (1 - t) * p0.x + 2 * t * (1 - t) * p1.x + t * t * p2.x;
        point.y = (1 - t) * (1 - t) * p0.y + 2 * t * (1 - t) * p1.y + t * t * p2.y;
        return point;
    }

    /**
     * [0-1]区间获取三次贝塞尔曲线上某点坐标
     * @param {cc.Vec2} p0 起点
     * @param {cc.Vec2} p1 控制点
     * @param {cc.Vec2} p2 控制点
     * @param {cc.Vec2} p3 终点
     * @param {number} t [0-1]区间
     * @param {cc.Vec2} 缓存的点对象，如不存在则生成新的点对象
     * @return {cc.Vec2}
     * */
    static getBezier3Point(p0: cc.Vec2, p1: cc.Vec2, p2: cc.Vec2, p3: cc.Vec2, t: number, point: cc.Vec2 = null): cc.Vec2 {
        if (!point) {
            point = new cc.Vec2();
        }
        const cx: number = 3 * (p1.x - p0.x);
        const bx: number = 3 * (p2.x - p1.x) - cx;
        const ax: number = p3.x - p0.x - cx - bx;
        const cy: number = 3 * (p1.y - p0.y);
        const by: number = 3 * (p2.y - p1.y) - cy;
        const ay: number = p3.y - p0.y - cy - by;
        point.x = ax * t * t * t + bx * t * t + cx * t + p0.x;
        point.y = ay * t * t * t + by * t * t + cy * t + p0.y;
        return point;
    }

    /**
     * [0-1]区间获取三次贝塞尔曲线点切线角度
     * @param {cc.Vec2} p0起点
     * @param {cc.Vec2} p1控制点
     * @param {cc.Vec2} p2控制点
     * @param {cc.Vec2} p3终点
     * @param {number} t [0-1]区间
     * @return {number}
     * */
    static getBezier3CutAngle(p0: cc.Vec2, p1: cc.Vec2, p2: cc.Vec2, p3: cc.Vec2, t: number): number {
        const _x: number = p0.x * 3 * (1 - t) * (1 - t) * (-1) +
            3 * p1.x * ((1 - t) * (1 - t) + t * 2 * (1 - t) * (-1)) +
            3 * p2.x * (2 * t * (1 - t) + t * t * (-1)) +
            p3.x * 3 * t * t;
        const _y: number = p0.y * 3 * (1 - t) * (1 - t) * (-1) +
            3 * p1.y * ((1 - t) * (1 - t) + t * 2 * (1 - t) * (-1)) +
            3 * p2.y * (2 * t * (1 - t) + t * t * (-1)) +
            p3.y * 3 * t * t;
        const angle: number = this.getAngle(Math.atan2(_y, _x));
        return angle;
    }


    /**
     *根据长度获得随机字符串
     * @param len 
     */
    static randomString(len: number) {
        var $chars = 'ABCDEFGHJKMNPQRSTWXYZabcdefhijkmnprstwxyz2345678';
        var maxPos = $chars.length;
        var pwd = '';
        for (let i = 0; i < len; i++) {
            pwd += $chars.charAt(Math.floor(Math.random() * maxPos));
        }
        return pwd;
    }

    static isInRange(begin: number, end: number, num: number) {
        return num >= begin && num <= end;
    }

    /**
   * 计算某个点旋转后的坐标 
   *
   * @param point  旋转的点 {x,y}
   * @param angle 旋转的角度 >0 逆时针  <0 顺时针
   * @param originPoint 基于哪个点旋转，默认值左上角原点{x: 0, y: 0}
   * @returns {{x: number, y: number}}
   */
    static rotatePoint(point: cc.Vec3, angle: number, originPoint: cc.Vec3 = cc.v3(), out?: cc.Vec3) {
        if (angle == 0) {
            return cc.v3(point.x, point.y);
        }
        let radian: number = MathUtil.getRadian(angle)
        const rx = (point.x - originPoint.x) * Math.cos(radian) - (point.y - originPoint.y) * Math.sin(radian) + originPoint.x
        const ry = (point.x - originPoint.x) * Math.sin(radian) + (point.y - originPoint.y) * Math.cos(radian) + originPoint.y

        if (out) {
            out.x = rx
            out.y = ry
            return out
        }
        return cc.v3(rx, ry);
    }

    static copyObj(obj: any) {
        let data = JSON.parse(JSON.stringify(obj))
        return data
    }

    /**深度拷贝*/
    static deepCopy(toObj, fromObj) {
        for (var k in fromObj) {
            // 判断我们的属性值属于那种数据类型
            // 1. 获取属性值  oldobj[k]
            var item = fromObj[k];
            // 2. 判断这个值是否是数组(首先判断数组是因为数组也属于对象)
            if (item instanceof Array) {
                toObj[k] = [];
                this.deepCopy(toObj[k], item);
            } else if (item instanceof Object) {
                // 3. 判断这个值是否是对象
                toObj[k] = {};
                this.deepCopy(toObj[k], item);
            } else {
                // 4. 属于简单数据类型
                toObj[k] = item;
            }
        }
    };

    /**复制*/
    static v3Copy(from: cc.Vec3, out?: cc.Vec3) {
        if (!out) {
            out = cc.v3()
        }
        out.x = from.x
        out.y = from.y
        out.z = from.z
        return out
    }

    /**比对坐标是否相等*/
    static v3Equals(origin: { x: number, y: number }, other: { x: number, y: number }) {
        return origin.x == other.x && origin.y == other.y
    }

    /**
     * 
     * @param point 某一点
     * @param origin 圆心坐标
     * @param radius 半径
     * @returns 
     */
    static pointInCircle(point: cc.Vec3, origin: cc.Vec3, radius: number) {
        if (radius === 0) return false
        var dx = origin.x - point.x
        var dy = origin.y - point.y
        return dx * dx + dy * dy <= radius * radius
    }
    /**
     * 
     * @param point 
     * @param rect 
     * @returns 
     */
    static pointInRect(point: cc.Vec3, rect: { left: number, right: number, bottom: number, top: number }) {
        if (point.x <= rect.left || point.x >= rect.right) {
            return false
        }
        if (point.y <= rect.bottom || point.y >= rect.top) {
            return false
        }
        return true
    }

    /**粗略计算 忽略4个角 */
    static circleInRect(circle: { pt: cc.Vec3, radius: number }, rect: { left: number, right: number, bottom: number, top: number }) {
        rect.bottom -= circle.radius
        rect.top += circle.radius
        rect.left -= circle.radius
        rect.right += circle.radius
        return this.pointInRect(circle.pt, rect)
    }

    /**
     * 复杂对象合并
     * @param opts 
     * @returns 
     */
    static merger(...opts) {
        if (opts.length == 0) {
            return null
        }

        if (opts.length == 1) {
            return opts[0]
        }

        let res = {};
        let combine = (src: any, target: any) => {
            for (let prop in target) {
                let propType = Object.prototype.toString.call(target[prop])
                if (propType == '[object Object]') {
                    src[prop] = this.merger(src[prop], target[prop]);
                }
                else {
                    src[prop] = target[prop]
                }
            }
        }
        //扩张运算符将两个对象合并到一个数组里因此可以调用length方法
        for (let i = 0; i < opts.length; i++) {
            combine(res, opts[i]);
        }
        return res;
    }

}
